package co.kuali.coeus.s3.impl;

import co.kuali.coeus.s3.api.S3File;
import co.kuali.coeus.s3.api.S3FileMetadata;
import co.kuali.coeus.s3.api.S3FileService;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.*;
import software.amazon.awssdk.utils.IoUtils;

import java.io.IOException;
import java.text.Normalizer;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class S3FileServiceimpl implements S3FileService {

    private static final Logger LOG = LogManager.getLogger(S3FileServiceimpl.class);

    private String bucketName;
    private String replicationBucketName;
    private S3Client amazonS3;
    private boolean encryptionEnabled;
    private String encryptionKey;
    private String confidentialDataTagName;
    private boolean confidentialData;

    @Override
    public String createFile(S3File s3File) {
        if (s3File == null) {
            throw new IllegalArgumentException("file cannot be null");
        }

        if (s3File.getByteContents() == null || s3File.getByteContents().length == 0) {
            throw new IllegalArgumentException("bytes cannot be null or empty");
        }

        if (StringUtils.isNotBlank(replicationBucketName)) {
            createBucketIfNeeded(replicationBucketName);
        }

        createBucketIfNeeded(bucketName);

        final String uniqueFileNameGuid = StringUtils.isNotBlank(s3File.getId()) ? s3File.getId() : UUID.randomUUID().toString();

        try {
            if (amazonS3.headObject(HeadObjectRequest.builder().bucket(this.bucketName).key(uniqueFileNameGuid).build()).sdkHttpResponse().isSuccessful()) {
                throw new IllegalStateException("file already exists with id " + uniqueFileNameGuid);
            }
        } catch (NoSuchKeyException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(e.getMessage(), e);
            }
        }

        final Map<String, String> metadata = createObjectMetadata(s3File);

        amazonS3.putObject(PutObjectRequest.builder().bucket(this.bucketName).key(uniqueFileNameGuid).metadata(metadata).build(), createRequestBody(s3File));

        s3File.setId(uniqueFileNameGuid);

        return uniqueFileNameGuid;
    }

    protected RequestBody createRequestBody(S3File file) {
        return RequestBody.fromBytes(file.getByteContents());
    }

    protected Map<String, String> createObjectMetadata(S3File file) {
        final Map<String, String> metadata = new HashMap<>();

        if (file.getFileMetaData() != null) {
            final Map<String, Object> suppliedMetadata = file.getFileMetaData().getMetadata();
            if (suppliedMetadata != null) {
                final String contentType = (String) suppliedMetadata.remove(S3FileMetadata.CONTENT_TYPE);
                if (StringUtils.isNotBlank(contentType)) {
                    metadata.put(S3FileMetadata.CONTENT_TYPE, flattenToAscii(contentType));
                }

                final Long contentLength = (Long) suppliedMetadata.remove(S3FileMetadata.CONTENT_LENGTH);
                if (contentLength != null) {
                    metadata.put(S3FileMetadata.CONTENT_LENGTH, contentLength.toString());
                }

                final String cacheControl = (String) suppliedMetadata.remove(S3FileMetadata.CACHE_CONTROL);
                if (StringUtils.isNotBlank(cacheControl)) {
                    metadata.put(S3FileMetadata.CACHE_CONTROL, flattenToAscii(cacheControl));
                }

                final String contentEncoding = (String) suppliedMetadata.remove(S3FileMetadata.CONTENT_ENCODING);
                if (StringUtils.isNotBlank(contentEncoding)) {
                    metadata.put(S3FileMetadata.CONTENT_ENCODING, flattenToAscii(contentEncoding));
                }

                final String contentMd5 = (String) suppliedMetadata.remove(S3FileMetadata.CONTENT_MD5);
                if (StringUtils.isNotBlank(contentMd5)) {
                    metadata.put(S3FileMetadata.CONTENT_MD5, contentMd5);
                }

                final String contentLanguage = (String) suppliedMetadata.remove(S3FileMetadata.CONTENT_LANGUAGE);
                if (StringUtils.isNotBlank(contentLanguage)) {
                    metadata.put(S3FileMetadata.CONTENT_LANGUAGE, flattenToAscii(contentLanguage));
                }

                final Date lastModified = (Date) suppliedMetadata.remove(S3FileMetadata.LAST_MODIFIED);
                if (lastModified != null) {
                    metadata.put(S3FileMetadata.LAST_MODIFIED, lastModified.toString());
                }
            }

            if (StringUtils.isNotBlank(file.getFileMetaData().getFileName())) {
                metadata.put(S3FileMetadata.CONTENT_DISPOSITION, "attachment; filename=" + flattenToAscii(file.getFileMetaData().getFileName()));
            }

            if (suppliedMetadata != null) {
                //any remaining entries add as user supplied metadata
                suppliedMetadata.entrySet().stream()
                        .filter(e -> e.getKey() != null && e.getValue() != null)
                        .forEach(e -> metadata.put( flattenToAscii(e.getKey()), flattenToAscii(String.valueOf(e.getValue()))));
            }
        }
        metadata.put(S3FileMetadata.SERVER_SIDE_ENCRYPTION, S3FileMetadata.SERVER_SIDE_ENCRYPTION_ALGORITHM);
        return metadata;
    }

    private static String flattenToAscii(String string) {
        final var sb = new StringBuilder(string.length());
        string = Normalizer.normalize(string, Normalizer.Form.NFD);
        for (char c : string.toCharArray()) {
            if (c <= '\u007F') sb.append(c);
        }
        return sb.toString();
    }

    @Override
    public S3File retrieveFile(String id) {
        if (StringUtils.isBlank(id)) {
            throw new IllegalArgumentException("id cannot be blank");
        }

        final ResponseInputStream<GetObjectResponse> s3Object;
        try {
            s3Object = amazonS3.getObject(GetObjectRequest.builder().bucket(this.bucketName).key(id).build());
        } catch (NoSuchKeyException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(e.getMessage(), e);
            }
            return null;
        }

        final S3File s3File = new S3File();
        try {
            s3File.setByteContents(IoUtils.toByteArray(s3Object));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        final Map<String, String> metadata = new HashMap<>(s3Object.response().metadata());

        final S3FileMetadata s3FileMetadata = new S3FileMetadata();
        s3FileMetadata.setMetadata(metadata);
        s3FileMetadata.setFileName(metadata.get(S3FileMetadata.FILE_NAME));

        s3File.setFileMetaData(s3FileMetadata);
        s3File.setId(id);

        return s3File;
    }

    @Override
    public void updateFile(S3File s3File) {
        if (s3File == null) {
            throw new IllegalArgumentException("file cannot be null");
        }

        if (StringUtils.isBlank(s3File.getId())) {
            throw new IllegalArgumentException("file id cannot be blank");
        }

        if (s3File.getByteContents() == null || s3File.getByteContents().length == 0) {
            throw new IllegalArgumentException("bytes cannot be null or empty");
        }

        try {
            if (!amazonS3.headObject(HeadObjectRequest.builder().bucket(this.bucketName).key(s3File.getId()).build()).sdkHttpResponse().isSuccessful()) {
                throw new IllegalStateException("file does not exist with id " + s3File.getId());
            }
        } catch (NoSuchKeyException e) {
            throw new IllegalStateException("file does not exist with id " + s3File.getId(), e);
        }

        final Map<String, String> metadata = createObjectMetadata(s3File);

        amazonS3.putObject(PutObjectRequest.builder().bucket(this.bucketName).key(s3File.getId()).metadata(metadata).build(), createRequestBody(s3File));
    }

    @Override
    public void deleteFile(String id) {
        if (StringUtils.isBlank(id)) {
            throw new IllegalArgumentException("id cannot be blank");
        }

        amazonS3.deleteObject(DeleteObjectRequest.builder().bucket(this.bucketName).key(id).build());
    }

    private void createBucketIfNeeded(String bucket) {
        try {
            final HeadBucketResponse headBucketResponse = amazonS3.headBucket(HeadBucketRequest.builder().bucket(bucket).build());
            if (headBucketResponse.sdkHttpResponse().statusCode() == 404) {
                createBucket(bucket);
            }
        } catch (NoSuchBucketException e) {
            createBucket(bucket);
        }
    }

    private void createBucket(String bucket) {
        amazonS3.createBucket(CreateBucketRequest.builder().bucket(bucket).build());
        if (isEncryptionEnabled()) {
            // If encryption is enabled and a KMS encryptionKey is specified by ID or ARN, then the S3 bucket will be set
            // to encrypt new objects using it by default. If no encryptionKey is provided, it will use the default AWS key.
            final ServerSideEncryptionByDefault.Builder encryptionConfigBuilder = ServerSideEncryptionByDefault.builder().sseAlgorithm(ServerSideEncryption.AWS_KMS);
            if (StringUtils.isNotBlank(getEncryptionKey())) {
                encryptionConfigBuilder.kmsMasterKeyID(getEncryptionKey());
            }
            amazonS3.putBucketEncryption(PutBucketEncryptionRequest.builder()
                    .bucket(bucket)
                    .serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration.builder()
                            .rules(ServerSideEncryptionRule.builder()
                                    .applyServerSideEncryptionByDefault(encryptionConfigBuilder.build())
                                    .build())
                            .build())
                    .build());
        }
        if (StringUtils.isNotBlank(getConfidentialDataTagName())) {
            amazonS3.putBucketTagging(PutBucketTaggingRequest.builder()
                    .bucket(bucket).tagging(Tagging.builder()
                            .tagSet(Tag.builder()
                                    .key(getConfidentialDataTagName())
                                    .value(Boolean.toString(isConfidentialData()))
                                    .build())
                            .build())
                    .build());
        }
    }

    public String getBucketName() {
        return bucketName;
    }

    public void setBucketName(String bucketName) {
        this.bucketName = bucketName;
    }

    public String getReplicationBucketName() {
        return replicationBucketName;
    }

    public void setReplicationBucketName(String replicationBucketName) {
        this.replicationBucketName = replicationBucketName;
    }

    public S3Client getAmazonS3() {
        return amazonS3;
    }

    public void setAmazonS3(S3Client amazonS3) {
        this.amazonS3 = amazonS3;
    }

    public boolean isEncryptionEnabled() {
        return encryptionEnabled;
    }

    public void setEncryptionEnabled(boolean encryptionEnabled) {
        this.encryptionEnabled = encryptionEnabled;
    }

    public String getEncryptionKey() {
        return encryptionKey;
    }

    /**
     * Sets a master KMS key to be used to encrypt auto-provisioned S3 buckets
     *
     * @param encryptionKey - A master KMS key ID or ARN
     */
    public void setEncryptionKey(String encryptionKey) {
        this.encryptionKey = encryptionKey;
    }

    public String getConfidentialDataTagName() {
        return confidentialDataTagName;
    }

    public void setConfidentialDataTagName(String confidentialDataTagName) {
        this.confidentialDataTagName = confidentialDataTagName;
    }

    public boolean isConfidentialData() {
        return confidentialData;
    }

    public void setConfidentialData(boolean confidentialData) {
        this.confidentialData = confidentialData;
    }
}
