/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.processors;

import org.apache.commons.lang.StringUtils;
import org.mule.api.*;
import org.mule.api.config.ConfigurationException;
import org.mule.api.construct.FlowConstruct;
import org.mule.api.construct.FlowConstructAware;
import org.mule.api.context.MuleContextAware;
import org.mule.api.devkit.NestedProcessorChain;
import org.mule.api.expression.ExpressionManager;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.registry.RegistrationException;
import org.mule.api.transformer.TransformerMessagingException;
import org.mule.config.i18n.CoreMessages;
import org.mule.config.i18n.MessageFactory;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.mock.MockModule;
import org.mule.munit.mock.model.Attribute;
import org.mule.munit.wrapped.transformers.ExpressionEvaluatorTransformer;
import org.mule.util.TemplateParser;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import static org.mule.munit.common.MunitCore.buildMuleStackTrace;

/**
 * <p>
 * Generic Mock Message Processor.
 * </p>
 *
 * @author Mulesoft Inc.
 */
public abstract class AbstractMockMessageProcessor implements Initialisable, MessageProcessor, MuleContextAware, FlowConstructAware {

    protected FlowConstruct flowConstruct;
    protected Object moduleObject;
    protected MuleContext muleContext;
    protected ExpressionManager expressionManager;
    protected TemplateParser.PatternInfo patternInfo;
    protected AtomicInteger retryCount;
    protected int retryMax;

    public void initialise() throws InitialisationException {
        retryCount = new AtomicInteger();
        expressionManager = muleContext.getExpressionManager();
        patternInfo = TemplateParser.createMuleStyleParser().getStyle();

        try {
            moduleObject = findOrCreate(MockModule.class, true);
        } catch (IllegalAccessException e) {
            throw new InitialisationException(MessageFactory.createStaticMessage("Cannot find object"), this);
        } catch (InstantiationException e) {
            throw new InitialisationException(MessageFactory.createStaticMessage("Cannot find object"), this);
        } catch (ConfigurationException e) {
            throw new InitialisationException(MessageFactory.createStaticMessage("Cannot find object"), this);
        } catch (RegistrationException e) {
            throw new InitialisationException(CoreMessages.initialisationFailure("org.mule.munit.mock.MockModule"), e, this);
        }

    }

    protected final Object findOrCreate(Class<?> moduleClass, boolean shouldAutoCreate)
            throws IllegalAccessException, InstantiationException, ConfigurationException, RegistrationException {

        Object temporaryObject = moduleObject;

        if (temporaryObject == null) {
            temporaryObject = (muleContext.getRegistry().lookupObject(moduleClass));
            if (temporaryObject == null) {
                if (shouldAutoCreate) {
                    temporaryObject = (moduleClass.newInstance());
                    muleContext.getRegistry().registerObject(moduleClass.getName(), temporaryObject);
                } else {
                    throw new ConfigurationException(MessageFactory.createStaticMessage("Cannot find object"));
                }
            }
        }
        return temporaryObject;
    }

    protected MockModule getModule(MuleEvent event, String methodName) throws MessagingException {
        MockModule castedModuleObject;
        if (moduleObject instanceof String) {
            castedModuleObject = ((MockModule) muleContext.getRegistry().lookupObject(((String) moduleObject)));
            if (castedModuleObject == null) {
                throw new MessagingException(CoreMessages.failedToCreate(methodName), event, new RuntimeException("Cannot find the configuration specified by the org.mule.munit.config-ref attribute."));
            }
        } else {
            castedModuleObject = ((MockModule) moduleObject);
        }
        return castedModuleObject;
    }


    protected Object evaluate(MuleMessage muleMessage, Object source) {
        if (source instanceof String) {
            String stringSource = ((String) source);
            if (stringSource.startsWith(patternInfo.getPrefix()) && stringSource.endsWith(patternInfo.getSuffix())) {
                return expressionManager.evaluate(stringSource, muleMessage);
            } else {
                return expressionManager.parse(stringSource, muleMessage);
            }
        }
        return source;
    }

    /**
     * <p>
     * Executes the message processor code. In case of an assertion error it throws a new exception with the
     * mule stack trace (@since 3.4)
     * </p>
     *
     * @param event <p>
     *              The mule event to be processed.
     *              </p>
     * @return <p>
     * The result mule event
     * </p>
     * @throws org.mule.api.MuleException <p>
     *                                    In case of error. If the assertion fails, it throws an {@link AssertionError}
     *                                    </p>
     */
    public MuleEvent process(MuleEvent event) throws MuleException {
        MockModule module = getModule(event, getProcessor());
        try {
            doProcess(event, module);
            return event;
        } catch (AssertionError error) {
            AssertionError exception = new AssertionError(getMessage(error));
            exception.setStackTrace(buildMuleStackTrace(muleContext).toArray(new StackTraceElement[]{}));

            throw exception;
        } catch (Exception e) {
            throw new MessagingException(CoreMessages.failedToInvoke(getProcessor()), event, e);
        }
    }


    /**
     * <p>
     * The method that do the actual process
     * </p>
     *
     * @param muleEvent <p>The mule Message</p>
     * @param module    <p>The instance of the mock module</p>
     */
    protected abstract void doProcess(MuleEvent muleEvent, MockModule module) throws Exception;

    /**
     * @return <p>The name of the processor</p>
     */
    protected abstract String getProcessor();

    public void setMuleContext(MuleContext context) {
        this.muleContext = context;
    }

    public void setModuleObject(Object moduleObject) {
        this.moduleObject = moduleObject;
    }

    public void setRetryMax(int value) {
        this.retryMax = value;
    }


    public void setFlowConstruct(FlowConstruct flowConstruct) {
        this.flowConstruct = flowConstruct;
    }

    private String getMessage(AssertionError error) {
        String message = error.getMessage();
        if (StringUtils.isEmpty(message)) {
            return this.getProcessor();
        }
        return message;
    }


    protected List<Attribute> transformAttributes(Object attributes, MuleEvent muleEvent) throws TransformerMessagingException {
        ExpressionEvaluatorTransformer evaluator = new ExpressionEvaluatorTransformer();

        List<Attribute> transformedAttribute = new ArrayList<Attribute>();
        if (attributes != null) {
            for (Attribute a : (List<Attribute>) attributes) {
                Attribute result = new Attribute();
                if (isInvalidExpression(a.getName())) {
                    throw new MunitError("mock:with-attributes MEL parsing fail. The name " + a.getName() + "is an invalid expression.");
                }

                if (isInvalidExpression(a.getWhereValue())) {
                    throw new MunitError("mock:with-attributes MEL parsing fail. The whereValue " + a.getWhereValue() + "is an invalid expression.");
                }

                result.setName((String) evaluator.evaluate(this.muleContext, muleEvent, a.getName()));
                result.setWhereValue(evaluator.evaluate(this.muleContext, muleEvent, a.getWhereValue()));
                transformedAttribute.add(result);
            }
        }

        return transformedAttribute;
    }

    protected boolean isInvalidExpression(Object expression) {
        return isExpression(expression) && !isValidExpression(expression);
    }

    protected boolean isExpression(Object expression) {
        if( null == expression){
            return false;
        }

        if (!String.class.isAssignableFrom(expression.getClass())) {
            return false;
        }

        return StringUtils.isNotBlank((String) expression) && expressionManager.isExpression((String) expression);
    }

    protected boolean isValidExpression(Object expression) {
        if( null == expression){
            return false;
        }

        if (!String.class.isAssignableFrom(expression.getClass())) {
            return false;
        }

        return expressionManager.isValidExpression((String) expression);
    }

    protected List<NestedProcessor> transformAssertions(Object assertions, MuleEvent muleEvent) {
        final List<NestedProcessor> transformedAssertions = new ArrayList<NestedProcessor>();
        if (assertions != null) {
            for (MessageProcessor messageProcessor : ((List<MessageProcessor>) assertions)) {

                transformedAssertions.add(new NestedProcessorChain(muleEvent, muleContext, messageProcessor));
            }
        }
        return transformedAssertions;
    }

    protected Map<String, Object> evaluateMap(Map<String, Object> originalMap, MuleEvent muleEvent, MuleContext muleContext) {

        Map<String, Object> resultMap = new HashMap<String, Object>();
        if (null != originalMap) {

            ExpressionEvaluatorTransformer evaluator = new ExpressionEvaluatorTransformer();
            for (Map.Entry entry : originalMap.entrySet()) {
                if (isInvalidExpression(entry.getKey())) {
                    throw new MunitError("key attribute MEL parsing fail. The value: " + entry.getKey() + "is an invalid expression.");
                }

                if (isInvalidExpression(entry.getValue())) {
                    throw new MunitError("key attribute MEL parsing fail. The value: " + entry.getValue() + "is an invalid expression.");
                }

                String newKey = (String) evaluator.evaluate(muleContext, muleEvent, entry.getKey());
                Object newValue = evaluator.evaluate(muleContext, muleEvent, entry.getValue());
                resultMap.put(newKey, newValue);
            }
        }
        return resultMap;
    }


}
