package com.rsmart.kuali.coeus.hr.rest.authn;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.HashSet;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.DatatypeConverter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.kuali.coeus.sys.framework.service.KcServiceLocator;
import org.kuali.rice.core.api.CoreApiServiceLocator;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.kim.api.identity.IdentityService;
import org.kuali.rice.kim.api.identity.principal.PrincipalContract;
import org.kuali.rice.kim.api.permission.PermissionService;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.UserSession;
import org.kuali.rice.krad.service.LegacyDataAdapter;
import org.kuali.rice.krad.util.GlobalVariables;

public class KCBasicAuthFilter implements Filter {
  private static final Logger LOG = LoggerFactory.getLogger(KCBasicAuthFilter.class);

  private static final String   IMPORT_AUTHN_USER = "hrimport.authn.username";
  private static final String   IMPORT_AUTHN_PASS = "hrimport.authn.password";
  private static final String   IMPORT_AUTHN_RUN_AS = "hrimport.authn.runas";
  private static final String   CORE_AUTH_ENABLED = "auth.core.enabled";
  private static final String   PERMISSION_NAMESPACE = "KR-IDM";
  private static final String   PERMISSION_NAME = "Modify Entity";
  private static final String   AUTH_HEADER = "Authorization";

  protected LegacyDataAdapter       legacyDataAdapter = null;
  protected IdentityService         identityService = null;
  protected PermissionService		permissionService = null;
  protected HashSet<String>         authorizedUsers = null;
  protected String                  username = null;
  protected String                  password = null;
  protected String                  runAs = null;

  public LegacyDataAdapter getLegacyDataAdapter() {
    if (legacyDataAdapter == null) {
        legacyDataAdapter = KcServiceLocator.getService("legacyDataAdapter");
    }

    return legacyDataAdapter;
  }

  public void setLegacyDataAdapter(LegacyDataAdapter legacyDataAdapter) {
    this.legacyDataAdapter = legacyDataAdapter;
  }

  public IdentityService getIdentityService() {
    if (identityService == null) {
      identityService = KimApiServiceLocator.getIdentityService();
    }

    return identityService;
  }

  public void setIdentityService(IdentityService identityService) {
    this.identityService = identityService;
  }

  protected String getUsername() {
    return ConfigContext.getCurrentContextConfig().getProperty(IMPORT_AUTHN_USER);
  }

  protected String getPassword() {
    return ConfigContext.getCurrentContextConfig().getProperty(IMPORT_AUTHN_PASS);
  }

  protected String getRunAsUser() {
    return ConfigContext.getCurrentContextConfig().getProperty(IMPORT_AUTHN_RUN_AS);
  }
  
  protected Boolean isCoreAuthEnabled() {
	return ConfigContext.getCurrentContextConfig().getBooleanProperty(CORE_AUTH_ENABLED);
  }
  
  protected Boolean doesUserHavePermission(String principalId) {
	  return getPermissionService().hasPermission(principalId, PERMISSION_NAMESPACE, PERMISSION_NAME);
  }

  protected UserSession authenticateKCUser(final String principalName, final String password)
      throws GeneralSecurityException {
    PrincipalContract principal = null;

    final String username = getUsername();
    final String fixedPassword = getPassword();
    final String runas = getRunAsUser();

    if (username != null && fixedPassword != null) {
      if (runas == null) {
        LOG.error("no runas user set! Fixed credentials cannot be used without it. Please configure " + IMPORT_AUTHN_RUN_AS
            + " to indicate the valid KIM user to use for import");
      } else if (username.equals(principalName) && fixedPassword.equals(password)) {
        LOG.debug ("user authenticated against fixed username and password");
        principal = getIdentityService().getPrincipalByPrincipalName(runas);
        if (principal == null) {
          LOG.error ("could not retrieve runas user '" + runas + "' -- user cannot authenitcate!");
          return null;
        }
      } else {
        return null;
      }
    } else {
      LOG.debug ("no fixed username and password configured: authenticating against KIM\n configure " + IMPORT_AUTHN_USER +
          ", " + IMPORT_AUTHN_PASS + ", and " + IMPORT_AUTHN_RUN_AS + " to authentication against fixed credentials");
    }

    if (principal == null) {
      if (authorizedUsers != null) {
        if (!authorizedUsers.contains(principalName)) {
          LOG.debug(principalName + " is not in the authorized users list: aborting authentication");
          return null;
        }
      }

      String convertedPw = null;

      convertedPw = CoreApiServiceLocator.getEncryptionService().hash(password);

      principal = getIdentityService().getPrincipalByPrincipalNameAndPassword(principalName, convertedPw);
    }

    if (principal != null) {
      final UserSession userSession = new UserSession(principal.getPrincipalName());
      GlobalVariables.setUserSession(userSession);
      return userSession;
    }

    LOG.debug("unable to retrieve user " + principalName + " with the supplied password");

    return null;
  }

  @Override
  public void doFilter(final ServletRequest request, final ServletResponse response,
      final FilterChain filterChain) throws IOException, ServletException {
    final HttpServletRequest httpReq = (HttpServletRequest)request;
    final String authHeader = httpReq.getHeader(AUTH_HEADER);
    
    if (isCoreAuthEnabled()) {
    	if (!this.doesUserHavePermission(GlobalVariables.getUserSession().getPrincipalId())) {
            LOG.error("user does not have approriate permission for access the hr-import server");
            final HttpServletResponse httpResp = (HttpServletResponse)response;
            httpResp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
            return;    		
    	}
      filterChain.doFilter(request, response);
      return;
    }

    UserSession session = null;

    if (authHeader != null) {
      final String[] parts = authHeader.split("\\s+");
      if (parts.length == 2 && "basic".equalsIgnoreCase(parts[0])) {
        final String plaintext = new String(DatatypeConverter.parseBase64Binary(parts[1]));
        if (plaintext != null && plaintext.length() > 0) {
          final String[] credentials = plaintext.split(":");
          try {
            session = authenticateKCUser(credentials[0], credentials[1]);
          } catch (GeneralSecurityException e) {
            LOG.error("security exception encountered during authentication", e);
            final HttpServletResponse httpResp = (HttpServletResponse)response;
            httpResp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            return;
          }
        }
      }
    }

    if (session == null) {
      LOG.debug ("user is not authenticated");
      final HttpServletResponse httpResp = (HttpServletResponse)response;
      httpResp.setHeader("WWW-Authentication", "Basic realm=\"" + httpReq.getRequestURI() + "\"");
      httpResp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
      return;
    } else {
      LOG.debug("authenticated user: '" + session.getPrincipalName() + "'");
      filterChain.doFilter(request, response);
    }
  }

  @Override
  public void init(final FilterConfig config) throws ServletException {
    final String hrUsers = config.getInitParameter("hrUsers");
    if (hrUsers != null && !hrUsers.isEmpty()) {
      final String users[] = hrUsers.split(",");
      if(users.length > 0) {
        final StringBuffer sb = new StringBuffer("HR import authorized users resricted to: ");
        String delim = "";
        authorizedUsers = new HashSet<String> (users.length);
        for (final String user : users) {
          sb.append(delim).append(user);
          delim = ",";
          authorizedUsers.add(user);
        }
        LOG.debug(sb.toString());
      }
    }
  }

  @Override
  public void destroy() {
  }

public PermissionService getPermissionService() {
	if (permissionService == null) {
		permissionService = KcServiceLocator.getService(PermissionService.class);
	}
	return permissionService;
}

public void setPermissionService(PermissionService permissionService) {
	this.permissionService = permissionService;
}

}
