package com.rsmart.kuali.coeus.hr.service.adapter.impl;

import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kim.impl.identity.affiliation.EntityAffiliationBo;
import org.kuali.rice.kim.impl.identity.employment.EntityEmploymentBo;
import com.rsmart.kuali.coeus.hr.rest.model.Affiliation;
import com.rsmart.kuali.coeus.hr.rest.model.Employment;
import com.rsmart.kuali.coeus.hr.service.adapter.PersistableBoMergeAdapter;
import org.kuali.rice.krad.service.LegacyDataAdapter;

/**
 * Implements logic necessary for merging {@link com.rsmart.kuali.coeus.hr.rest.model.Affiliation Affiliation}
 * objects from import into the list of {@link org.kuali.rice.kim.impl.identity.employment.EntityEmploymentBo EntityEmploymentBo}
 * objects already attached to an Entity.
 * 
 * Note that this class is not bound to {@link com.rsmart.kuali.coeus.hr.rest.model.Employment Employment} JAXB
 * objects. This is because KIM implements EntityEmploymentBo as a dependent object on an EntityAffiliationBo
 * object. When updating an employment record we must update both an EntityEmploymentBo and an EntityAffiliationBo.
 * This requires special handling in {@link com.rsmart.kuali.coeus.hr.service.impl.HRImportServiceImpl#updateEntityBo}
 * to handle Affiliations with no employment record and Affiliations with an employment record separately.
 * 
 * @author duffy
 *
 */
public class EntityEmploymentBoAdapter extends
    PersistableBoMergeAdapter<EntityEmploymentBo, Employment> {

  protected int compareAppointment(final EntityAffiliationBo aff0, final EntityAffiliationBo aff1) {
    int comp = 0;
    if (aff0 == null && aff1 == null) {
    	return 0;
    } else if (aff0 != null && aff1 == null) {
    	return 1;
    } else if (aff0 == null && aff1 != null) {
    	return -1;
    }
    comp = nullSafeCompare(aff0.getAffiliationTypeCode(),aff1.getAffiliationTypeCode());
    if (comp != 0) {
      return comp;
    }
    
    return nullSafeCompare(aff0.getCampusCode(),aff1.getCampusCode());
  }
  
  @Override
  public int compareBOProperties(EntityEmploymentBo emp0, EntityEmploymentBo emp1) {
    int comp = 0;
        
    comp = nullSafeCompare(emp0.getEmployeeId(), emp1.getEmployeeId());
    if (comp != 0) {
      return comp;
    }
    
    comp = nullSafeCompare(emp0.getEmployeeStatusCode(), emp1.getEmployeeStatusCode());
    if (comp != 0) {
      return comp;
    }
    
    comp = nullSafeCompare(emp0.getEmployeeTypeCode(), emp1.getEmployeeTypeCode());
    if (comp != 0) {
      return comp;
    }
    
    comp = nullSafeCompare(emp0.getPrimaryDepartmentCode(), emp1.getPrimaryDepartmentCode());
    if (comp != 0) {
      return comp;
    }
    
    final KualiDecimal emp0Base = emp0.getBaseSalaryAmount();
    final KualiDecimal emp1Base = emp1.getBaseSalaryAmount();
    
    if (emp0Base == null && emp1Base != null) {
      return -1;
    } else if (emp0Base != null && emp1Base == null) {
      return 1;
    } else if (emp0Base != null && emp1Base != null) {
      comp = emp0.getBaseSalaryAmount().compareTo(emp1.getBaseSalaryAmount());
    } else {
      comp = 0;
    }
    if (comp != 0) {
      return comp;
    }
    
    return comp;
  }

  @Override
  public EntityEmploymentBo newBO(String entityId) {
    final EntityEmploymentBo empl = new EntityEmploymentBo();
    
    empl.setEntityId(entityId);
    
    return empl;
  }

  @Override
  public Class<EntityEmploymentBo> getBusinessObjectClass() {
    return EntityEmploymentBo.class;
  }

  @Override
  public Class<Employment> getIncomingClass() {
    return Employment.class;
  }

  @Override
  public EntityEmploymentBo setFields(EntityEmploymentBo bo, Employment source) {    
    bo.setActive(true);
    bo.setBaseSalaryAmount(new KualiDecimal(source.getBaseSalaryAmount()));
    bo.setEmployeeId(source.getEmployeeId());
    bo.setEmployeeStatusCode(source.getEmployeeStatus());
    bo.setEmployeeTypeCode(source.getEmployeeType());
    bo.setPrimary(source.isPrimaryEmployment());
    bo.setPrimaryDepartmentCode(source.getPrimaryDepartment());
    bo.setEntityAffiliationId(source.getEntityAffiliationId());
    
    return bo;
  }
  
  @Override
  public EntityEmploymentBo setFields(final int index, EntityEmploymentBo bo, Employment source) {
    if (bo.getEmploymentRecordId() == null) {
      bo.setEmploymentRecordId(Integer.toString(index));
    }
    return setFields(bo, source);
  }
  
  /**
   * save(...) is overridden here in order to handle the dependency between employment and
   * affiliation records
   */
  @Override
  public EntityEmploymentBo save(final int index, final LegacyDataAdapter legacyDataAdapter, EntityEmploymentBo bo) {
    /* EntityEmploymentBo's must have a unique entityRecordId within their collection */
    if (bo.getEmploymentRecordId() == null) {
      bo.setEmploymentRecordId(Integer.toString(index));
    }

      return legacyDataAdapter.save(bo);
  }

  /**
   * save(...) is overridden here in order to handle the dependency between employment and
   * affiliation records
   */
  @Override
  public void delete(final LegacyDataAdapter legacyDataAdapter, EntityEmploymentBo bo) {
      legacyDataAdapter.delete(bo);
  }

}
