/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2020 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package co.kuali.coeus.data.migration;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.MigrationVersion;
import org.flywaydb.core.api.configuration.FluentConfiguration;
import org.flywaydb.core.api.resolver.MigrationResolver;
import co.kuali.coeus.data.migration.custom.CoeusMigrationResolver;
import javax.sql.DataSource;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Needs to be injected and migrated *before* OJB/JPA or any other database
 * framework. This only depends on a working DataSource, usually a JDBC
 * connection pool.
 */
public class FlywayMigrator {
    private static final Logger LOG = LogManager.getLogger(FlywayMigrator.class);

    protected String initVersion;
    protected String targetVersion;
    protected DataSource dataSource;
    protected DataSource riceDataSource;

    protected CoeusMigrationResolver coeusMigrationResolver;

    protected String sqlMigrationPath;
    protected String javaMigrationPath;
    protected List<String> extraMigrationPaths;

    //KC database scripts
    protected String kcPath = "kc";
    //Rice database scripts, applied whether embedded or bundled
    protected String ricePath = "rice";
    //Rice database scripts, applied if managing rice, should always be applied for bundled (manageRice)
    protected String riceServer = "rice_server";
    //Rice database scripts, applied if not managing rice (manageRice = false)
    protected String riceDataOnly = "rice_data_only";
    //KRC scripts, applied to the KC database is embedded
    protected String embeddedClientScripts = "kc/embedded_client_scripts";
    protected String bootstrapPath = "bootstrap";
    protected String testingPath = "testing";
    protected String stagingDataPath = "staging";
    protected String demoDataPath = "demo";
    protected String grmDataPath = "grm";
    
    protected String checksumUpdatesFilename = "checksum_updates.csv";

    protected Boolean enabled;
    protected Boolean applyTesting;
    protected Boolean applyStaging;
    protected Boolean applyDemo;
    protected Boolean outOfOrder;
    protected Boolean ignoreMissingMigrations;

    protected Boolean grm;
    protected Boolean manageRice;

    //if embeddedMode is false (either set or detected) implies manageRice
    protected Boolean embeddedMode;

    private String mysqlUpdateChecksumStatement = "update schema_version set checksum = ? where version = ?";
    private List<String> mysqlPreMigrationSql = new ArrayList<>();
    
    private String oracleUpdateChecksumStatement = "update \"schema_version\" set \"checksum\" = ? where \"version\" = ?";
    private List<String> oraclePreMigrationSql = new ArrayList<>();
    

    public void migrate() {
        if (!enabled) {
            LOG.info("Flyway Migration is not enabled. Skipping.");
            return;
        }
        if (dataSource == null) {
            throw new IllegalStateException("dataSource == null");
        }
        if (riceDataSource == null) {
            throw new IllegalStateException("riceDataSource == null");
        }

        if (!embeddedMode) {
            manageRice = true;
        }

        List<String> kcLocations = new ArrayList<>(buildLocations(kcPath));
        if (!embeddedMode) {
            kcLocations.addAll(buildLocations(ricePath));
            kcLocations.addAll(buildLocations(riceServer));
        } else {
            kcLocations.add(embeddedClientScripts);
        }

        if (grm) {
            kcLocations.add(grmDataPath);
        }

        coeusMigrationResolver = new CoeusMigrationResolver(riceDataSource);
        coeusMigrationResolver.setJavaMigrationPath(getJavaMigrationPath());
        performMigration(dataSource, kcLocations, coeusMigrationResolver);

        if (embeddedMode) {
            List<String> riceLocations = new ArrayList<>(buildLocations(ricePath));
            if (manageRice) {
                riceLocations.addAll(buildLocations(riceServer));
            } else {
                riceLocations.addAll(buildLocations(riceDataOnly));
            }
            performMigration(riceDataSource, riceLocations);
        }
    }

    protected void performMigration(DataSource dataSource, List<String> locations, MigrationResolver ... migrationResolvers) {

    	runPreMigrationSql(dataSource);

    	final List<String> migrationLocations = new ArrayList<>((prefixLocationsWithDb(getSqlMigrationPath(), locations)));
        getExtraMigrationPaths().stream().map(path -> prefixLocationsWithDb(path, locations)).forEach(migrationLocations::addAll);

        final FluentConfiguration flywayConfig = Flyway.configure();
        flywayConfig
                .dataSource(dataSource)
                .table("schema_version")
                .locations(filterForExistence(migrationLocations))
                .resolvers(migrationResolvers)
                .placeholderPrefix("PLACEHOLDERS_DISABLED$$$$$") //there is no way to turn off placeholder replacement and the default(${}) is used in sql scripts. So use a unlikely string to make sure no placeholders are detected
                .baselineOnMigrate(true)
                .baselineVersion(MigrationVersion.fromVersion(getInitVersion()))
                .target(MigrationVersion.fromVersion(getTargetVersion()))
                .ignoreMissingMigrations(getIgnoreMissingMigrations())
                .outOfOrder(getOutOfOrder());

        final var flyway = flywayConfig.load();

        if (LOG.isInfoEnabled()) {
            LOG.info(Stream.of(flyway.info().all()).map(i -> "flyway migration: " + i.getVersion() + " : '"
                    + i.getDescription() + "' from file: " + i.getScript() + "\n").collect(Collectors.joining()));
        }

        final int numApplied = flyway.migrate();
        LOG.info("flyway migrations applied: " + numApplied);
    }

    protected List<String> prefixLocationsWithDb(String dbPath, List<String> locations) {
        List<String> result = new ArrayList<>();
        for (String location : locations) {
            result.add(dbPath + "/" + location);
        }
        return result;
    }

    protected String[] filterForExistence(List<String> locations) {
        List<String> result = new ArrayList<>();
        for (String location : locations) {
            if (this.getClass().getClassLoader().getResource(location) != null) {
                result.add(location);
            }
        }
        return result.toArray(new String[0]);
    }
    
    protected void runPreMigrationSql(DataSource dataSource) {
    	try (Connection conn = dataSource.getConnection()) {
    		conn.setAutoCommit(true);
    	    String databaseProductName = conn.getMetaData().getDatabaseProductName();
    		if (StringUtils.containsIgnoreCase(databaseProductName, "oracle")) {
    			runChecksumPreMigration(oracleUpdateChecksumStatement, conn);
    			runSpecificPreMigrationSql(conn, oraclePreMigrationSql);
    		} else {
    			runChecksumPreMigration(mysqlUpdateChecksumStatement, conn);
    			runSpecificPreMigrationSql(conn, mysqlPreMigrationSql);
    		}
    	} catch (SQLException e) {
    		LOG.warn("Error getting connection to run pre migration sql", e);
    	}
    }
    
    void runChecksumPreMigration(final String checksumUpdateStatement, final Connection conn) {
    	runChecksumPreMigrationFromLocation(checksumUpdateStatement, getSqlMigrationPath(), conn);
    	for (String path : getExtraMigrationPaths()) {
    		runChecksumPreMigrationFromLocation(checksumUpdateStatement, path, conn);
    	}
    }

	protected void runChecksumPreMigrationFromLocation(final String checksumUpdateStatement, final String location,
			final Connection conn) {
    			runChecksumUpdateSqlFromFile(checksumUpdateStatement, 
    					this.getClass().getClassLoader().getResource(location + "/" + this.checksumUpdatesFilename),
    					conn);
	}
    
    void runChecksumUpdateSqlFromFile(String checksumUpdatePattern, URL checksumFileURL, Connection conn) {
    	if (checksumFileURL == null) {
    		return;
    	}
    	try (Stream<String> stream = new BufferedReader(new InputStreamReader(checksumFileURL.openStream())).lines();
    			PreparedStatement stmt = conn.prepareStatement(checksumUpdatePattern)) {
    		stream.filter(StringUtils::isNotEmpty).map(line -> line.replaceAll("\\s", ""))
    			.forEach(line -> runChecksumUpdateStatementFromCsvLine(line, stmt));
    	} catch (IOException|SQLException e) {
    		if (LOG.isDebugEnabled()) {
    			LOG.debug("Unable to run checksum updates.", e);
    		} else {
    			LOG.warn("Unable to run checksum updates. " + e.getMessage());
    		}
    	}
    }

	void runChecksumUpdateStatementFromCsvLine(String line, PreparedStatement stmt) {
		String[] parts = line.split(",");
		try {
			stmt.setLong(1, Long.parseLong(parts[1]));
			stmt.setString(2, parts[0]);
			stmt.executeUpdate();
		} catch (SQLException e) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Unable to run checksum update - '" + line + "'", e);
			} else {
				LOG.warn("Unable to run checksum update - '" + line + "' - " + e.getMessage());
			}
		}
	}

	void runSpecificPreMigrationSql(final Connection conn, final List<String> preMigrationSql) {
		for (String preSql : preMigrationSql) {
			try (Statement stmt = conn.createStatement()) {
				stmt.executeUpdate(preSql);
			} catch (SQLException e) {
				if (LOG.isDebugEnabled()) {
					LOG.debug("Error running pre migration sql.", e);
				} else {
					LOG.warn("Error running pre migration sql " + e.getMessage());
				}
			}
		}
	}

    protected List<String> buildLocations(String rootPath) {
        List<String> locations = new ArrayList<>();
        locations.add(rootPath + "/" + bootstrapPath);
        if (getApplyTesting()) {
            locations.add(rootPath + "/" + testingPath);
        }
        if (getApplyStaging()) {
            locations.add(rootPath + "/" + stagingDataPath);
        }
        if (getApplyDemo()) {
            locations.add(rootPath + "/" + demoDataPath);
        }
        return locations;
    }

    protected Boolean getDefinedOption(String option, Boolean defaultValue) {
        if (System.getProperty(option) != null) {
            return Boolean.valueOf(System.getProperty(option));
        } else if (System.getenv().containsKey(option)) {
            return Boolean.valueOf(System.getProperty(option));
        } else {
            return defaultValue;
        }
    }

    protected String getDefinedOption(String option, String defaultValue) {
        if (System.getProperty(option) != null) {
            return System.getProperty(option);
        } else if (System.getenv().containsKey(option)) {
            return System.getProperty(option);
        } else {
            return defaultValue;
        }
    }

    public void setInitVersion(final String initVersion) {
        this.initVersion = initVersion;
    }

    public void setTargetVersion(String targetVersion) {
        this.targetVersion = targetVersion;
    }

    /**
     * Sets the KC datasource
     */
    public void setDataSource(final DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public DataSource getRiceDataSource() {
        return riceDataSource;
    }

    /**
     * Sets the Rice server datasource
     */
    public void setRiceDataSource(DataSource riceDataSource) {
        this.riceDataSource = riceDataSource;
    }

    public String getBootstrapPath() {
        return bootstrapPath;
    }

    /**
     * Sets the path for the bootstrap data scripts
     * @param bootstrapPath (default : bootstrap)
     */
    public void setBootstrapPath(String bootstrapPath) {
        this.bootstrapPath = bootstrapPath;
    }

    public String getTestingPath() {
        return testingPath;
    }

    /**
     * Sets the path for the testing data scripts
     * @param testingPath (default : testing)
     */
    public void setTestingPath(String testingPath) {
        this.testingPath = testingPath;
    }

    public String getStagingDataPath() {
        return stagingDataPath;
    }

    /**
     * Sets the path for staging data scripts
     * @param stagingDataPath (default : staging)
     */
    public void setStagingDataPath(String stagingDataPath) {
        this.stagingDataPath = stagingDataPath;
    }

    public String getDemoDataPath() {
        return demoDataPath;
    }

    /**
     * Sets the path for the demo data scripts
     * @param demoDataPath (default : demo)
     */
    public void setDemoDataPath(String demoDataPath) {
        this.demoDataPath = demoDataPath;
    }

    public String getGrmDataPath() {
        return grmDataPath;
    }

    /**
     * Sets the path for the grm data scripts
     * @param grmDataPath (default : grm)
     */
    public void setGrmDataPath(String grmDataPath) {
        this.grmDataPath = grmDataPath;
    }


    public Boolean getApplyTesting() {
        if (applyTesting == null) {
            applyTesting = getDefinedOption("kc.flyway.testing", Boolean.FALSE);
        }
        return applyTesting;
    }

    /**
     * Sets whether to apply testing data scripts or not
     * @param applyTesting (default : Configuration parameter flyway.migrations.apply_testing or false if not set)
     */
    public void setApplyTesting(boolean applyTesting) {
        this.applyTesting = applyTesting;
    }

    public Boolean getApplyStaging() {
        if (applyStaging == null) {
            applyStaging = getDefinedOption("kc.flyway.staging", Boolean.FALSE);
        }
        return applyStaging;
    }

    /**
     * Sets whether to apply staging data scripts or not
     * @param applyStaging (default : Configuration parameter flyway.migrations.apply_staging or false if not set)
     */
    public void setApplyStaging(boolean applyStaging) {
        this.applyStaging = applyStaging;
    }

    public Boolean getApplyDemo() {
        if (applyDemo == null) {
            applyDemo = getDefinedOption("kc.flyway.demo", Boolean.FALSE);
        }
        return applyDemo;
    }

    /**
     * Sets whether to apply demo data scripts or not
     * @param applyDemo (default : Configuration parameter flyway.migrations.apply_demo or false if not set)
     */
    public void setApplyDemo(boolean applyDemo) {
        this.applyDemo = applyDemo;
    }

    public String getInitVersion() {
    	if (initVersion == null) {
    		initVersion = getDefinedOption("kc.flyway.baseline.version", "0");
    	}
        return initVersion;
    }

    public String getTargetVersion() {
        if (targetVersion == null) {
            targetVersion = getDefinedOption("kc.flyway.target.version", Long.toString(Long.MAX_VALUE));
        }
        return targetVersion;
    }

    public DataSource getDataSource() {
        return dataSource;
    }

    public Boolean getManageRice() {
        if (manageRice == null) {
            manageRice = getDefinedOption("kc.flyway.manageRice", Boolean.TRUE);
        }
        return manageRice;
    }

    public void setManageRice(Boolean manageRice) {
        this.manageRice = manageRice;
    }

    public Boolean getEmbeddedMode() {
        if (embeddedMode == null) {
            embeddedMode = getDefinedOption("kc.flyway.embedded", Boolean.FALSE);
        }
        return embeddedMode;
    }

    public void setEmbeddedMode(Boolean embeddedMode) {
        this.embeddedMode = embeddedMode;
    }

    public Boolean getGrm() {
        if (embeddedMode == null) {
            embeddedMode = getDefinedOption("kc.flyway.grm", Boolean.FALSE);
        }
        return embeddedMode;
    }

    public void setGrm(Boolean grm) {
        this.grm = grm;
    }

    public CoeusMigrationResolver getCoeusMigrationResolver() {
        return coeusMigrationResolver;
    }

    public void setCoeusMigrationResolver(
            CoeusMigrationResolver coeusMigrationResolver) {
        this.coeusMigrationResolver = coeusMigrationResolver;
    }

    public Boolean getEnabled() {
        if (enabled == null) {
            enabled = getDefinedOption("kc.flyway.enabled", Boolean.FALSE);
        }
        return enabled;
    }

    public void setEnabled(Boolean enabled) {
        this.enabled = enabled;
    }

    public String getSqlMigrationPath() {
        if (sqlMigrationPath == null) {
            sqlMigrationPath = getDefinedOption("kc.flyway.sql.migration.path", "");
        }

        return sqlMigrationPath;
    }

    public void setSqlMigrationPath(String migrationPath) {
        this.sqlMigrationPath = migrationPath;
    }

    public String getJavaMigrationPath() {
        if (javaMigrationPath == null) {
            javaMigrationPath = getDefinedOption("kc.flyway.java.migration.path", "");
        }

        return javaMigrationPath;
    }

    public void setJavaMigrationPath(String javaMigrationPath) {
        this.javaMigrationPath = javaMigrationPath;
    }

	public List<String> getExtraMigrationPaths() {
		if (extraMigrationPaths == null) {
			extraMigrationPaths = Arrays.asList(getDefinedOption("kc.flyway.sql.extra.paths", "").split(","));
		}
		return extraMigrationPaths;
	}

	public void setExtraMigrationPaths(List<String> extraMigrationPaths) {
		this.extraMigrationPaths = extraMigrationPaths;
	}

	public Boolean getOutOfOrder() {
		if (outOfOrder == null) {
			outOfOrder = getDefinedOption("kc.flyway.outOfOrder", Boolean.FALSE);
		}
		return outOfOrder;
	}

	public void setOutOfOrder(Boolean outOfOrder) {
		this.outOfOrder = outOfOrder;
	}

	public Boolean getIgnoreMissingMigrations() {
		if (ignoreMissingMigrations == null) {
            ignoreMissingMigrations = getDefinedOption("kc.flyway.ignoreMissingMigrations", Boolean.FALSE);
		}
		return ignoreMissingMigrations;
	}

	public void setIgnoreMissingMigrations(Boolean ignoreMissingMigrations) {
		this.ignoreMissingMigrations = ignoreMissingMigrations;
	}
}
