/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate;

import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.kuali.coeus.propdev.api.s2s.S2sUserAttachedFormContract;
import org.kuali.coeus.propdev.api.s2s.UserAttachedFormService;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.impl.util.XmlBeansUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.util.*;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

/**
 * 
 * This class is used as a service implementation that is used to create instances of opportunity form generator classes. It
 * provides an abstraction level over the different generator class implementations.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@Component("s2SFormGeneratorRetrievalService")
public class S2SFormGeneratorRetrievalServiceImpl implements S2SFormGeneratorRetrievalService, InitializingBean {

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    @Qualifier("userAttachedFormService")
    private UserAttachedFormService userAttachedFormService;

    private Map<String, String> namespaceToGenerator;

    @Override
    public <T extends XmlObject> S2SFormGenerator<T> getS2SGenerator(String namespace) throws S2SException {
        final String generatorName = namespaceToGenerator.get(namespace);
        if (StringUtils.isNotBlank(generatorName)) {
            try {
                return (S2SFormGenerator<T>) applicationContext.getBean(generatorName);
            } catch (NoSuchBeanDefinitionException e) {
                throw new S2SException("Generator not found for namespace: " + namespace, e);
            }
        } else {
            throw new S2SException("Generator not found for namespace: " + namespace);
        }

    }

    @Override
    public final <T extends XmlObject> S2SFormGenerator<T> getS2SGenerator(String namespace, String proposalNumber) throws S2SException {
        S2SFormGenerator<T> supportedFormGenerator;
        try {
            supportedFormGenerator = getS2SGenerator(namespace);
        } catch (S2SException e) {
            supportedFormGenerator = null;
        }

        final S2sUserAttachedFormContract s2sUserAttachedForm = getUserAttachedFormService().findFormByProposalNumberAndNamespace(proposalNumber, namespace);
        try {
            final S2SFormGenerator<T> formGenerator = s2sUserAttachedForm != null ? (S2SFormGenerator<T>) applicationContext.getBean("UserAttachedFormGenerator") : supportedFormGenerator;
            if (formGenerator instanceof DynamicForm) {
                ((DynamicForm) formGenerator).setS2sUserAttachedForm(s2sUserAttachedForm);
                ((DynamicForm) formGenerator).setSupportedS2SFormGenerator(Optional.ofNullable(supportedFormGenerator));
            }

            return formGenerator;
        } catch (NoSuchBeanDefinitionException e) {
            throw new S2SException("Generator not found for namespace: " + namespace, e);
        }
    }

    @Override
    public <T extends XmlObject> List<S2SFormGenerator<T>> getS2SGenerators(Set<String> namespaces, String proposalNumber) throws S2SException {
        return namespaces.stream()
                .map(namespace -> this.<T>getS2SGenerator(namespace, proposalNumber))
                .sorted(Comparator.comparingInt(S2SFormGenerator::getSortIndex))
                .toList();
    }

    @Override
    public <T extends XmlObject> List<S2SFormGenerator<T>> getS2SGenerators() throws S2SException {
        return applicationContext.getBeansOfType(S2SFormGenerator.class)
                .values()
                .stream()
                .map(s2SFormGenerator -> (S2SFormGenerator<T>) s2SFormGenerator)
                .filter(s2SFormGenerator -> !(s2SFormGenerator instanceof DynamicForm)).toList();
    }

    @Override
    public XmlOptions getXmlOptionsPrefixes() {
        return XmlBeansUtils.getXmlOptionsPrefixes();
    }

    public UserAttachedFormService getUserAttachedFormService() {
        return userAttachedFormService;
    }

    public void setUserAttachedFormService(UserAttachedFormService userAttachedFormService) {
        this.userAttachedFormService = userAttachedFormService;
    }

    @Override
    public void afterPropertiesSet() {
        namespaceToGenerator = Arrays.stream(applicationContext.getBeanNamesForType(S2SFormGenerator.class))
                .map(beanName -> {
                    final S2SFormGenerator<?> s2SFormGenerator = (S2SFormGenerator<?>) applicationContext.getBean(beanName);
                    return !(s2SFormGenerator instanceof DynamicForm) ?
                        entry(s2SFormGenerator.getNamespace(), s2SFormGenerator.getBeanName()) : null;
                })
                .filter(Objects::nonNull)
                .collect(entriesToMap());
    }
}
