/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.phsCoverLetter12V12.PHSCoverLetter12Document;
import gov.grants.apply.forms.phsCoverLetter12V12.PHSCoverLetter12Document.PHSCoverLetter12;
import gov.grants.apply.forms.phsCoverLetter12V12.PHSCoverLetter12Document.PHSCoverLetter12.CoverLetterFile;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;


@FormGenerator("PHSCoverLetterV1_2Generator")
public class PHSCoverLetterV1_2Generator extends PHSCoverLetterBaseGenerator<PHSCoverLetter12Document> implements S2SFormGeneratorPdfFillable<PHSCoverLetter12Document> {

    @Value("http://apply.grants.gov/forms/PHS_CoverLetter_1_2-V1.2")
    private String namespace;

    @Value("PHS_CoverLetter_1_2-V1.2")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PHS_CoverLetter-V1.2.fo.xsl")
    private List<Resource> stylesheets;

	@Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/PHS_CoverLetter_1_2-V1.2.pdf")
	private Resource pdfForm;

    @Value("110")
    private int sortIndex;

	/**
	 * This method is used to get PHSCoverLetter12Document attachment from the
	 * narrative attachments.
	 * 
	 * @return phsCoverLetter12Document {@link XmlObject} of type
	 *         PHS398CoverLetterDocument.
	 */
	private PHSCoverLetter12Document getPHSCoverLetter() {

		PHSCoverLetter12Document phsCoverLetterDocument = PHSCoverLetter12Document.Factory
				.newInstance();
		PHSCoverLetter12 phsCoverLetter = PHSCoverLetter12.Factory
				.newInstance();
        CoverLetterFile coverLetterFile = CoverLetterFile.Factory.newInstance();
		phsCoverLetter.setFormVersion(FormVersion.v1_2.getVersion());
		AttachedFileDataType attachedFileDataType = null;
		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
				.getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null
					&& Integer.parseInt(narrative.getNarrativeType().getCode()) == NARRATIVE_PHS_COVER_LETTER) {
				attachedFileDataType = addAttachedFileType(narrative);
				if(attachedFileDataType != null){
					coverLetterFile.setCoverLetterFilename(attachedFileDataType);
					break;
				}
			}
		}
        phsCoverLetter.setCoverLetterFile(coverLetterFile);
		phsCoverLetterDocument.setPHSCoverLetter12(phsCoverLetter);
		return phsCoverLetterDocument;
	}

	/**
	 * This method creates {@link XmlObject} of type
	 * {@link PHSCoverLetter12Document} by populating data from the given
	 * {@link ProposalDevelopmentDocumentContract}
	 * 
	 * @param proposalDevelopmentDocument
	 *            for which the {@link XmlObject} needs to be created
	 * @return {@link XmlObject} which is generated using the given
	 *         {@link ProposalDevelopmentDocumentContract}
	 */
	@Override
    public PHSCoverLetter12Document getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		return getPHSCoverLetter();
	}

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

	@Override
	public Resource getPdfForm() {
		return pdfForm;
	}

	public void setPdfForm(Resource pdfForm) {
		this.pdfForm = pdfForm;
	}

	@Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

	@Override
	public S2SFormGeneratorPdfFillable.Attachments getMappedAttachments(PHSCoverLetter12Document form, List<AttachmentData> attachments) {
		final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
			final CoverLetterFile coverLetterFile = form.getPHSCoverLetter12().getCoverLetterFile();
			if (coverLetterFile != null && coverLetterFile.getCoverLetterFilename() != null && a.getContentId().equals(coverLetterFile.getCoverLetterFilename().getFileLocation().getHref())) {
				return entry("PHS_CoverLetter_1_2_P1.mandatoryFile0", a);
			}

			return entry((String) null, a);
		}).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

		return new S2SFormGeneratorPdfFillable.Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
				attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
	}

	@Override
	public DocumentFactory<PHSCoverLetter12Document> factory() {
		return PHSCoverLetter12Document.Factory;
	}
}
