/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.util;

import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

/**
 * Class processes XPath Queries.
 */
public class XPathExecutor {

	private Document doc;
	
	public XPathExecutor( String xml ) {

		try {
			init( xml );
		} catch (IOException|ParserConfigurationException|SAXException e) {
			throw new S2SException(e);
		}
	}
	
	/**
	 * Method evaulates the XPath expression against the xml string.
	 * Currently utilizing a DOM implementation.
	 * @return first node value returned
	 *
	 */
	public String execute( String xPath ) {

		if ( xPath == null ){
			return null;
		}
		
		// Evaluate the xpath expression
		try {
			return XPathFactory.newInstance().newXPath().evaluate(xPath, getDoc());
		} catch (XPathExpressionException e) {
			throw new S2SException(e);
		}


	}
	
	/**
	 * For a given XPath, a DOM Node that the XPath resolve to is returned.
	 * @param xpath A valid XPath referring to the Node that is to be returned
	 * @return The Node referred to by the xpath argument.
	 */
	public Node getNode(String xpath)  {
		try {
			return (Node) XPathFactory.newInstance().newXPath().evaluate(xpath, getDoc(), XPathConstants.NODE);
		} catch (XPathExpressionException e) {
			throw new S2SException(e);
		}
	}
	
	private void init( String xml ) throws IOException, ParserConfigurationException, SAXException {
	
		if ( xml == null ) {
			return;
		}
		

		try(ByteArrayInputStream stream = new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8))) {
		    DocumentBuilderFactory dfactory = SafeXmlUtils.safeDocumentBuilderFactory();
		    dfactory.setNamespaceAware(true);
		    setDoc( dfactory.newDocumentBuilder().parse(stream) );
		}
	}

	public Document getDoc() {
		return doc;
	}

	public void setDoc(Document doc) {
		this.doc = doc;
	}

}

