/*
 * Decompiled with CFR 0.152.
 */
package org.kuali.coeus.elasticsearch;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.xcontent.XContentType;
import org.kuali.coeus.common.framework.auth.perm.DocumentLevelPermissionable;
import org.kuali.coeus.common.framework.version.VersionStatus;
import org.kuali.coeus.common.framework.version.history.VersionHistoryDao;
import org.kuali.coeus.elasticsearch.ElasticsearchAccessControlService;
import org.kuali.coeus.elasticsearch.ElasticsearchIndexService;
import org.kuali.coeus.elasticsearch.GrantsSearchDocument;
import org.kuali.coeus.elasticsearch.serializers.ElasticsearchDocumentSerializer;
import org.kuali.rice.core.api.criteria.QueryByCriteria;
import org.kuali.rice.coreservice.framework.parameter.ParameterService;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.identity.PersonService;
import org.kuali.rice.krad.bo.DocumentHeader;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.DocumentService;
import org.springframework.beans.factory.annotation.Autowire;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;

@Configurable(autowire=Autowire.BY_TYPE)
@Transactional(readOnly=true, propagation=Propagation.REQUIRES_NEW)
public class ElasticsearchIndexServiceImpl
implements ElasticsearchIndexService {
    private static final Logger LOG = LogManager.getLogger(ElasticsearchIndexServiceImpl.class);
    private static final int INDEX_TIME_WARN_THRESHOLD = 5000;
    private static final int INDEX_PROGRESS_SIZE = 100;
    private static final QueryByCriteria PLACEHOLDER_DOC_CRITERIA = QueryByCriteria.Builder.forAttribute((String)"documentDescription", (Object)"*****PLACEHOLDER*****").build();
    @Autowired
    private List<ElasticsearchDocumentSerializer> serializers;
    @Autowired
    private DataObjectService dataObjectService;
    @Autowired
    private DocumentService documentService;
    @Autowired
    private ElasticsearchAccessControlService elasticsearchAccessControlService;
    @Autowired
    @Qualifier(value="elasticsearchClient")
    private RestHighLevelClient elasticsearchClient;
    @Autowired
    @Qualifier(value="elasticsearchObjectMapper")
    private ObjectMapper objectMapper;
    @Autowired
    private ParameterService parameterService;
    @Autowired
    private PersonService personService;
    @Autowired
    private VersionHistoryDao versionHistoryDao;
    @Autowired
    @Qualifier(value="elasticsearchForkJoinPool")
    private ForkJoinPool indexThreadPool;
    private String indexName = "documents";

    @Override
    public List<CompletableFuture<Boolean>> bulkIndex(List<String> documentIds) {
        this.filterInactiveDocumentIds(documentIds);
        int numDocs = documentIds.size();
        AtomicInteger progress = new AtomicInteger();
        AtomicInteger successfullyIndexed = new AtomicInteger();
        Stopwatch timer = Stopwatch.createStarted();
        LOG.info(String.format("Bulk indexing %d documents using %d threads...", numDocs, this.indexThreadPool.getParallelism()));
        List<CompletableFuture<Boolean>> results = documentIds.stream().map(docId -> CompletableFuture.supplyAsync(() -> this.retrieveDocument((String)docId).map(document -> {
            int numProcessed;
            boolean success = this.doIndex((Document)document);
            if (success) {
                successfullyIndexed.incrementAndGet();
            }
            if ((numProcessed = progress.incrementAndGet()) % 100 == 0) {
                LOG.info(String.format("%d / %d documents processed for indexing", numProcessed, numDocs));
            }
            return success;
        }).orElse(false), this.indexThreadPool)).collect(Collectors.toList());
        CompletableFuture.allOf(results.toArray(new CompletableFuture[0])).whenCompleteAsync((result, e) -> {
            timer.stop();
            LOG.info(String.format("Took %d seconds to index %d / %d documents", timer.elapsed(TimeUnit.SECONDS), successfullyIndexed.get(), progress.get()));
        });
        return results;
    }

    protected void filterInactiveDocumentIds(List<String> documentIds) {
        Set<String> inactiveVersions = Set.of(VersionStatus.CANCELED.name(), VersionStatus.ARCHIVED.name());
        documentIds.removeAll(this.versionHistoryDao.getAwardDocumentNumbersByVersionStatus(inactiveVersions));
        documentIds.removeAll(this.versionHistoryDao.getIPDocumentNumbersByVersionStatus(inactiveVersions));
        documentIds.removeAll(this.versionHistoryDao.getSubAwardDocumentNumbersByVersionStatus(inactiveVersions));
        documentIds.removeAll(this.getAwardPlaceholderDocIds());
    }

    protected Set<String> getAwardPlaceholderDocIds() {
        return this.dataObjectService.findMatching(DocumentHeader.class, PLACEHOLDER_DOC_CRITERIA).getResults().stream().map(DocumentHeader::getDocumentNumber).collect(Collectors.toSet());
    }

    @Override
    public void indexPostCommit(final String documentId) {
        TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new TransactionSynchronization(){

            public void afterCommit() {
                ElasticsearchIndexServiceImpl.this.index(documentId);
            }
        });
    }

    @Override
    public Future<Boolean> index(String docId) {
        return this.indexThreadPool.submit(() -> this.retrieveDocument(docId).map(this::doIndex).orElse(false));
    }

    @Override
    public Future<Boolean> index(Document document) {
        return this.indexThreadPool.submit(() -> this.doIndex(document));
    }

    protected boolean doIndex(final Document document) {
        Stopwatch timer = Stopwatch.createStarted();
        try {
            return this.buildIndexRequestForDocument(document).map(request -> {
                this.elasticsearchClient.indexAsync(request, RequestOptions.DEFAULT, (ActionListener)new ActionListener<IndexResponse>(){

                    public void onResponse(IndexResponse indexResponse) {
                        LOG.trace(String.format("Document %s indexed into Elasticsearch successfully", document.getDocumentNumber()));
                    }

                    public void onFailure(Exception e) {
                        LOG.error(String.format("Failed to index document %s into Elasticsearch", document.getDocumentNumber()), (Throwable)e);
                    }
                });
                timer.stop();
                long elapsed = timer.elapsed(TimeUnit.MILLISECONDS);
                if (elapsed > 5000L) {
                    LOG.warn(String.format("Took %d ms to index document %s into Elasticsearch", elapsed, document.getDocumentNumber()));
                }
                return true;
            }).orElse(false);
        }
        catch (Exception e) {
            LOG.error(String.format("Failed to build index request for document %s", document.getDocumentNumber()), (Throwable)e);
            return false;
        }
    }

    protected Optional<Document> retrieveDocument(String docId) {
        try {
            return Optional.ofNullable(this.documentService.getByDocumentHeaderId(docId));
        }
        catch (WorkflowException e) {
            LOG.error(String.format("Failed to retrieve document #%s", docId), (Throwable)e);
            return Optional.empty();
        }
    }

    protected Optional<IndexRequest> buildIndexRequestForDocument(Document document) {
        if (document instanceof DocumentLevelPermissionable) {
            return this.translateDocument((DocumentLevelPermissionable)document).map(esDoc -> {
                try {
                    IndexRequest indexRequest = new IndexRequest(this.indexName);
                    indexRequest.id(document.getDocumentNumber());
                    indexRequest.source(this.objectMapper.writeValueAsBytes(esDoc), XContentType.JSON);
                    return indexRequest;
                }
                catch (JsonProcessingException e) {
                    LOG.error(String.format("Exception encountered preparing document %s for indexing", document.getDocumentNumber()), (Throwable)e);
                    return null;
                }
            });
        }
        return Optional.empty();
    }

    @Override
    public Future<Boolean> delete(String documentId) {
        return this.indexThreadPool.submit(() -> this.doDelete(documentId));
    }

    protected boolean doDelete(final String documentId) {
        this.elasticsearchClient.deleteAsync(this.buildDeleteRequestForDocument(documentId), RequestOptions.DEFAULT, (ActionListener)new ActionListener<DeleteResponse>(){

            public void onResponse(DeleteResponse deleteResponse) {
                LOG.trace(String.format("Document %s deleted from Elasticsearch successfully", documentId));
            }

            public void onFailure(Exception e) {
                LOG.error(String.format("Failed to delete document %s from Elasticsearch", documentId), (Throwable)e);
            }
        });
        return true;
    }

    protected DeleteRequest buildDeleteRequestForDocument(String documentId) {
        return new DeleteRequest(this.indexName, documentId);
    }

    protected Optional<GrantsSearchDocument> translateDocument(DocumentLevelPermissionable document) {
        try {
            if (StringUtils.isBlank((CharSequence)document.getDocumentNumberForPermission())) {
                return Optional.empty();
            }
        }
        catch (NullPointerException e) {
            LOG.warn(String.format("Document %s has no associated transactional data", document.getDocumentNumber()), (Throwable)e);
            return Optional.empty();
        }
        return this.serializers.stream().filter(serializer -> serializer.supports(document)).findFirst().map(serializer -> {
            try {
                GrantsSearchDocument esDoc = serializer.translateDocument(document);
                esDoc.setViewers(this.elasticsearchAccessControlService.getDocumentViewers(document, serializer.getViewPermissions()));
                esDoc.setViewerUserNames(esDoc.getViewers().stream().map(arg_0 -> ((PersonService)this.personService).getPerson(arg_0)).filter(Objects::nonNull).map(Person::getPrincipalName).collect(Collectors.toSet()));
                esDoc.setViewUnits(this.elasticsearchAccessControlService.getSubUnitsForView(esDoc.getNamespace(), esDoc.getLeadUnitNumber()));
                return esDoc;
            }
            catch (RuntimeException e) {
                LOG.error(String.format("Failed to serialize document %s using %s", document.getDocumentNumber(), serializer.getClass().getSimpleName()), (Throwable)e);
                return null;
            }
        });
    }

    @Override
    public boolean isIdle() {
        return this.indexThreadPool.isQuiescent();
    }

    public void setSerializers(List<ElasticsearchDocumentSerializer> serializers) {
        this.serializers = serializers;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setElasticsearchClient(RestHighLevelClient elasticsearchClient) {
        this.elasticsearchClient = elasticsearchClient;
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    public void setIndexThreadPool(ForkJoinPool indexThreadPool) {
        this.indexThreadPool = indexThreadPool;
    }

    public void setIndexName(String indexName) {
        this.indexName = indexName;
    }
}

