/*-
 * #%L
 * %%
 * Copyright (C) 2019 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.research.gg.fetcher

import java.io.File
import org.apache.logging.log4j.kotlin.Logging
import org.kuali.research.gg.fetcher.fetch.GrantsGovFetchService
import org.kuali.research.gg.fetcher.fetch.SchemaManagerService
import org.kuali.research.gg.fetcher.fetch.search.SearchParams
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.CommandLineRunner
import org.springframework.core.io.Resource
import org.springframework.stereotype.Component

@Component
class GgFetcherCommandLineRunner(
    @Autowired val grantsGovFetchService: GrantsGovFetchService,
    @Autowired val schemaManagerService: SchemaManagerService
) : CommandLineRunner, Logging {

    override fun run(vararg args: String) {
        if (args.isEmpty() || args[0].isBlank()) {
            logger.error("Utility requires a valid, writable directory to save files in")
            return
        } else {
            val dir = File(args[0])
            if (!dir.isDirectory || !dir.exists()) {
                logger.error("Utility requires a valid, writable directory to save files in")
                return
            }
        }

        val savePath = args[0]

        schemaManagerService.refreshSchemaTree(listOf(META_MULTI_GRANT_APPLICATION_URL, META_GRANT_APPLICATION_FORM_URL, META_GRANT_APPLICATION_URL), "$savePath/xsd")

        @Suppress("RECEIVER_NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")
        val schemas = File(savePath, "xsd").listFiles().filter { it.isFile && it.name.endsWith(".xsd") }

        schemas.map { it.name.removeSuffix(".xsd") }.forEach {
            // cannot get the formId from the schema file name
            // save("$savePath/dat", grantsGovFetchService.fetchDat(it))
            save("$savePath/pdf", grantsGovFetchService.fetchPdf("$it.pdf"))
            save("$savePath/instructions", grantsGovFetchService.fetchInstructions("$it-Instructions.pdf"))
            save("$savePath/xsl", grantsGovFetchService.fetchXsl("$it.fo.xsl"))
            save("$savePath/xsl_nih", grantsGovFetchService.fetchNihXsl("$it.xsl"))
        }

        val all = grantsGovFetchService.search(SearchParams(formSet = "all"))
        val retired = grantsGovFetchService.search(SearchParams(formSet = "retiredForms"))
        val status = grantsGovFetchService.status()

        (all.families + retired.families).forEach { it.forms.parallelStream().forEach { form ->
            saveAll("$savePath/dat", grantsGovFetchService.fetchDat(form.name, form.version, form.id))
            saveAll("$savePath/pdf", grantsGovFetchService.fetchPdf(form.name, form.version, form.id))
            saveAll("$savePath/xsd", grantsGovFetchService.fetchSchema(form.name, form.version, form.id))
            saveAll("$savePath/instructions", grantsGovFetchService.fetchInstructions(form.name, form.version, form.id))
            saveAll("$savePath/xsl", grantsGovFetchService.fetchXsl(form.name, form.version, form.id))
            saveAll("$savePath/xsl_nih", grantsGovFetchService.fetchNihXsl(form.name, form.version, form.id))
        } }

        status.formStatuses.parallelStream().forEach { form ->
            saveAll("$savePath/dat", grantsGovFetchService.fetchDat(form.formName, form.formVersion, "${form.formId}"))
            saveAll("$savePath/pdf", grantsGovFetchService.fetchPdf(form.formName, form.formVersion, "${form.formId}"))
            saveAll("$savePath/xsd", grantsGovFetchService.fetchSchema(form.formName, form.formVersion, "${form.formId}"))
            saveAll("$savePath/instructions", grantsGovFetchService.fetchInstructions(form.formName, form.formVersion, "${form.formId}"))
            saveAll("$savePath/xsl", grantsGovFetchService.fetchXsl(form.formName, form.formVersion, "${form.formId}"))
            saveAll("$savePath/xsl_nih", grantsGovFetchService.fetchNihXsl(form.formName, form.formVersion, "${form.formId}"))
        }
    }

    private fun saveAll(dir: String, resourceMap: Map<String, Resource>) {
        resourceMap.forEach { save(dir, it.toPair()) }
    }

    private fun save(dir: String, resourcePair: Pair<String, Resource?>) {
        val (name, resource) = resourcePair
        if (resource != null) {
            File(dir, name).outputStream().use { output ->
                resource.inputStream.use { input ->
                    input.copyTo(output)
                }
            }
        }
    }

    companion object {
        private const val META_MULTI_GRANT_APPLICATION_URL = "https://apply07.grants.gov/apply/system/schemas/MetaMultiGrantApplication.xsd"
        private const val META_GRANT_APPLICATION_FORM_URL = "https://apply07.grants.gov/apply/system/schemas/MetaGrantApplicationForm.xsd"
        private const val META_GRANT_APPLICATION_URL = "https://apply07.grants.gov/apply/system/schemas/MetaGrantApplication.xsd"
    }
}
