/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.core;

import org.springframework.beans.factory.BeanFactory;

/**
 * Class that holds a singleton reference to NotificationServiceLocator
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class GlobalNotificationServiceLocator {

    /**
     * The NotificationServiceLocator singleton
     */
    private static NotificationServiceLocator locator;

    /**
     * The global initializer that constructs the NotificationServiceLocator singleton
     * @param beanFactory the beanFactory from which to construct the SpringNotificationServiceLocator
     */
    public static synchronized void init(BeanFactory beanFactory) {
        if (locator != null) {
            throw new IllegalStateException("GlobalNotificationServiceLocator already initialized");
        }
        locator = new SpringNotificationServiceLocator(beanFactory);
    }

    /**
     * Returns whether the GlobalNotificationServiceLocator has already been initialized (classloader scoped)
     * @return whether the GlobalNotificationServiceLocator has already been initialized (classloader scoped)
     */
    public static synchronized boolean isInitialized() {
        return locator != null;
    }

    /**
     * Un-sets the NotificationServiceLocator singleton, in order to fulfill a "lifecycle"
     * contract (whereby init may be called again in the same class loader), specifically for
     * unit tests. 
     */
    public static synchronized void destroy() {
        locator = null;
    }

    /**
     * Return the NotificationServiceLocator singleton
     * @return the NotificationServiceLocator singleton
     */
    public static NotificationServiceLocator getInstance() {
        return locator;
    }
}
