/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.engine;

import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.kuali.rice.kew.actionrequest.ActionRequestValue;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.engine.node.RouteNodeInstance;
import org.kuali.rice.kew.routeheader.DocumentContent;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.rice.kew.routeheader.StandardDocumentContent;


/**
 * Represents the current context of a Document being processed by the engine.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class RouteContext implements Serializable {

	private static final long serialVersionUID = -7125137491367944594L;

    private String id;

	private DocumentRouteHeaderValue routeHeader;

	private DocumentContent documentContent;

	private RouteNodeInstance nodeInstance;

	private EngineState engineState;

	private ActionRequestValue actionRequest;

	private ActivationContext activationContext = new ActivationContext(!ActivationContext.CONTEXT_IS_SIMULATION);

	private boolean doNotSendApproveNotificationEmails = false;

	private Map<String, Object> parameters = new HashMap<>();
	
	private boolean searchIndexingRequestedForContext = false;

	public RouteContext() {
        id = "";
	}

	private static final ThreadLocal<List<RouteContext>> ROUTE_CONTEXT_STACK = ThreadLocal.withInitial(() -> {
		List<RouteContext> contextStack = new LinkedList<>();
		contextStack.add(0, new RouteContext());
		return contextStack;
	});

	public static RouteContext getCurrentRouteContext() {
		return ROUTE_CONTEXT_STACK.get().get(0);
	}

    public static void clearRouteContextByDocumentId(String documentId) {
        if(StringUtils.isNotBlank(documentId)) {
            // Pop top stack element with matching document id and replace with empty
            for(int i = 0; i < ROUTE_CONTEXT_STACK.get().size(); i++) {
                if(ROUTE_CONTEXT_STACK.get().get(i).routeHeader.getDocumentId().equals(documentId)) {
                    ROUTE_CONTEXT_STACK.get().remove(i);
                    ROUTE_CONTEXT_STACK.get().add(i, new RouteContext());
                }
            }
        }

    }

	public static void clearCurrentRouteContext() {
        ROUTE_CONTEXT_STACK.get().remove(0);
        ROUTE_CONTEXT_STACK.get().add(0, new RouteContext());
	}

	public static RouteContext createNewRouteContext() {
		ROUTE_CONTEXT_STACK.get().add(0, new RouteContext());
		return getCurrentRouteContext();
	}

	public static RouteContext releaseCurrentRouteContext() {
		return ROUTE_CONTEXT_STACK.get().remove(0);
	}

	/**
	 * @deprecated use getDocument() instead
	 */
	@Deprecated
	public DocumentRouteHeaderValue getRouteHeader() {
		return routeHeader;
	}

	/**
	 * @deprecated user setDocument() instead
	 */
	@Deprecated
	public void setRouteHeader(DocumentRouteHeaderValue routeHeader) {
		this.routeHeader = routeHeader;
	}

	public DocumentRouteHeaderValue getDocument() {
		return routeHeader;
	}

	public void setDocument(DocumentRouteHeaderValue routeHeader) {
		this.routeHeader = routeHeader;
		try {
			setDocumentContent(new StandardDocumentContent(routeHeader.getDocContent(), this));
		} catch (Exception e) {
			throw new WorkflowRuntimeException(e);
		}
	}

	public DocumentContent getDocumentContent() {
		return documentContent;
	}

	public void setDocumentContent(DocumentContent documentContent) {
		this.documentContent = documentContent;
	}

	public RouteNodeInstance getNodeInstance() {
		return nodeInstance;
	}

	public void setNodeInstance(RouteNodeInstance nodeInstance) {
		this.nodeInstance = nodeInstance;
	}

	public EngineState getEngineState() {
		return engineState;
	}

	public void setEngineState(EngineState engineState) {
		this.engineState = engineState;
	}

	public ActionRequestValue getActionRequest() {
		return actionRequest;
	}

	public void setActionRequest(ActionRequestValue actionRequest) {
		this.actionRequest = actionRequest;
	}

	public boolean isSimulation() {
		if (activationContext == null) {
			return false;
		}
		return activationContext.isSimulation();
	}

	public ActivationContext getActivationContext() {
		return activationContext;
	}

	public void setActivationContext(ActivationContext activationContext) {
		this.activationContext = activationContext;
	}

	public boolean isDoNotSendApproveNotificationEmails() {
		return doNotSendApproveNotificationEmails;
	}

	public void setDoNotSendApproveNotificationEmails(boolean sendNotificationEmails) {
		this.doNotSendApproveNotificationEmails = sendNotificationEmails;
	}

	public Map<String, Object> getParameters() {
		return parameters;
	}

	public void setParameters(Map<String, Object> parameters) {
		this.parameters = parameters;
	}

	/**
	 * Determines if search indexing has already been requested during this context
	 * @return the searchIndexingRequestedForContext: true if search indexing has been requested, false otherwise
	 */
	public boolean isSearchIndexingRequestedForContext() {
		return this.searchIndexingRequestedForContext;
	}

	/**
	 * Sets the route context to let it know that search indexing has been requested
	 */
	public void requestSearchIndexingForContext() {
		this.searchIndexingRequestedForContext = true;
	}
	
}
