/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kim.config;

import org.kuali.rice.core.api.config.module.RunMode;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.core.framework.config.module.ModuleConfigurer;
import org.kuali.rice.core.framework.config.module.WebModuleConfiguration;
import org.kuali.rice.kim.api.KimApiConstants;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * This class handles the Spring based KIM configuration that is part of the Rice Configurer that must
 * exist in all Rice based systems and clients.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KIMConfigurer extends ModuleConfigurer {
    public static final String KIM_DATASOURCE_OBJ = "kim.datasource";
    private static final String KIM_UI_SPRING_BEANS_PATH = "classpath:org/kuali/rice/kim/impl/config/KimWebSpringBeans.xml";
    private DataSource dataSource;

    public KIMConfigurer() {
        super(KimApiConstants.Namespaces.MODULE_NAME);
        setValidRunModes(Arrays.asList(RunMode.THIN, RunMode.REMOTE, RunMode.EMBEDDED, RunMode.LOCAL));
    }

    @Override
    protected String getDefaultConfigPackagePath() {
        return "classpath:org/kuali/rice/kim/impl/config/";
    }

    @Override
    public List<String> getPrimarySpringFiles() {
        LOG.info("KIMConfigurer:getPrimarySpringFiles: getRunMode => " + getRunMode());
        List<String> springFileLocations = new ArrayList<String>();
        if (RunMode.THIN == getRunMode()) {
            springFileLocations.add(getDefaultConfigPackagePath() + "KimThinSpringBeans.xml");
        } else if (RunMode.REMOTE == getRunMode()) {
            springFileLocations.add(getDefaultConfigPackagePath() + "KimRemoteSpringBeans.xml");
        } else if (RunMode.EMBEDDED == getRunMode()) {
            springFileLocations.add(getDefaultConfigPackagePath() + "KimEmbeddedSpringBeans.xml");
        } else if (RunMode.LOCAL == getRunMode()) {
            springFileLocations.add(getDefaultConfigPackagePath() + "KimLocalSpringBeans.xml");
        }
        return springFileLocations;
    }

    @Override
    public void addAdditonalToConfig() {
        configureDataSource();
    }

    private void configureDataSource() {
        if (getDataSource() != null) {
            ConfigContext.getCurrentContextConfig().putObject(KIM_DATASOURCE_OBJ, getDataSource());
        }
    }

    public DataSource getDataSource() {
        return dataSource;
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    @Override
    public boolean hasWebInterface() {
        return true;
    }

    @Override
    protected WebModuleConfiguration loadWebModule() {
        WebModuleConfiguration configuration = super.loadWebModule();
        configuration.setWebSpringFiles(Arrays.asList(KIM_UI_SPRING_BEANS_PATH));
        return configuration;
    }
}
