/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.service.impl;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.service.NotificationRecipientService;
import org.kuali.rice.kim.api.KimConstants.KimGroupMemberTypes;
import org.kuali.rice.kim.api.group.Group;
import org.kuali.rice.kim.api.group.GroupService;
import org.kuali.rice.kim.api.identity.principal.Principal;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;

import java.util.List;

/**
 * NotificationRecipientService implementation
 * This implementation relies on KIM user and group management
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class NotificationRecipientServiceKimImpl implements NotificationRecipientService
{
    private static final Logger LOG =
        LogManager.getLogger(NotificationRecipientServiceKimImpl.class);

    protected GroupService getGroupService()
    {
        return KimApiServiceLocator.getGroupService();
    }

    /**
     * Uses the IdentityManagementService of KIM to get the members of a group.
     *
     * @param groupRecipientId the String name of the recipient group
     * @return a String array of all direct (child) principals and descendent principals
     * @see org.kuali.rice.ken.service.NotificationRecipientService#getGroupMembers(java.lang.String)
     */
    @Override
    public String[] getGroupMembers(String groupRecipientId)
    {
        Group group = getGroupService().getGroup(groupRecipientId);

        List<String> ids = getGroupService().getMemberPrincipalIds(group.getId());

        String[] array = new String[ids.size()];
        return ids.toArray(array);
    }

    /**
     * This method retrieves the display name for a user.
     * @param userId
     * @return String
     */
    @Override
    public String getUserDisplayName(String userId)
    {
        //Gary's handling user conversion
        return null;
    }

    /**
     *
     * @see org.kuali.rice.ken.service.NotificationRecipientService#isRecipientValid(java.lang.String, java.lang.String)
     */
     @Override
     public boolean isRecipientValid(String recipientId, String recipientType)
     {
         boolean b = false;

         if( KimGroupMemberTypes.GROUP_MEMBER_TYPE.getCode().equals(recipientType) )
         {
             b = isGroupRecipientValid( recipientId );
         }
         else if( KimGroupMemberTypes.PRINCIPAL_MEMBER_TYPE.getCode().equals(recipientType) )
         {
             b = isUserRecipientValid( recipientId );
         }
         else
         {
             if( LOG.isEnabled(Level.ERROR) )
             {
                 LOG.error("Recipient Type is neither of two acceptable values");
             }
         }
         return b;
     }

    @Override
    public boolean isGroupRecipientValid(String groupRecipientId)
    {
        return (KimApiServiceLocator.getGroupService().getGroup( groupRecipientId ) != null);
    }

    @Override
    public boolean isUserRecipientValid(final String principalName) {
      if (principalName != null && principalName.length() > 0) {
        Principal p = null;
        try {
          p = KimApiServiceLocator.getIdentityService().getPrincipalByPrincipalName(
              principalName);
        } catch (org.kuali.rice.core.api.exception.RiceIllegalArgumentException e) {
          LOG.info("Recipient cannot be found: " + principalName, e);
        }
        if (p != null) {
          return true;
        }
      }
      return false;
    }

}
