/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.impl.action;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.core.api.reflect.DataDefinition;
import org.kuali.rice.kew.actions.ActionTakenEvent;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.api.action.ActionInvocation;
import org.kuali.rice.kew.api.action.ActionInvocationQueue;
import org.kuali.rice.kew.api.action.ActionType;
import org.kuali.rice.kew.api.exception.InvalidActionTakenException;
import org.kuali.rice.kew.api.exception.ResourceUnavailableException;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.kuali.rice.kim.api.identity.principal.Principal;
import org.kuali.rice.ksb.api.messaging.AsyncCapableService;
import org.kuali.rice.ksb.api.messaging.AsyncWrappable;

import javax.xml.namespace.QName;

/**
 * Reference implementation of the ActionInvocationQueue.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ActionInvocationQueueImpl implements ActionInvocationQueue, AsyncWrappable<ActionInvocationQueue> {

    private static final Logger LOG = LogManager.getLogger(ActionInvocationQueueImpl.class);
    private AsyncCapableService asyncCapableService;

    @Override
    public void invokeAction(String principalId, String documentId, ActionInvocation invocation) {
        if (StringUtils.isBlank(principalId)) {
            throw new RiceIllegalArgumentException("principalId is null or blank");
        }

        if (StringUtils.isBlank(documentId)) {
            throw new RiceIllegalArgumentException("documentId is null");
        }

        if (invocation == null) {
            throw new RiceIllegalArgumentException("invocation is null");
        }


        KEWServiceLocator.getRouteHeaderService().lockRouteHeader(documentId);
        DocumentRouteHeaderValue document = KEWServiceLocator.getRouteHeaderService().getRouteHeader(documentId);

        Principal principal = KEWServiceLocator.getIdentityHelperService().getPrincipal(principalId);
        List<DataDefinition> parameters = new ArrayList<DataDefinition>();
        parameters.add(new DataDefinition(document));
        parameters.add(new DataDefinition(principal));
        parameters.add(new DataDefinition(""));

        try {
            final ActionTakenEvent action = KEWServiceLocator.getActionRegistry().createAction(invocation.getAction().getCode(), parameters);
            if (!document.isValidActionToTake(invocation.getAction().getCode())) {
                LOG.warn("Action "
                        + invocation.getAction()
                        + " is not a valid action to take against document "
                        + document.getDocumentId()
                        + " by principal with name '"
                        + principal.getPrincipalName()
                        + "'");
                return;
            } else if (!KEWServiceLocator.getActionRegistry().getValidActions(principal, document).getValidActions()
                    .contains(ActionType.fromCode(action.getActionTakenCode()))) {
                LOG.warn("Action "
                        + action.getActionTakenCode()
                        + " is not valid for document "
                        + document.getDocumentId()
                        + " by principal with name '"
                        + principal.getPrincipalName()
                        + "'");
                return;
            }
            action.performAction();
        } catch (ResourceUnavailableException e) {
            throw new WorkflowRuntimeException(e);
        } catch (InvalidActionTakenException e) {
            throw new WorkflowRuntimeException(e);
        }

    }

    @Override
    public ActionInvocationQueue wrap(QName qname, String applicationId) {
        return new ActionInvocationQueueAsyncCapableImpl(getAsyncCapableService(), this, qname, applicationId);
    }

    public AsyncCapableService getAsyncCapableService() {
        return asyncCapableService;
    }

    public void setAsyncCapableService(AsyncCapableService asyncCapableService) {
        this.asyncCapableService = asyncCapableService;
    }
}
