/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.impl.actionlist;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.kew.api.action.ActionItem;
import org.kuali.rice.kew.api.actionlist.ActionListService;
import org.kuali.rice.kew.service.KEWServiceLocator;

/**
 * Reference implementation of the {@link org.kuali.rice.kew.api.actionlist.ActionListService} api.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class ActionListServiceNewImpl implements ActionListService {

    @Override
    public Integer getUserActionItemCount(String principalId) {
        incomingParamCheck(principalId, "principalId");
        return Integer.valueOf(KEWServiceLocator.getActionListService().getCount(principalId));
    }

    @Override
    public List<ActionItem> getActionItemsForPrincipal(String principalId) {
        incomingParamCheck(principalId, "principalId");
        Collection<org.kuali.rice.kew.actionitem.ActionItem> actionItems
                = KEWServiceLocator.getActionListService().getActionList(principalId, null);
        List<ActionItem> actionItemVOs = new ArrayList<ActionItem>(actionItems.size());
        for (org.kuali.rice.kew.actionitem.ActionItem actionItem : actionItems) {
            actionItemVOs.add(org.kuali.rice.kew.actionitem.ActionItem.to(actionItem));
        }
        return actionItemVOs;
    }

    @Override
    public List<ActionItem> getAllActionItems(String documentId) {
        incomingParamCheck(documentId, "documentId");
        Collection<org.kuali.rice.kew.actionitem.ActionItem> actionItems
                = KEWServiceLocator.getActionListService().getActionListForSingleDocument(documentId);
        List<ActionItem> actionItemVOs = new ArrayList<ActionItem>(actionItems.size());
        for (org.kuali.rice.kew.actionitem.ActionItem actionItem : actionItems) {
            actionItemVOs.add(org.kuali.rice.kew.actionitem.ActionItem.to(actionItem));
        }
        return actionItemVOs;
    }

    @Override
    public List<ActionItem> getActionItems(String documentId, List<String> actionRequestedCodes) {
        incomingParamCheck(documentId, "documentId");
        List<ActionItem> actionItems = getAllActionItems(documentId);
        List<ActionItem> matchingActionitems = new ArrayList<ActionItem>();
        for (ActionItem actionItemVO : actionItems) {
            if (actionRequestedCodes.contains(actionItemVO.getActionRequestCd())) {
                matchingActionitems.add(actionItemVO);
            }
        }
        return matchingActionitems;
    }

    private void incomingParamCheck(Object object, String name) {
        if (object == null) {
            throw new RiceIllegalArgumentException(name + " was null");
        } else if (object instanceof String
                && StringUtils.isBlank((String) object)) {
            throw new RiceIllegalArgumentException(name + " was blank");
        }
    }
}
