/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.impl.document;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.kew.actionrequest.ActionRequestValue;
import org.kuali.rice.kew.actionrequest.service.ActionRequestService;
import org.kuali.rice.kew.actionrequest.service.impl.NotificationSuppression;
import org.kuali.rice.kew.api.document.DocumentRefreshQueue;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.engine.OrchestrationConfig;
import org.kuali.rice.kew.engine.OrchestrationConfig.EngineCapability;
import org.kuali.rice.kew.engine.RouteHelper;
import org.kuali.rice.kew.engine.node.RouteNodeInstance;
import org.kuali.rice.kew.engine.node.service.RouteNodeService;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.kuali.rice.kew.util.PerformanceLogger;
import org.kuali.rice.ksb.api.messaging.AsyncCapableService;
import org.kuali.rice.ksb.api.messaging.AsyncWrappable;

import javax.xml.namespace.QName;


/**
 * A service which effectively "refreshes" and requeues a document.  It first deletes any
 * pending action requests on the documents and then requeues the document for standard routing.
 * Addionally, it adds duplicate notification suppression state to RouteNodeInstanceS for 
 * which ActionRequestS will be regenerated. 
 * 
 * <p>Intended to be called async and wired that way in server/client spring beans.</p>
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class DocumentRefreshQueueImpl implements DocumentRefreshQueue, AsyncWrappable<DocumentRefreshQueue> {

    private AsyncCapableService asyncCapableService;
	private RouteHelper helper = new RouteHelper();
    
	/**
	 * Requeues a document, and sets notification suppression data
	 * 
	 * @see org.kuali.rice.kew.api.document.DocumentRefreshQueue#refreshDocument(java.lang.String)
	 */
	@Override
	public void refreshDocument(String documentId) {
		if (StringUtils.isBlank(documentId)) {
            throw new RiceIllegalArgumentException("documentId is null or blank");
        }

        PerformanceLogger performanceLogger = new PerformanceLogger();
        KEWServiceLocator.getRouteHeaderService().lockRouteHeader(documentId);
        Collection<RouteNodeInstance> activeNodes = getRouteNodeService().getActiveNodeInstances(documentId);
        List<ActionRequestValue> requestsToDelete = new ArrayList<ActionRequestValue>();
        
		NotificationSuppression notificationSuppression = new NotificationSuppression();
		
        for (RouteNodeInstance nodeInstance : activeNodes) {
            // only "requeue" if we're dealing with a request activation node
            if (helper.isRequestActivationNode(nodeInstance.getRouteNode())) {
            	List<ActionRequestValue> deletesForThisNode = 
            		getActionRequestService().findPendingRootRequestsByDocIdAtRouteNode(documentId, nodeInstance.getRouteNodeInstanceId());

            	for (ActionRequestValue deleteForThisNode : deletesForThisNode) {
                    // check either the request or its first present child request to see if it is system generated
                    boolean containsRoleOrRuleRequests = deleteForThisNode.isRouteModuleRequest();
                    if (!containsRoleOrRuleRequests) {
                        if (CollectionUtils.isNotEmpty(deleteForThisNode.getChildrenRequests())) {
                            containsRoleOrRuleRequests = deleteForThisNode.getChildrenRequests().get(0).isRouteModuleRequest();
                        }
                    }

                    if (containsRoleOrRuleRequests) {
                        // remove all route or rule system generated requests
                        requestsToDelete.add(deleteForThisNode);

                        // suppress duplicate notifications
            			notificationSuppression.addNotificationSuppression(nodeInstance, deleteForThisNode);
            		}
            	}

                // this will trigger a regeneration of requests
                nodeInstance.setInitial(true);
                getRouteNodeService().save(nodeInstance);
            }
        }
        for (ActionRequestValue requestToDelete : requestsToDelete) {
            getActionRequestService().deleteActionRequestGraphNoOutbox(requestToDelete);
        }
        try {
            OrchestrationConfig config = new OrchestrationConfig(EngineCapability.STANDARD);
        	KEWServiceLocator.getWorkflowEngineFactory().newEngine(config).process(documentId);
        } catch (Exception e) {
        	throw new WorkflowRuntimeException(e);
        }
        performanceLogger.log("Time to run DocumentRequeuer for document " + documentId);	
	}

    private ActionRequestService getActionRequestService() {
        return KEWServiceLocator.getActionRequestService();
    }
    
    private RouteNodeService getRouteNodeService() {
        return KEWServiceLocator.getRouteNodeService();
    }

    @Override
    public DocumentRefreshQueue wrap(QName qname, String applicationId) {
        return new DocumentRefreshQueueAsyncCapableImpl(asyncCapableService,this, qname, applicationId);
    }

    public AsyncCapableService getAsyncCapableService() {
        return asyncCapableService;
    }

    public void setAsyncCapableService(AsyncCapableService asyncCapableService) {
        this.asyncCapableService = asyncCapableService;
    }
}
