/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.notification.service.impl;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.ComparatorUtils;
import org.kuali.rice.core.api.config.module.RunMode;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.kew.actionitem.ActionItemComparator;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.api.action.ActionItem;
import org.kuali.rice.kew.api.mail.ImmediateEmailReminderQueue;
import org.kuali.rice.kew.doctype.bo.DocumentType;
import org.kuali.rice.kew.engine.RouteContext;
import org.kuali.rice.kew.notification.service.NotificationService;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.kuali.rice.kew.api.KewApiConstants;


/**
 * The default implementation of the NotificationService.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class DefaultNotificationService implements NotificationService {

	protected final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(getClass());
	private static final String KEW_RUN_MODE_PROPERTY = "kew.mode";
	private static final Comparator notificationPriorityComparator = ComparatorUtils.reversedComparator(new ActionItemComparator());

	/**
	 * Queues up immediate email processors for ActionItem notification.  Prioritizes the list of
	 * Action Items passed in and attempts to not send out multiple emails to the same user.
	 */
	@Override
    public void notify(List<ActionItem> actionItems) {
		// sort the list of action items using the same comparator as the Action List
		actionItems.sort(notificationPriorityComparator);
		final Set<String> sentNotifications = new HashSet<>();

		for (ActionItem actionItem : actionItems) {
			if (!sentNotifications.contains(actionItem.getPrincipalId()) && shouldNotify(actionItem)) {
				sentNotifications.add(actionItem.getPrincipalId());
				sendNotification(actionItem);
			}
		}
	}

	/**
	 * Sends a notification
	 * @param actionItem the action item
	 */
	protected void sendNotification(ActionItem actionItem) {
		boolean doNotSend = RouteContext.getCurrentRouteContext().isDoNotSendApproveNotificationEmails();
		if (RunMode.valueOf(ConfigContext.getCurrentContextConfig().getProperty(KEW_RUN_MODE_PROPERTY)) == RunMode.LOCAL) {
			KEWServiceLocator.getActionListEmailService().sendImmediateReminder(actionItem, doNotSend);
		} else {
			final ImmediateEmailReminderQueue immediateEmailQueue = KewApiServiceLocator.getImmediateEmailReminderQueue();
			immediateEmailQueue.sendReminder(actionItem, doNotSend);
		}
	}

	protected boolean shouldNotify(ActionItem actionItem) {
		try {
            boolean sendEmail = true;
            // Removed preferences items since they will be checked before it sends
            // the email in the action list email service

			// don't send notification if this action item came from a SAVE action and the NOTIFY_ON_SAVE policy is not set
			if (isItemOriginatingFromSave(actionItem) && !shouldNotifyOnSave(actionItem)) {
				sendEmail = false;
			}
			return sendEmail;
		} catch (Exception e) {
			throw new WorkflowRuntimeException("Error loading user with workflow id " + actionItem.getPrincipalId() + " for notification.", e);
		}
	}

	/**
	 * Returns true if the ActionItem doesn't represent a request generated from a "SAVE" action or, if it does,
	 * returns true if the document type policy
	 */
	protected boolean isItemOriginatingFromSave(ActionItem actionItem) {
		return actionItem.getResponsibilityId() != null && actionItem.getResponsibilityId().equals(KewApiConstants.SAVED_REQUEST_RESPONSIBILITY_ID);
	}

	protected boolean shouldNotifyOnSave(ActionItem actionItem) {
		DocumentRouteHeaderValue document = KEWServiceLocator.getRouteHeaderService().getRouteHeader(actionItem.getDocumentId());
		DocumentType documentType = KEWServiceLocator.getDocumentTypeService().findById(document.getDocumentTypeId());
		return documentType.getNotifyOnSavePolicy().getPolicyValue();
	}

    @Override
    public void removeNotification(List<ActionItem> actionItems) {
        // nothing
    }
}
