/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.core.framework.persistence.platform;

import org.apache.ojb.broker.PersistenceBroker;

import java.sql.Connection;

public abstract class ANSISqlDatabasePlatform implements DatabasePlatform
{
    /**
     * @see DatabasePlatform#getTruncateTableSql(String)
     */
    @Override
    public String getTruncateTableSql(String tableName)
    {
        return "truncate table " + tableName;
    }
    
    /**
     * @see DatabasePlatform#getCreateTableFromTableSql(String, String)
     */
    @Override
    public String getCreateTableFromTableSql(String tableToCreate, String fromTable)
     {
        return new StringBuffer("create table ").append(tableToCreate)
                .append(" as select * from ").append(fromTable).toString();
    }
    
    /**
     * @see DatabasePlatform#getInsertDataFromTableSql(String, String)
     */
    @Override
    public String getInsertDataFromTableSql(String restoreTableName, String fromTableName)
    {
        return new StringBuffer("insert into ").append(restoreTableName)
                .append(" select * from ").append(fromTableName).toString();
    }
    
    /**
     * @see DatabasePlatform#getDropTableSql(String)
     */
    @Override
    public String getDropTableSql(String tableName) {
    	return new StringBuffer("drop table ").append(tableName).toString();
    }
    
    /**
     * Returns an expression equivalent to oracle's NVL statement using the CASE and IS NULL expressions, which should
     * be supported by many database systems
     * 
     * @see DatabasePlatform#getIsNullFunction(java.lang.String, java.lang.String)
     */
    @Override
    public String getIsNullFunction(String exprToTest, String exprToReplaceIfTestExprNull) {
    	return new StringBuilder(" case when ").append(exprToTest).append(" is null then ").append(exprToReplaceIfTestExprNull)
        .append(" else ").append(exprToTest).append(" end ").toString();
    }
    
    @Override
    public String getDateSQL(String date, String time) {
        // SQL 92 date literal syntax:
        // http://www.stanford.du/dept/itss/docs/oracle/9i/java.920/a96654/ref.htm#1005145
        String d = date.replace('/', '-');
        if (time == null) {
            return new StringBuilder("{d '").append(d).append("'}").toString();    
        } else {
            return new StringBuilder("{ts '").append(d).append(" ").append(time).append("'}").toString(); 
        }
    }
    
    /**
     * @see DatabasePlatform#getUpperCaseFunction()
     * @return the String "UPPER"
     */
    //chb: this was copied over from the legacy code, but is it really necessary?
    @Override
    public String getUpperCaseFunction() {
    	return "UPPER";
    }

    @Override
    public String toString() {
        return "[ANSISqlDatabasePlatform]";
    }

    @Override
    public Long getNextValSQL(String sequenceName, Object nextValSource) {
        if (nextValSource instanceof Connection) {
            return getNextValSqlJdbc(sequenceName, (Connection)nextValSource);
        } else if (nextValSource instanceof PersistenceBroker) {
            return getNextValSqlOjb(sequenceName, (PersistenceBroker)nextValSource);
        }
        throw new IllegalArgumentException("No next value strategy found for given nextValSource: " + nextValSource);
    }

    @Deprecated
    protected abstract Long getNextValSqlOjb(String sequenceName, PersistenceBroker persistenceBroker);

    @Deprecated
    protected abstract Long getNextValSqlJdbc(String sequenceName, Connection connection);

}
