/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa;

import org.kuali.rice.core.api.criteria.CountFlag;
import org.kuali.rice.core.api.criteria.GenericQueryResults;
import org.kuali.rice.core.api.criteria.QueryByCriteria;

import javax.persistence.Query;
import java.util.List;
import java.util.Objects;

/**
 * Base class for QueryByCriteria lookups and deletes for JPA PersistenceProvider implementations.
 *
 * <p>
 * Implements the core api CriteriaLookupService, as that is the exact interface required, however this class is not
 * intended to be defined as a "service", it is merely a helper.
 * </p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
abstract class DataObjectCriteriaQueryBase<C, Q, QC> implements CriteriaQuery {

    /**
     * Gets the QueryTranslator to translate from the API to implementation-specific classes.
     *
     * @return the QueryTranslator to translate from the API to implementation-specific classes.
     */
    protected abstract QueryTranslator<C, Q, QC> getQueryTranslator();

    /**
     * Gets the row count for the given query.
     *
     * @param query the query to count the rows on.
     * @return the row count for the given query.
     */
    protected abstract int getRowCount(QC query);

    /**
     * Gets the row count to include along with the results of the query.
     *
     * @param query the query to count the rows on.
     *
     * @return The row count to include along with the results of the query.
     */
    protected abstract int getIncludedRowCount(QC query);

    /**
     * Gets the results from the given query.
     *
     * @param query the query to use to get the results.
     * @param <T> the type of results to return.
     * @return a list of results from the given query.
     */
    protected abstract <T> List<T> getResults(Q query);

    /**
     * Executes the given query.
     *
     * @param query the query to execute.
     * @return the number of records successfully committed.
     */
    protected abstract int executeUpdate(Query query);

    /**
     * {@inheritDoc}
     */
    @Override
    public <T> void deleteMatching(Class<T> type, QueryByCriteria criteria) {

        if (type == null) {
            throw new IllegalArgumentException("class type is null");
        }

        // do not allow delete * on an entire table, by default
        if (criteria == null || criteria.getPredicate() == null) {
            throw new IllegalArgumentException("criteria is null");
        }

        final C parent = getQueryTranslator().translateCriteria(type, criteria);
        final Query query = getQueryTranslator().createDeletionQuery(type, parent);
        executeUpdate(query);
    }

    @Override
    public <T> void deleteAll(Class<T> type) {
        if (type == null) {
            throw new IllegalArgumentException("class type is null");
        }

        final C parent = getQueryTranslator().translateCriteria(type,
                QueryByCriteria.Builder.create().build());
        final Query query = getQueryTranslator().createDeletionQuery(type, parent);
        executeUpdate(query);
    }

    @Override
    public <T> GenericQueryResults<T> lookup(Class<T> queryClass, QueryByCriteria criteria) {
        if (queryClass == null) {
            throw new IllegalArgumentException("queryClass is null");
        }

        if (criteria == null) {
            throw new IllegalArgumentException("criteria is null");
        }

        final C resultsCriteria = getQueryTranslator().translateCriteria(queryClass, criteria);
        final C countCriteria = getQueryTranslator().translateCriteriaForCount(queryClass, criteria);

        switch (Objects.requireNonNull(criteria.getCountFlag())) {
            case ONLY:
                return forCountOnly(queryClass, criteria, countCriteria);
            case NONE:
                return forRowResults(queryClass, criteria, resultsCriteria);
            case INCLUDE:
                return forRowResultsWithCount(queryClass, criteria, resultsCriteria, countCriteria);
            default: throw new UnsupportedCountFlagException(criteria.getCountFlag());
        }
    }

    /**
     * Gets results where the actual rows are requested.
     *
     * @param queryClass the type of the results to return.
     * @param criteria the criteria to use to get the results.
     * @param resultsCriteria the implementation-specific criteria.
     *
     * @return results where the actual rows are requested.
     */
    protected <T> GenericQueryResults<T> forRowResults(final Class<T> queryClass, final QueryByCriteria criteria, final C resultsCriteria) {
        final GenericQueryResults.Builder<T> results = GenericQueryResults.Builder.create();
        addResults(queryClass, criteria, resultsCriteria, results);
        return results.build();
    }

    protected <T> GenericQueryResults<T> forRowResultsWithCount(final Class<T> queryClass, final QueryByCriteria criteria, final C resultsCriteria, final C countCriteria) {
        final GenericQueryResults.Builder<T> results = GenericQueryResults.Builder.create();
        final QC queryForCount = getQueryTranslator().createQueryForCount(queryClass, countCriteria);
        results.setTotalRowCount(getIncludedRowCount(queryForCount));

        addResults(queryClass, criteria, resultsCriteria, results);
        return results.build();
    }

    private <T> void addResults(Class<T> queryClass, QueryByCriteria criteria, C ojbCriteria, GenericQueryResults.Builder<T> results) {
        final Q query = getQueryTranslator().createQuery(queryClass, ojbCriteria);
        getQueryTranslator().convertQueryFlags(criteria, query);

        final List<T> rows = getResults(query);

        if (criteria.getMaxResults() != null && rows.size() > criteria.getMaxResults()) {
            results.setMoreResultsAvailable(true);
            //remove the extra row that was returned
            rows.remove(criteria.getMaxResults().intValue());
        }

        results.setResults(rows);
    }

    /**
     * Gets results where only the count is requested.
     *
     * @param queryClass the type of the results to return.
     * @param criteria the criteria to use to get the results.
     * @param platformCriteria the implementation-specific criteria.
     *
     * @return results where only the count is requested.
     */
    protected <T> GenericQueryResults<T> forCountOnly(final Class<T> queryClass, final QueryByCriteria criteria, final C platformCriteria) {
        final QC queryForCount = getQueryTranslator().createQueryForCount(queryClass, platformCriteria);
        final GenericQueryResults.Builder<T> results = GenericQueryResults.Builder.<T>create();
        results.setTotalRowCount(getRowCount(queryForCount));
        return results.build();
    }

    /**
     * An error to throw when the CountFlag is not recognized.
     *
     * <p>This is a fatal error since this implementation should support all known count flags.</p>
     */
    protected static class UnsupportedCountFlagException extends RuntimeException {

        /**
         * Creates an exception for if the CountFlag is not recognized.
         * @param flag the flag in error.
         */
        protected UnsupportedCountFlagException(CountFlag flag) {
            super("Unsupported predicate [" + flag + "]");
        }
    }
}
