/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa.eclipselink;

import java.util.Map;

import org.eclipse.persistence.config.PersistenceUnitProperties;
import org.kuali.rice.krad.data.jpa.KradEntityManagerFactoryBean;

/**
 * A KRAD-managed {@link javax.persistence.EntityManagerFactory} factory bean which can be used to configure an
 * EclipseLink persistence unit using JPA.
 *
 * <p>This class inherits the behavior from {@link KradEntityManagerFactoryBean} but adds the following:</p>
 *
 * <ul>
 *     <li>Detects if JTA is being used and, if so sets a JPA property value for
 *         {@link PersistenceUnitProperties#TARGET_SERVER} to {@link JtaTransactionController} which allows for
 *         EclipseLink integration with JTA.</li>
 *     <li>Configures an EclipseLink "customizer" which allows for a configurable sequence management strategy</li>
 *     <li>Disables the shared cache (defined by {@link PersistenceUnitProperties#CACHE_SHARED_DEFAULT} by default</li>
 * </ul>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KradEclipseLinkEntityManagerFactoryBean extends KradEntityManagerFactoryBean {

    /**
     * Creates a KRAD-managed {@link javax.persistence.EntityManagerFactory} factory bean.
     */
    public KradEclipseLinkEntityManagerFactoryBean() throws Exception {
        super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void loadCustomJpaDefaults(Map<String, String> jpaProperties) {
		if (getPersistenceUnitManager().getDefaultJtaDataSource() != null
				&& !jpaProperties.containsKey(PersistenceUnitProperties.TARGET_SERVER)) {
            jpaProperties.put(PersistenceUnitProperties.TARGET_SERVER, JtaTransactionController.class.getName());
        }

		if (!jpaProperties.containsKey(PersistenceUnitProperties.SESSION_CUSTOMIZER)) {
			jpaProperties.put(PersistenceUnitProperties.SESSION_CUSTOMIZER, KradEclipseLinkCustomizer.class.getName());
		}

		if (!jpaProperties.containsKey(PersistenceUnitProperties.CACHE_SHARED_DEFAULT)) {
			jpaProperties.put(PersistenceUnitProperties.CACHE_SHARED_DEFAULT, "false");
		}
    }

}
