/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data;

import org.kuali.rice.core.api.resourceloader.GlobalResourceLoader;
import org.kuali.rice.core.framework.resourceloader.BeanFactoryResourceLoader;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.data.KradDataServiceLocator;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import javax.xml.namespace.QName;

/**
 * A factory bean which load and/or acquires a reference to the data framework.
 *
 * <p>Will lazy-initialize the framework if it's not already been initialized. The factory bean will return a reference
 * the {@link DataObjectService} which is the main API entry point into the data framework.</p>
 *
 * @see DataObjectService
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KradDataFactoryBean implements FactoryBean<DataObjectService> {

    private static final String SPRING_FILE = "classpath:org/kuali/rice/krad/data/config/KRADDataSpringBeans.xml";

    @Override
    public DataObjectService getObject() throws Exception {
        // first, let's see if it's already been initialized
        DataObjectService dataObjectService = KradDataServiceLocator.getDataObjectService();
        if (dataObjectService == null) {
            ClassPathXmlApplicationContext context = new ClassPathXmlApplicationContext(SPRING_FILE);
            BeanFactoryResourceLoader rl = new BeanFactoryResourceLoader(new QName("krad-data"), context);
            rl.start();
            GlobalResourceLoader.addResourceLoader(rl);
            dataObjectService = KradDataServiceLocator.getDataObjectService();
        }
        if (dataObjectService == null) {
            throw new IllegalStateException("Failed to locate or initialize krad data framework.");
        }
        return dataObjectService;
    }

    @Override
    public Class<?> getObjectType() {
        return DataObjectService.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

}
