/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa.converters;

import javax.persistence.AttributeConverter;
import javax.persistence.Converter;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.CoreApiServiceLocator;

/**
 * Calls the core service to encrypt values going to the database and decrypt values coming back from the database.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@Converter
public class EncryptionConverter implements AttributeConverter<String, String> {

    /**
     * {@inheritDoc}
     *
     * This implementation encrypts the value going to the database.
     */
	@Override
	public String convertToDatabaseColumn(String objectValue) {
		// don't attempt to encrypt nulls or empty strings
		if (objectValue == null) {
			return null;
		}
		if (StringUtils.isEmpty(objectValue.toString())) {
			return "";
		}
		try {
			// check if the encryption service is enable before using it
			if (CoreApiServiceLocator.getEncryptionService().isEnabled()) {
				return CoreApiServiceLocator.getEncryptionService().encrypt(objectValue);
			}
		} catch (Exception e) {
			throw new RuntimeException("Exception while attempting to encrypt value for DB: ", e);
		}
		return objectValue;
	}

    /**
     * {@inheritDoc}
     *
     * This implementation decrypts the value coming from the database.
     */
	@Override
	public String convertToEntityAttribute(String dataValue) {
		// don't attempt to decrypt nulls or empty strings
		if (dataValue == null) {
			return null;
		}
		if (StringUtils.isEmpty(dataValue.toString())) {
			return "";
		}
		try {
			// check if the encryption service is enable before using it
			if (CoreApiServiceLocator.getEncryptionService().isEnabled()) {
				return CoreApiServiceLocator.getEncryptionService().decrypt(dataValue.toString());
			}
		} catch (Exception e) {
			throw new RuntimeException("Exception while attempting to decrypt value from DB: ", e);
		}
		return dataValue;
	}

}
