/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.service.impl;

import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.DocumentSerializerService;
import org.kuali.rice.krad.service.XmlObjectSerializerService;
import org.kuali.rice.krad.util.documentserializer.AlwaysTruePropertySerializibilityEvaluator;
import org.kuali.rice.krad.util.documentserializer.PropertySerializabilityEvaluator;

/**
 * Default implementation of the {@link DocumentSerializerService}.  If no &lt;workflowProperties&gt; have been defined in the
 * data dictionary for a document type (i.e. {@link Document#getDocumentPropertySerizabilityEvaluator()} returns an instance of
 * {@link AlwaysTruePropertySerializibilityEvaluator}), then this service will revert to using the {@link XmlObjectSerializerService}
 * bean, which was the old way of serializing a document for routing.  If workflowProperties are defined, then this implementation
 * will selectively serialize items.
 */
public class DocumentSerializerServiceImpl extends SerializerServiceBase implements DocumentSerializerService {

    /**
     * Serializes a document for routing
     *
     * @see org.kuali.rice.krad.service.DocumentSerializerService#serializeDocumentToXmlForRouting(org.kuali.rice.krad.document.Document)
     */
    @Override
    public String serializeDocumentToXmlForRouting(Document document) {
        final PropertySerializabilityEvaluator evaluator = document.getDocumentPropertySerizabilityEvaluator();
        return doSerialization(evaluator, document, new Serializer<Document>() {
            @Override
            public String serialize(Document document) {
                Object xmlWrapper = wrapDocumentWithMetadata(document);
                String xml;
                if (evaluator instanceof AlwaysTruePropertySerializibilityEvaluator) {
                    xml = getXmlObjectSerializerService().toXml(xmlWrapper);
                }
                else {
                    xml = xstream.toXML(xmlWrapper);
                }
                return xml;
            }
        });
    }

    /**
     * Wraps the document before it is routed.  This implementation defers to {@link Document#wrapDocumentWithMetadataForXmlSerialization()}.
     *
     * @param document
     * @return may return the document, or may return another object that wraps around the document to provide additional metadata
     */
    protected Object wrapDocumentWithMetadata(Document document) {
        return document.wrapDocumentWithMetadataForXmlSerialization();
    }

    @Override
    protected PropertySerializabilityEvaluator getPropertySerizabilityEvaluator(Object dataObject) {
        return null;
    }
}
