/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.bind;

import java.beans.PropertyEditorSupport;
import java.io.Serializable;

/**
 * PropertyEditor for booleans supports y/n which the spring version does not
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class UifBooleanEditor extends PropertyEditorSupport implements Serializable {
	private static final long serialVersionUID = -6333792216543862346L;

	private static final String TRUE_VALUES = "/true/yes/y/on/1/";
	private static final String FALSE_VALUES = "/false/no/n/off/0/";

	private static final String TRUE_VALUE = "true";
	private static final String FALSE_VALUE = "false";

    /**
     * Gets the property value as a string suitable for presentation
     * to a human to edit
     *
     * @return The property value as a string suitable for presentation
     *       to a human to edit.
     * <p>   Returns String "true" or "false".
     * <p>   Returns "null" is the value can't be expressed as a string.
     * <p>   If a non-null value is returned, then the PropertyEditor should
     *	     be prepared to parse that string back in setAsText().
     */
    @Override
	public String getAsText() {
		if(this.getValue() == null) {
			return "";
		}
		else if(((Boolean)this.getValue()).booleanValue()) {
			return TRUE_VALUE;
		}
		else {
			return FALSE_VALUE;
		}
	}

    /**
     * Sets the property value by parsing a given String
     *
     * <p>
     *     The text is compared against the configured acceptable string values for
     *     boolean true and false
     * </p>
     *
     * @param text  The string to be parsed.
     * @throws IllegalArgumentException if text does not contain either true or false
     */
	@Override
	public void setAsText(String text) throws IllegalArgumentException {
		String input = null;

		if(text != null) {
			StringBuilder builder = new StringBuilder();
			builder.append("/").append(text.toLowerCase()).append("/");
			input = builder.toString();

			if(TRUE_VALUES.contains(input)) {
				this.setValue(Boolean.TRUE);
			}
			else if(FALSE_VALUES.contains(input)) {
				this.setValue(Boolean.FALSE);
			}
			else {
				input = null;
			}
		}

		if(input == null) {
			throw new IllegalArgumentException("Invalid boolean input: " + text);
		}
	}

}
