/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.listener;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.krad.UserSession;
import org.kuali.rice.krad.document.authorization.PessimisticLock;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.util.GlobalVariables;
import org.kuali.rice.krad.util.KRADConstants;

import javax.servlet.http.HttpSessionEvent;
import javax.servlet.http.HttpSessionListener;
import java.util.List;

/**
 * Used to handle session timeouts where {@link PessimisticLock} objects should
 * be removed from a document
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KualiHttpSessionListener implements HttpSessionListener {

    /**
     * HttpSession hook for additional setup method when sessions are created
     *
     * @param se - the HttpSessionEvent containing the session
     * @see javax.servlet.http.HttpSessionListener#sessionCreated(javax.servlet.http.HttpSessionEvent)
     */
    @Override
    public void sessionCreated(HttpSessionEvent se) {
        // no operation required at this time
    }

    /**
     * HttpSession hook for additional cleanup when sessions are destroyed
     *
     * @param se - the HttpSessionEvent containing the session
     * @see javax.servlet.http.HttpSessionListener#sessionDestroyed(javax.servlet.http.HttpSessionEvent)
     */
    @Override
    public void sessionDestroyed(HttpSessionEvent se) {
        releaseLocks(se);
    }

    /**
     * Remove any locks that the user has for this session
     */
    private void releaseLocks(HttpSessionEvent se) {
        if (se != null && se.getSession() != null && se.getSession().getAttribute(KRADConstants.USER_SESSION_KEY) != null) {
            UserSession userSession = (UserSession) se.getSession().getAttribute(KRADConstants.USER_SESSION_KEY);
            releaseUserSessionLocks(userSession);
        }

        if (GlobalVariables.getUserSession() != null) {
            releaseUserSessionLocks(GlobalVariables.getUserSession());
        }
    }

    private void releaseUserSessionLocks(UserSession userSession) {
        String sessionId = userSession.getKualiSessionId();
        Person user = userSession.getPerson();
        if (StringUtils.isNotBlank(sessionId) && user != null) {
            List<PessimisticLock> locks =
                    KRADServiceLocatorWeb.getPessimisticLockService().getPessimisticLocksForSession(sessionId);

            KRADServiceLocatorWeb.getPessimisticLockService().releaseAllLocksForUser(locks, user);
        }
    }
}
