/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.maintenance;

import org.kuali.rice.krad.document.Document;

/**
 * Common interface for all maintenance documents.
 */
public interface MaintenanceDocument extends Document {

    /**
     * Get the XML representation of the maintenance document
     *
     * @return String containing the xml representation of the maintenance document
     */
    String getXmlDocumentContents();

    /**
     * Get the new maintainable object
     *
     * @return Maintainable which holds the new maintenance record
     */
    Maintainable getNewMaintainableObject();

    /**
     * Get the old maintainable object
     *
     * @return Maintainable which holds the old maintenance record
     */
    Maintainable getOldMaintainableObject();

    /**
     * Sets the xml contents of the maintenance document
     *
     * @param documentContents String xml
     */
    void setXmlDocumentContents(String documentContents);

    /**
     * Set the new maintainable object
     *
     * @param newMaintainableObject maintainable with the new maintenance record
     */
    void setNewMaintainableObject(Maintainable newMaintainableObject);

    /**
     * Set the old maintainable object
     *
     * @param oldMaintainableObject maintainable with the old maintenance record
     */
    void setOldMaintainableObject(Maintainable oldMaintainableObject);

    /**
     * Return the data object that this MaintenanceDocument is maintaining
     *
     * @return document data object instance
     */
    Object getDocumentDataObject();

    /**
     * Build the xml document string from the contents of the old and new maintainables.
     */
    void populateXmlDocumentContentsFromMaintainables();

    /**
     * Populates the old and new maintainables from the xml document contents string.
     */
    void populateMaintainablesFromXmlDocumentContents();

    /**
     * Check if maintenance document has old maintenance data

     * @return true if this maintenance document has old data, false otherwise
     */
    boolean isOldDataObjectInDocument();

    /**
     * Check if maintenance document is creating a new Business Object
     *
     * @return true if this maintenance document is creating a new Business Object, false otherwise
     */
    boolean isNew();

    /**
     * Check if maintenance document is editing an existing Business Object
     *
     * @return true if this maintenance document is editing an existing Business Object, false otherwise
     */
    boolean isEdit();

    /**
     * Check if maintenance document is creating a new Business Object out of an existing Business Object
     *
     * <p>
     * For example, a new division vendor out of an existing parent vendor.
     * </p>
     *
     * @return true if maintenance document is creating a new Business Object out of an existing Business object, false otherwise
     */
    boolean isNewWithExisting();

    /**
     * Check if fields are cleared on copy
     *
     * <p>
     * For copy action the primary keys need to be cleared.  This flag indicates if the clearing has occurred.
     * </p>
     *
     * @return true if the primary keys have been cleared already, false otherwise
     */
    boolean isFieldsClearedOnCopy();

    /**
     * Set the keys cleared on copy flag
     *
     * @param keysClearedOnCopy
     *
     */
    void setFieldsClearedOnCopy(boolean keysClearedOnCopy);

    /**
     * Check if the topic field should be displayed in the notes section
     *
     * @return true if the topic field should be displayed in the notes section, false otherwise
     */
    boolean isDisplayTopicFieldInNotes();

    /**
     * Set the display topic field in notes flag
     *
     * @parm displayTopicFieldInNotes
     */
    void setDisplayTopicFieldInNotes(boolean displayTopicFieldInNotes);

}
