/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.threadpool;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.ksb.util.KSBConstants;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

public class KSBScheduledThreadPoolExecutor extends ScheduledThreadPoolExecutor implements KSBScheduledPool {

	private static final Logger LOG = LogManager.getLogger(KSBScheduledThreadPoolExecutor.class);

	private boolean started;
	private static final int DEFAULT_SIZE = 2;

	public KSBScheduledThreadPoolExecutor() {
		super(DEFAULT_SIZE, new KSBThreadFactory());
	}

	@Override
    public boolean isStarted() {
		return started;
	}

	@Override
    public void start() throws Exception {
		LOG.info("Starting the KSB scheduled thread pool...");
		try {
			Integer size = new Integer(ConfigContext.getCurrentContextConfig().getProperty(KSBConstants.Config.FIXED_POOL_SIZE));
			this.setCorePoolSize(size);
		} catch (NumberFormatException nfe) {
			// ignore this, instead the pool will be set to DEFAULT_SIZE
		}
		LOG.info("...KSB scheduled thread pool successfully started.");
	}

	@Override
    public void stop() throws Exception {
		LOG.info("Stopping the KSB scheduled thread pool...");
		try {
            int pendingTasks = this.shutdownNow().size();
            LOG.info(pendingTasks + " pending tasks...");
			LOG.info("awaiting termination: " + this.awaitTermination(20, TimeUnit.SECONDS));
			LOG.info("...KSB scheduled thread pool successfully stopped, isShutdown=" + this.isShutdown() + ", isTerminated=" + this.isTerminated());
		} catch (Exception e) {
			LOG.warn("Exception thrown shutting down " + KSBScheduledThreadPoolExecutor.class.getSimpleName(), e);
		}

	}
	
	private static class KSBThreadFactory implements ThreadFactory {
		
		private ThreadFactory defaultThreadFactory = Executors.defaultThreadFactory();
		
		@Override
        public Thread newThread(Runnable runnable) {
			Thread thread = defaultThreadFactory.newThread(runnable);
			thread.setName("KSB-Scheduled-" + thread.getName());
			return thread;
	    }
	}

}
