/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings for a one-time message that's sent directly to an endpoint through the ADM (Amazon Device
 * Messaging) channel.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ADMMessage implements SdkPojo, Serializable, ToCopyableBuilder<ADMMessage.Builder, ADMMessage> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(ADMMessage::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Body")
            .getter(getter(ADMMessage::body)).setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Body").build()).build();

    private static final SdkField<String> CONSOLIDATION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConsolidationKey").getter(getter(ADMMessage::consolidationKey))
            .setter(setter(Builder::consolidationKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsolidationKey").build()).build();

    private static final SdkField<Map<String, String>> DATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Data")
            .getter(getter(ADMMessage::data))
            .setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> EXPIRES_AFTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpiresAfter").getter(getter(ADMMessage::expiresAfter)).setter(setter(Builder::expiresAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpiresAfter").build()).build();

    private static final SdkField<String> ICON_REFERENCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IconReference").getter(getter(ADMMessage::iconReference)).setter(setter(Builder::iconReference))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IconReference").build()).build();

    private static final SdkField<String> IMAGE_ICON_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageIconUrl").getter(getter(ADMMessage::imageIconUrl)).setter(setter(Builder::imageIconUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageIconUrl").build()).build();

    private static final SdkField<String> IMAGE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUrl").getter(getter(ADMMessage::imageUrl)).setter(setter(Builder::imageUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUrl").build()).build();

    private static final SdkField<String> MD5_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("MD5")
            .getter(getter(ADMMessage::md5)).setter(setter(Builder::md5))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MD5").build()).build();

    private static final SdkField<String> RAW_CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RawContent").getter(getter(ADMMessage::rawContent)).setter(setter(Builder::rawContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RawContent").build()).build();

    private static final SdkField<Boolean> SILENT_PUSH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SilentPush").getter(getter(ADMMessage::silentPush)).setter(setter(Builder::silentPush))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SilentPush").build()).build();

    private static final SdkField<String> SMALL_IMAGE_ICON_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SmallImageIconUrl").getter(getter(ADMMessage::smallImageIconUrl))
            .setter(setter(Builder::smallImageIconUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SmallImageIconUrl").build()).build();

    private static final SdkField<String> SOUND_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Sound")
            .getter(getter(ADMMessage::sound)).setter(setter(Builder::sound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sound").build()).build();

    private static final SdkField<Map<String, List<String>>> SUBSTITUTIONS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Substitutions")
            .getter(getter(ADMMessage::substitutions))
            .setter(setter(Builder::substitutions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Substitutions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(ADMMessage::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(ADMMessage::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, BODY_FIELD,
            CONSOLIDATION_KEY_FIELD, DATA_FIELD, EXPIRES_AFTER_FIELD, ICON_REFERENCE_FIELD, IMAGE_ICON_URL_FIELD,
            IMAGE_URL_FIELD, MD5_FIELD, RAW_CONTENT_FIELD, SILENT_PUSH_FIELD, SMALL_IMAGE_ICON_URL_FIELD, SOUND_FIELD,
            SUBSTITUTIONS_FIELD, TITLE_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final String body;

    private final String consolidationKey;

    private final Map<String, String> data;

    private final String expiresAfter;

    private final String iconReference;

    private final String imageIconUrl;

    private final String imageUrl;

    private final String md5;

    private final String rawContent;

    private final Boolean silentPush;

    private final String smallImageIconUrl;

    private final String sound;

    private final Map<String, List<String>> substitutions;

    private final String title;

    private final String url;

    private ADMMessage(BuilderImpl builder) {
        this.action = builder.action;
        this.body = builder.body;
        this.consolidationKey = builder.consolidationKey;
        this.data = builder.data;
        this.expiresAfter = builder.expiresAfter;
        this.iconReference = builder.iconReference;
        this.imageIconUrl = builder.imageIconUrl;
        this.imageUrl = builder.imageUrl;
        this.md5 = builder.md5;
        this.rawContent = builder.rawContent;
        this.silentPush = builder.silentPush;
        this.smallImageIconUrl = builder.smallImageIconUrl;
        this.sound = builder.sound;
        this.substitutions = builder.substitutions;
        this.title = builder.title;
        this.url = builder.url;
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the deep-linking
     * features of the Android platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
     *         deep-linking features of the Android platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public final Action action() {
        return Action.fromValue(action);
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the deep-linking
     * features of the Android platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
     *         deep-linking features of the Android platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The body of the notification message.
     * </p>
     * 
     * @return The body of the notification message.
     */
    public final String body() {
        return body;
    }

    /**
     * <p>
     * An arbitrary string that indicates that multiple messages are logically the same and that Amazon Device Messaging
     * (ADM) can drop previously enqueued messages in favor of this message.
     * </p>
     * 
     * @return An arbitrary string that indicates that multiple messages are logically the same and that Amazon Device
     *         Messaging (ADM) can drop previously enqueued messages in favor of this message.
     */
    public final String consolidationKey() {
        return consolidationKey;
    }

    /**
     * Returns true if the Data property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasData() {
        return data != null && !(data instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The JSON data payload to use for the push notification, if the notification is a silent push notification. This
     * payload is added to the data.pinpoint.jsonBody object of the notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasData()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The JSON data payload to use for the push notification, if the notification is a silent push
     *         notification. This payload is added to the data.pinpoint.jsonBody object of the notification.
     */
    public final Map<String, String> data() {
        return data;
    }

    /**
     * <p>
     * The amount of time, in seconds, that ADM should store the message if the recipient's device is offline. Amazon
     * Pinpoint specifies this value in the expiresAfter parameter when it sends the notification message to ADM.
     * </p>
     * 
     * @return The amount of time, in seconds, that ADM should store the message if the recipient's device is offline.
     *         Amazon Pinpoint specifies this value in the expiresAfter parameter when it sends the notification message
     *         to ADM.
     */
    public final String expiresAfter() {
        return expiresAfter;
    }

    /**
     * <p>
     * The icon image name of the asset saved in your app.
     * </p>
     * 
     * @return The icon image name of the asset saved in your app.
     */
    public final String iconReference() {
        return iconReference;
    }

    /**
     * <p>
     * The URL of the large icon image to display in the content view of the push notification.
     * </p>
     * 
     * @return The URL of the large icon image to display in the content view of the push notification.
     */
    public final String imageIconUrl() {
        return imageIconUrl;
    }

    /**
     * <p>
     * The URL of an image to display in the push notification.
     * </p>
     * 
     * @return The URL of an image to display in the push notification.
     */
    public final String imageUrl() {
        return imageUrl;
    }

    /**
     * <p>
     * The base64-encoded, MD5 checksum of the value specified by the Data property. ADM uses the MD5 value to verify
     * the integrity of the data.
     * </p>
     * 
     * @return The base64-encoded, MD5 checksum of the value specified by the Data property. ADM uses the MD5 value to
     *         verify the integrity of the data.
     */
    public final String md5() {
        return md5;
    }

    /**
     * <p>
     * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
     * overrides all other content for the message.
     * </p>
     * 
     * @return The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
     *         value overrides all other content for the message.
     */
    public final String rawContent() {
        return rawContent;
    }

    /**
     * <p>
     * Specifies whether the notification is a silent push notification, which is a push notification that doesn't
     * display on a recipient's device. Silent push notifications can be used for cases such as updating an app's
     * configuration or supporting phone home functionality.
     * </p>
     * 
     * @return Specifies whether the notification is a silent push notification, which is a push notification that
     *         doesn't display on a recipient's device. Silent push notifications can be used for cases such as updating
     *         an app's configuration or supporting phone home functionality.
     */
    public final Boolean silentPush() {
        return silentPush;
    }

    /**
     * <p>
     * The URL of the small icon image to display in the status bar and the content view of the push notification.
     * </p>
     * 
     * @return The URL of the small icon image to display in the status bar and the content view of the push
     *         notification.
     */
    public final String smallImageIconUrl() {
        return smallImageIconUrl;
    }

    /**
     * <p>
     * The sound to play when the recipient receives the push notification. You can use the default stream or specify
     * the file name of a sound resource that's bundled in your app. On an Android platform, the sound file must reside
     * in /res/raw/.
     * </p>
     * 
     * @return The sound to play when the recipient receives the push notification. You can use the default stream or
     *         specify the file name of a sound resource that's bundled in your app. On an Android platform, the sound
     *         file must reside in /res/raw/.
     */
    public final String sound() {
        return sound;
    }

    /**
     * Returns true if the Substitutions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSubstitutions() {
        return substitutions != null && !(substitutions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The default message variables to use in the notification message. You can override the default variables with
     * individual address variables.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubstitutions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The default message variables to use in the notification message. You can override the default variables
     *         with individual address variables.
     */
    public final Map<String, List<String>> substitutions() {
        return substitutions;
    }

    /**
     * <p>
     * The title to display above the notification message on the recipient's device.
     * </p>
     * 
     * @return The title to display above the notification message on the recipient's device.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
     * value of the Action property is URL.
     * </p>
     * 
     * @return The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and
     *         the value of the Action property is URL.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(consolidationKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasData() ? data() : null);
        hashCode = 31 * hashCode + Objects.hashCode(expiresAfter());
        hashCode = 31 * hashCode + Objects.hashCode(iconReference());
        hashCode = 31 * hashCode + Objects.hashCode(imageIconUrl());
        hashCode = 31 * hashCode + Objects.hashCode(imageUrl());
        hashCode = 31 * hashCode + Objects.hashCode(md5());
        hashCode = 31 * hashCode + Objects.hashCode(rawContent());
        hashCode = 31 * hashCode + Objects.hashCode(silentPush());
        hashCode = 31 * hashCode + Objects.hashCode(smallImageIconUrl());
        hashCode = 31 * hashCode + Objects.hashCode(sound());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubstitutions() ? substitutions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ADMMessage)) {
            return false;
        }
        ADMMessage other = (ADMMessage) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(body(), other.body())
                && Objects.equals(consolidationKey(), other.consolidationKey()) && hasData() == other.hasData()
                && Objects.equals(data(), other.data()) && Objects.equals(expiresAfter(), other.expiresAfter())
                && Objects.equals(iconReference(), other.iconReference()) && Objects.equals(imageIconUrl(), other.imageIconUrl())
                && Objects.equals(imageUrl(), other.imageUrl()) && Objects.equals(md5(), other.md5())
                && Objects.equals(rawContent(), other.rawContent()) && Objects.equals(silentPush(), other.silentPush())
                && Objects.equals(smallImageIconUrl(), other.smallImageIconUrl()) && Objects.equals(sound(), other.sound())
                && hasSubstitutions() == other.hasSubstitutions() && Objects.equals(substitutions(), other.substitutions())
                && Objects.equals(title(), other.title()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ADMMessage").add("Action", actionAsString()).add("Body", body())
                .add("ConsolidationKey", consolidationKey()).add("Data", hasData() ? data() : null)
                .add("ExpiresAfter", expiresAfter()).add("IconReference", iconReference()).add("ImageIconUrl", imageIconUrl())
                .add("ImageUrl", imageUrl()).add("MD5", md5()).add("RawContent", rawContent()).add("SilentPush", silentPush())
                .add("SmallImageIconUrl", smallImageIconUrl()).add("Sound", sound())
                .add("Substitutions", hasSubstitutions() ? substitutions() : null).add("Title", title()).add("Url", url())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Body":
            return Optional.ofNullable(clazz.cast(body()));
        case "ConsolidationKey":
            return Optional.ofNullable(clazz.cast(consolidationKey()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "ExpiresAfter":
            return Optional.ofNullable(clazz.cast(expiresAfter()));
        case "IconReference":
            return Optional.ofNullable(clazz.cast(iconReference()));
        case "ImageIconUrl":
            return Optional.ofNullable(clazz.cast(imageIconUrl()));
        case "ImageUrl":
            return Optional.ofNullable(clazz.cast(imageUrl()));
        case "MD5":
            return Optional.ofNullable(clazz.cast(md5()));
        case "RawContent":
            return Optional.ofNullable(clazz.cast(rawContent()));
        case "SilentPush":
            return Optional.ofNullable(clazz.cast(silentPush()));
        case "SmallImageIconUrl":
            return Optional.ofNullable(clazz.cast(smallImageIconUrl()));
        case "Sound":
            return Optional.ofNullable(clazz.cast(sound()));
        case "Substitutions":
            return Optional.ofNullable(clazz.cast(substitutions()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ADMMessage, T> g) {
        return obj -> g.apply((ADMMessage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ADMMessage> {
        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         * deep-linking features of the Android platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         *        deep-linking features of the Android platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(String action);

        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         * deep-linking features of the Android platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         *        deep-linking features of the Android platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(Action action);

        /**
         * <p>
         * The body of the notification message.
         * </p>
         * 
         * @param body
         *        The body of the notification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * An arbitrary string that indicates that multiple messages are logically the same and that Amazon Device
         * Messaging (ADM) can drop previously enqueued messages in favor of this message.
         * </p>
         * 
         * @param consolidationKey
         *        An arbitrary string that indicates that multiple messages are logically the same and that Amazon
         *        Device Messaging (ADM) can drop previously enqueued messages in favor of this message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consolidationKey(String consolidationKey);

        /**
         * <p>
         * The JSON data payload to use for the push notification, if the notification is a silent push notification.
         * This payload is added to the data.pinpoint.jsonBody object of the notification.
         * </p>
         * 
         * @param data
         *        The JSON data payload to use for the push notification, if the notification is a silent push
         *        notification. This payload is added to the data.pinpoint.jsonBody object of the notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(Map<String, String> data);

        /**
         * <p>
         * The amount of time, in seconds, that ADM should store the message if the recipient's device is offline.
         * Amazon Pinpoint specifies this value in the expiresAfter parameter when it sends the notification message to
         * ADM.
         * </p>
         * 
         * @param expiresAfter
         *        The amount of time, in seconds, that ADM should store the message if the recipient's device is
         *        offline. Amazon Pinpoint specifies this value in the expiresAfter parameter when it sends the
         *        notification message to ADM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresAfter(String expiresAfter);

        /**
         * <p>
         * The icon image name of the asset saved in your app.
         * </p>
         * 
         * @param iconReference
         *        The icon image name of the asset saved in your app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iconReference(String iconReference);

        /**
         * <p>
         * The URL of the large icon image to display in the content view of the push notification.
         * </p>
         * 
         * @param imageIconUrl
         *        The URL of the large icon image to display in the content view of the push notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIconUrl(String imageIconUrl);

        /**
         * <p>
         * The URL of an image to display in the push notification.
         * </p>
         * 
         * @param imageUrl
         *        The URL of an image to display in the push notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUrl(String imageUrl);

        /**
         * <p>
         * The base64-encoded, MD5 checksum of the value specified by the Data property. ADM uses the MD5 value to
         * verify the integrity of the data.
         * </p>
         * 
         * @param md5
         *        The base64-encoded, MD5 checksum of the value specified by the Data property. ADM uses the MD5 value
         *        to verify the integrity of the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder md5(String md5);

        /**
         * <p>
         * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
         * overrides all other content for the message.
         * </p>
         * 
         * @param rawContent
         *        The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
         *        value overrides all other content for the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawContent(String rawContent);

        /**
         * <p>
         * Specifies whether the notification is a silent push notification, which is a push notification that doesn't
         * display on a recipient's device. Silent push notifications can be used for cases such as updating an app's
         * configuration or supporting phone home functionality.
         * </p>
         * 
         * @param silentPush
         *        Specifies whether the notification is a silent push notification, which is a push notification that
         *        doesn't display on a recipient's device. Silent push notifications can be used for cases such as
         *        updating an app's configuration or supporting phone home functionality.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder silentPush(Boolean silentPush);

        /**
         * <p>
         * The URL of the small icon image to display in the status bar and the content view of the push notification.
         * </p>
         * 
         * @param smallImageIconUrl
         *        The URL of the small icon image to display in the status bar and the content view of the push
         *        notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smallImageIconUrl(String smallImageIconUrl);

        /**
         * <p>
         * The sound to play when the recipient receives the push notification. You can use the default stream or
         * specify the file name of a sound resource that's bundled in your app. On an Android platform, the sound file
         * must reside in /res/raw/.
         * </p>
         * 
         * @param sound
         *        The sound to play when the recipient receives the push notification. You can use the default stream or
         *        specify the file name of a sound resource that's bundled in your app. On an Android platform, the
         *        sound file must reside in /res/raw/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sound(String sound);

        /**
         * <p>
         * The default message variables to use in the notification message. You can override the default variables with
         * individual address variables.
         * </p>
         * 
         * @param substitutions
         *        The default message variables to use in the notification message. You can override the default
         *        variables with individual address variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder substitutions(Map<String, ? extends Collection<String>> substitutions);

        /**
         * <p>
         * The title to display above the notification message on the recipient's device.
         * </p>
         * 
         * @param title
         *        The title to display above the notification message on the recipient's device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
         * value of the Action property is URL.
         * </p>
         * 
         * @param url
         *        The URL to open in the recipient's default mobile browser, if a recipient taps the push notification
         *        and the value of the Action property is URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private String body;

        private String consolidationKey;

        private Map<String, String> data = DefaultSdkAutoConstructMap.getInstance();

        private String expiresAfter;

        private String iconReference;

        private String imageIconUrl;

        private String imageUrl;

        private String md5;

        private String rawContent;

        private Boolean silentPush;

        private String smallImageIconUrl;

        private String sound;

        private Map<String, List<String>> substitutions = DefaultSdkAutoConstructMap.getInstance();

        private String title;

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(ADMMessage model) {
            action(model.action);
            body(model.body);
            consolidationKey(model.consolidationKey);
            data(model.data);
            expiresAfter(model.expiresAfter);
            iconReference(model.iconReference);
            imageIconUrl(model.imageIconUrl);
            imageUrl(model.imageUrl);
            md5(model.md5);
            rawContent(model.rawContent);
            silentPush(model.silentPush);
            smallImageIconUrl(model.smallImageIconUrl);
            sound(model.sound);
            substitutions(model.substitutions);
            title(model.title);
            url(model.url);
        }

        public final String getAction() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(Action action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final String getBody() {
            return body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        public final String getConsolidationKey() {
            return consolidationKey;
        }

        @Override
        public final Builder consolidationKey(String consolidationKey) {
            this.consolidationKey = consolidationKey;
            return this;
        }

        public final void setConsolidationKey(String consolidationKey) {
            this.consolidationKey = consolidationKey;
        }

        public final Map<String, String> getData() {
            if (data instanceof SdkAutoConstructMap) {
                return null;
            }
            return data;
        }

        @Override
        public final Builder data(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
            return this;
        }

        public final void setData(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
        }

        public final String getExpiresAfter() {
            return expiresAfter;
        }

        @Override
        public final Builder expiresAfter(String expiresAfter) {
            this.expiresAfter = expiresAfter;
            return this;
        }

        public final void setExpiresAfter(String expiresAfter) {
            this.expiresAfter = expiresAfter;
        }

        public final String getIconReference() {
            return iconReference;
        }

        @Override
        public final Builder iconReference(String iconReference) {
            this.iconReference = iconReference;
            return this;
        }

        public final void setIconReference(String iconReference) {
            this.iconReference = iconReference;
        }

        public final String getImageIconUrl() {
            return imageIconUrl;
        }

        @Override
        public final Builder imageIconUrl(String imageIconUrl) {
            this.imageIconUrl = imageIconUrl;
            return this;
        }

        public final void setImageIconUrl(String imageIconUrl) {
            this.imageIconUrl = imageIconUrl;
        }

        public final String getImageUrl() {
            return imageUrl;
        }

        @Override
        public final Builder imageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
            return this;
        }

        public final void setImageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
        }

        public final String getMd5() {
            return md5;
        }

        @Override
        public final Builder md5(String md5) {
            this.md5 = md5;
            return this;
        }

        public final void setMd5(String md5) {
            this.md5 = md5;
        }

        public final String getRawContent() {
            return rawContent;
        }

        @Override
        public final Builder rawContent(String rawContent) {
            this.rawContent = rawContent;
            return this;
        }

        public final void setRawContent(String rawContent) {
            this.rawContent = rawContent;
        }

        public final Boolean getSilentPush() {
            return silentPush;
        }

        @Override
        public final Builder silentPush(Boolean silentPush) {
            this.silentPush = silentPush;
            return this;
        }

        public final void setSilentPush(Boolean silentPush) {
            this.silentPush = silentPush;
        }

        public final String getSmallImageIconUrl() {
            return smallImageIconUrl;
        }

        @Override
        public final Builder smallImageIconUrl(String smallImageIconUrl) {
            this.smallImageIconUrl = smallImageIconUrl;
            return this;
        }

        public final void setSmallImageIconUrl(String smallImageIconUrl) {
            this.smallImageIconUrl = smallImageIconUrl;
        }

        public final String getSound() {
            return sound;
        }

        @Override
        public final Builder sound(String sound) {
            this.sound = sound;
            return this;
        }

        public final void setSound(String sound) {
            this.sound = sound;
        }

        public final Map<String, ? extends Collection<String>> getSubstitutions() {
            if (substitutions instanceof SdkAutoConstructMap) {
                return null;
            }
            return substitutions;
        }

        @Override
        public final Builder substitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
            return this;
        }

        public final void setSubstitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getUrl() {
            return url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public ADMMessage build() {
            return new ADMMessage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
