/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.lang.document;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import net.sourceforge.pmd.internal.util.IOUtil;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.LanguageVersion;
import net.sourceforge.pmd.lang.LanguageVersionDiscoverer;
import net.sourceforge.pmd.lang.document.FileId;
import net.sourceforge.pmd.lang.document.TextFile;
import net.sourceforge.pmd.util.AssertionUtil;
import net.sourceforge.pmd.util.log.PmdReporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileCollector
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(FileCollector.class);
    private final Set<TextFile> allFilesToProcess = new LinkedHashSet<TextFile>();
    private final List<Closeable> resourcesToClose = new ArrayList<Closeable>();
    private Charset charset = StandardCharsets.UTF_8;
    private final LanguageVersionDiscoverer discoverer;
    private final PmdReporter reporter;
    private final FileId outerFsPath;
    private boolean closed;
    private boolean recursive = true;
    private Predicate<FileId> fileFilter = file -> true;

    private FileCollector(LanguageVersionDiscoverer discoverer, PmdReporter reporter, FileId outerFsPath) {
        this.discoverer = discoverer;
        this.reporter = reporter;
        this.outerFsPath = outerFsPath;
        LOG.debug("Created new FileCollector with {}", (Object)discoverer);
    }

    static FileCollector newCollector(LanguageVersionDiscoverer discoverer, PmdReporter reporter) {
        return new FileCollector(discoverer, reporter, null);
    }

    FileCollector newCollector(PmdReporter logger) {
        FileCollector fileCollector = new FileCollector(this.discoverer, logger, null);
        fileCollector.charset = this.charset;
        return fileCollector;
    }

    public List<TextFile> getCollectedFiles() {
        if (this.closed) {
            throw new IllegalStateException("Collector was closed!");
        }
        ArrayList<TextFile> allFilesToProcess = new ArrayList<TextFile>(this.allFilesToProcess);
        allFilesToProcess.sort(Comparator.comparing(TextFile::getFileId));
        return Collections.unmodifiableList(allFilesToProcess);
    }

    public PmdReporter getReporter() {
        return this.reporter;
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        Exception exception = IOUtil.closeAll(this.resourcesToClose);
        if (exception != null) {
            this.reporter.errorEx("Error while closing resources", exception);
        }
    }

    public boolean addFile(Path file) {
        if (!Files.isRegularFile(file, new LinkOption[0])) {
            this.reporter.error("Not a regular file: {0}", file);
            return false;
        }
        LanguageVersion languageVersion = this.discoverLanguage(file.toString());
        return languageVersion != null && this.addFileImpl(TextFile.builderForPath(file, this.charset, languageVersion).setParentFsPath(this.outerFsPath).build());
    }

    public boolean addFile(Path file, Language language) {
        AssertionUtil.requireParamNotNull("language", language);
        if (!Files.isRegularFile(file, new LinkOption[0])) {
            this.reporter.error("Not a regular file: {0}", file);
            return false;
        }
        LanguageVersion lv = this.discoverer.getDefaultLanguageVersion(language);
        Objects.requireNonNull(lv);
        return this.addFileImpl(TextFile.builderForPath(file, this.charset, lv).setParentFsPath(this.outerFsPath).build());
    }

    public boolean addFile(TextFile textFile) {
        AssertionUtil.requireParamNotNull("textFile", textFile);
        return this.checkContextualVersion(textFile) && this.addFileImpl(textFile);
    }

    public boolean addSourceFile(FileId fileId, String sourceContents) {
        AssertionUtil.requireParamNotNull("sourceContents", sourceContents);
        AssertionUtil.requireParamNotNull("pathId", fileId);
        LanguageVersion version = this.discoverLanguage(fileId.getFileName());
        return version != null && this.addFileImpl(TextFile.builderForCharSeq(sourceContents, fileId, version).setParentFsPath(this.outerFsPath).build());
    }

    private boolean addFileImpl(TextFile textFile) {
        LOG.trace("Adding file {} (lang: {}) ", (Object)textFile.getFileId().getAbsolutePath(), (Object)textFile.getLanguageVersion().getTerseName());
        if (!this.fileFilter.test(textFile.getFileId())) {
            LOG.trace("File was skipped due to fileFilter...");
            return false;
        }
        if (this.allFilesToProcess.add(textFile)) {
            return true;
        }
        LOG.trace("File was already collected, skipping");
        return false;
    }

    private LanguageVersion discoverLanguage(String file) {
        if (this.discoverer.getForcedVersion() != null) {
            return this.discoverer.getForcedVersion();
        }
        List<Language> languages = this.discoverer.getLanguagesForFile(file);
        if (languages.isEmpty()) {
            LOG.trace("File {} matches no known language, ignoring", (Object)file);
            return null;
        }
        Language lang = languages.get(0);
        if (languages.size() > 1) {
            LOG.trace("File {} matches multiple languages ({}), selecting {}", new Object[]{file, languages, lang});
        }
        return this.discoverer.getDefaultLanguageVersion(lang);
    }

    private boolean checkContextualVersion(TextFile textFile) {
        Language language;
        LanguageVersion contextVersion;
        LanguageVersion fileVersion = textFile.getLanguageVersion();
        if (!fileVersion.equals(contextVersion = this.discoverer.getDefaultLanguageVersion(language = fileVersion.getLanguage()))) {
            this.reporter.error("Cannot add file {0}: version ''{1}'' does not match ''{2}''", textFile.getFileId(), fileVersion, contextVersion);
            return false;
        }
        return true;
    }

    public boolean addDirectory(Path dir) throws IOException {
        return this.addDirectory(dir, this.recursive);
    }

    public boolean addDirectory(Path dir, boolean recurse) throws IOException {
        if (!Files.isDirectory(dir, new LinkOption[0])) {
            this.reporter.error("Not a directory {0}", dir);
            return false;
        }
        int maxDepth = recurse ? Integer.MAX_VALUE : 1;
        Files.walkFileTree(dir, EnumSet.of(FileVisitOption.FOLLOW_LINKS), maxDepth, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (attrs.isRegularFile()) {
                    FileCollector.this.addFile(file);
                }
                return super.visitFile(file, attrs);
            }
        });
        return true;
    }

    public boolean addFileOrDirectory(Path file) throws IOException {
        return this.addFileOrDirectory(file, true);
    }

    public boolean addFileOrDirectory(Path file, boolean recurseIfDirectory) throws IOException {
        if (Files.isDirectory(file, new LinkOption[0])) {
            return this.addDirectory(file, recurseIfDirectory);
        }
        if (Files.isRegularFile(file, new LinkOption[0])) {
            return this.addFile(file);
        }
        this.reporter.error("Not a file or directory {0}", file);
        return false;
    }

    public boolean addZipFileWithContent(Path zipFile) throws IOException {
        FileSystem fs;
        if (!Files.isRegularFile(zipFile, new LinkOption[0])) {
            throw new IllegalArgumentException("Not a regular file: " + zipFile);
        }
        URI zipUri = URI.create("jar:" + zipFile.toUri());
        boolean isNewFileSystem = false;
        try {
            fs = FileSystems.getFileSystem(zipUri);
        }
        catch (FileSystemNotFoundException ignored) {
            try {
                fs = FileSystems.newFileSystem(zipUri, Collections.emptyMap());
                isNewFileSystem = true;
            }
            catch (IOException | ProviderNotFoundException e) {
                this.reporter.errorEx("Cannot open zip file " + zipFile, e);
                return false;
            }
        }
        try (FileCollector zipCollector = this.newZipCollector(zipFile);){
            for (Path zipRoot : fs.getRootDirectories()) {
                zipCollector.addFileOrDirectory(zipRoot);
            }
            this.absorb(zipCollector);
            if (isNewFileSystem) {
                this.resourcesToClose.add(fs);
            }
        }
        catch (IOException ioe) {
            this.reporter.errorEx("Error reading zip file " + zipFile + ", will be skipped", ioe);
            fs.close();
            return false;
        }
        return true;
    }

    private FileCollector newZipCollector(Path zipFilePath) {
        return new FileCollector(this.discoverer, this.reporter, FileId.fromPath(zipFilePath));
    }

    public void setRecursive(boolean collectFilesRecursively) {
        this.recursive = collectFilesRecursively;
    }

    public void setCharset(Charset charset) {
        this.charset = Objects.requireNonNull(charset);
    }

    public void setFileFilter(Predicate<FileId> fileFilter) {
        this.fileFilter = Objects.requireNonNull(fileFilter);
    }

    public void exclude(FileCollector excludeCollector) {
        HashSet<TextFile> toExclude = new HashSet<TextFile>(excludeCollector.allFilesToProcess);
        Iterator<TextFile> iterator = this.allFilesToProcess.iterator();
        while (iterator.hasNext()) {
            TextFile file = iterator.next();
            if (!toExclude.contains(file)) continue;
            LOG.trace("Excluding file {}", (Object)file.getFileId());
            iterator.remove();
        }
    }

    public void absorb(FileCollector otherCollector) {
        this.allFilesToProcess.addAll(otherCollector.allFilesToProcess);
        this.resourcesToClose.addAll(otherCollector.resourcesToClose);
        otherCollector.allFilesToProcess.clear();
        otherCollector.resourcesToClose.clear();
    }

    public void filterLanguages(Set<Language> languages) {
        Iterator<TextFile> iterator = this.allFilesToProcess.iterator();
        while (iterator.hasNext()) {
            TextFile file = iterator.next();
            Language lang = file.getLanguageVersion().getLanguage();
            if (languages.contains(lang)) continue;
            LOG.trace("Filtering out {}, no rules for language {}", (Object)file.getFileId(), (Object)lang);
            iterator.remove();
        }
    }

    public String toString() {
        return "FileCollector{filesToProcess=" + this.allFilesToProcess + '}';
    }
}

