/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.configuration;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.directory.server.core.DefaultDirectoryService;
import org.apache.directory.server.core.DirectoryService;
import org.apache.directory.server.core.entry.ClonedServerEntry;
import org.apache.directory.server.core.entry.ServerEntry;
import org.apache.directory.server.ldap.LdapServer;
import org.apache.directory.server.protocol.shared.store.LdifFileLoader;
import org.apache.directory.server.protocol.shared.store.LdifLoadFilter;
import org.apache.directory.server.schema.registries.AttributeTypeRegistry;
import org.apache.directory.shared.ldap.name.LdapDN;
import org.apache.directory.shared.ldap.util.StringTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApacheDS {
    private static final Logger LOG = LoggerFactory.getLogger((String)ApacheDS.class.getName());
    private static final long DEFAULT_SYNC_PERIOD_MILLIS = 20000L;
    private long synchPeriodMillis = 20000L;
    private File ldifDirectory;
    private final List<LdifLoadFilter> ldifFilters = new ArrayList<LdifLoadFilter>();
    private final LdapServer ldapServer;
    private DirectoryService directoryService;

    public ApacheDS(LdapServer ldapServer) {
        LOG.info("Starting the Apache Directory Server");
        this.ldapServer = ldapServer;
        this.directoryService = ldapServer.getDirectoryService();
        if (this.directoryService == null) {
            this.directoryService = new DefaultDirectoryService();
        }
    }

    public void startup() throws Exception {
        LOG.debug("Starting the server");
        if (!this.directoryService.isStarted()) {
            LOG.debug("1. Starting the DirectoryService");
            this.directoryService.startup();
        }
        this.loadLdifs();
        if (this.ldapServer != null && !this.ldapServer.isStarted()) {
            LOG.debug("3. Starting the LDAP server");
            this.ldapServer.start();
        }
        LOG.debug("Server successfully started");
    }

    public boolean isStarted() {
        if (this.ldapServer != null) {
            return this.ldapServer.isStarted();
        }
        return this.directoryService.isStarted();
    }

    public void shutdown() throws Exception {
        if (this.ldapServer != null && this.ldapServer.isStarted()) {
            this.ldapServer.stop();
        }
        this.directoryService.shutdown();
    }

    public LdapServer getLdapServer() {
        return this.ldapServer;
    }

    public DirectoryService getDirectoryService() {
        return this.directoryService;
    }

    public long getSynchPeriodMillis() {
        return this.synchPeriodMillis;
    }

    public void setSynchPeriodMillis(long synchPeriodMillis) {
        LOG.info("Set the synchPeriodMillis to {}", (Object)synchPeriodMillis);
        this.synchPeriodMillis = synchPeriodMillis;
    }

    public File getLdifDirectory() {
        return this.ldifDirectory;
    }

    public void setLdifDirectory(File ldifDirectory) {
        LOG.info("The LDIF directory file is {}", (Object)ldifDirectory.getAbsolutePath());
        this.ldifDirectory = ldifDirectory;
    }

    private void ensureLdifFileBase() throws Exception {
        LdapDN dn = new LdapDN("ou=loadedLdifFiles,ou=configuration,ou=system");
        ClonedServerEntry entry = null;
        try {
            entry = this.directoryService.getAdminSession().lookup(dn);
        }
        catch (Exception e) {
            LOG.info("Failure while looking up {}. The entry will be created now.", (Object)"ou=loadedLdifFiles,ou=configuration,ou=system", (Object)e);
        }
        if (entry == null) {
            entry = this.directoryService.newEntry(new LdapDN("ou=loadedLdifFiles,ou=configuration,ou=system"));
            entry.add("ou", new String[]{"loadedLdifFiles"});
            entry.add("objectClass", new String[]{"top", "organizationalUnit"});
            this.directoryService.getAdminSession().add((ServerEntry)entry);
        }
    }

    private LdapDN buildProtectedFileEntryDn(File ldif) throws Exception {
        String fileSep = File.separatorChar == '\\' ? "windowsFilePath" : "unixFilePath";
        return new LdapDN(fileSep + "=" + StringTools.dumpHexPairs((byte[])StringTools.getBytesUtf8((String)this.getCanonical(ldif))) + "," + "ou=loadedLdifFiles,ou=configuration,ou=system");
    }

    private void addFileEntry(File ldif) throws Exception {
        String rdnAttr = File.separatorChar == '\\' ? "windowsFilePath" : "unixFilePath";
        String oc = File.separatorChar == '\\' ? "windowsFile" : "unixFile";
        ServerEntry entry = this.directoryService.newEntry(this.buildProtectedFileEntryDn(ldif));
        entry.add(rdnAttr, new String[]{this.getCanonical(ldif)});
        entry.add("objectClass", new String[]{"top", oc});
        this.directoryService.getAdminSession().add(entry);
    }

    private String getCanonical(File file) {
        String canonical;
        try {
            canonical = file.getCanonicalPath();
        }
        catch (IOException e) {
            LOG.error("could not get canonical path", (Throwable)e);
            return null;
        }
        return StringUtils.replace((String)canonical, (String)"\\", (String)"\\\\");
    }

    private void loadLdif(File ldifFile) throws Exception {
        ClonedServerEntry fileEntry = null;
        try {
            fileEntry = this.directoryService.getAdminSession().lookup(this.buildProtectedFileEntryDn(ldifFile));
        }
        catch (Exception e) {
            // empty catch block
        }
        if (fileEntry != null) {
            String time = fileEntry.getOriginalEntry().get("createTimestamp").getString();
            LOG.info("Load of LDIF file '" + this.getCanonical(ldifFile) + "' skipped.  It has already been loaded on " + time + ".");
        } else {
            LdifFileLoader loader = new LdifFileLoader(this.directoryService.getAdminSession(), ldifFile, this.ldifFilters);
            int count = loader.execute();
            LOG.info("Loaded " + count + " entries from LDIF file '" + this.getCanonical(ldifFile) + "'");
            this.addFileEntry(ldifFile);
        }
    }

    public void loadLdifs() throws Exception {
        if (this.ldifDirectory == null) {
            LOG.info("LDIF load directory not specified.  No LDIF files will be loaded.");
            return;
        }
        if (!this.ldifDirectory.exists()) {
            LOG.warn("LDIF load directory '{}' does not exist.  No LDIF files will be loaded.", (Object)this.getCanonical(this.ldifDirectory));
            return;
        }
        LdapDN dn = new LdapDN("uid=admin,ou=system");
        AttributeTypeRegistry reg = this.directoryService.getRegistries().getAttributeTypeRegistry();
        dn.normalize(reg.getNormalizerMapping());
        this.ensureLdifFileBase();
        if (this.ldifDirectory.isFile()) {
            if (LOG.isInfoEnabled()) {
                LOG.info("LDIF load directory '{}' is a file. Will attempt to load as LDIF.", (Object)this.getCanonical(this.ldifDirectory));
            }
            try {
                this.loadLdif(this.ldifDirectory);
            }
            catch (Exception ne) {
                LOG.error("Cannot load the ldif file '{}', error : ", (Object)this.ldifDirectory.getAbsolutePath(), (Object)ne.getMessage());
                throw ne;
            }
        } else {
            File[] ldifFiles = this.ldifDirectory.listFiles(new FileFilter(){

                public boolean accept(File pathname) {
                    boolean isLdif = pathname.getName().toLowerCase().endsWith(".ldif");
                    return pathname.isFile() && pathname.canRead() && isLdif;
                }
            });
            if (ldifFiles == null || ldifFiles.length == 0) {
                LOG.warn("LDIF load directory '{}' does not contain any LDIF files. No LDIF files will be loaded.", (Object)this.getCanonical(this.ldifDirectory));
                return;
            }
            for (File ldifFile : ldifFiles) {
                try {
                    LOG.info("Loading LDIF file '{}'", (Object)ldifFile.getName());
                    this.loadLdif(ldifFile);
                }
                catch (Exception ne) {
                    LOG.error("Cannot load the ldif file '{}', error : {}", (Object)ldifFile.getAbsolutePath(), (Object)ne.getMessage());
                    throw ne;
                }
            }
        }
    }
}

