/*
 * Copyright (C) 2002-2003, Simon Nieuviarts
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */
package org.objectweb.carol.cmi;

import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * The aim of this class is to generate unique IDs to identify an instance of a
 * clustered server. The IDs generated by this class are guaranteed to be
 * unique on the whole network. Several methods may be provided.
 * To generate an ID, time may be needed, because they can be generated by
 * locking a well know resource for a period of time.
 * @author Simon Nieuviarts
 */
public class ClusterIdFactory {
    private static byte localIdArray[];
    private static ClusterId localId;
    private static long date;
    private static java.net.ServerSocket ss;

    private ClusterIdFactory() {
    }

    public static synchronized void start() throws ClusterException {
        startTypeIp();
    }

    /**
     * Generate a cluster id based on a IP address.
     * @param a ip address
     */
    private static synchronized void startTypeIp() throws ClusterException {
        if (localIdArray != null) {
            return;
        }
        InetAddress a;
        try {
            a = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
            throw new ClusterException("Error in getLocalHost() : " + e.toString());
        }
        byte aa[] = a.getAddress();
        if (aa.length != 4) {
            throw new ClusterException(
                "IP Address of size " + aa.length + " not supported by ClusterIdFactory");
        }
        if (aa[0] == 127) {
            throw new ClusterException(
                "Loopback IP address not allowed in ClusterIdFactory : fix /etc/hosts (or equivalent) so that java.net.InetAddress.getLocalHost() does not return 127.x.x.x");
        }
        try {
            ss = new java.net.ServerSocket(0, 1, a);
        } catch (Exception e) {
            throw new ClusterException(e.toString());
        }
        int p = ss.getLocalPort();
        if ((p & 0xffff) != p) {
            throw new ClusterException(
                "Invalid port number (more than 16 bits) in ClusterIdFactory");
        }
        date = System.currentTimeMillis() / 1000;
        if (date >= 128L << 24) {
            throw new ClusterException("Invalid date (too high) in ClusterIdFactory");
        }
        localIdArray = new byte[10];
        localIdArray[0] = (byte) (p & 0xff);
        localIdArray[1] = (byte) (p >> 8);
        localIdArray[2] = aa[3];
        localIdArray[3] = aa[2];
        localIdArray[4] = aa[1];
        localIdArray[5] = aa[0];
        int n = 4;
        long d = date;
        while (n > 0) {
            n--;
            localIdArray[9 - n] = (byte) (d & 0xff);
            d >>= 8;
        }
        localId = ClusterId.toClusterId(localIdArray);
        SecureRandom.setSeed(localIdArray);
    }

    /**
     * Return the local cluster Id. Is <code>null</code> if generate()
     * has not been called before.
     * @return the cluster Id
     */
    public static synchronized ClusterId getLocalId() throws ClusterException {
        if (localIdArray == null) {
            start();
        }
        if (ss == null) {
            return localId;
        }
        while (true) {
            long d = System.currentTimeMillis();
            // d / 1000 should not be lower than date
            if ((d / 1000) > date) {
                try {
                    ss.close();
                } catch (java.io.IOException e) {
                    // Can not close. Nothing to do...
                }
                ss = null;
                return localId;
            }
            try {
                synchronized (localIdArray) {
                    localIdArray.wait(1001 - (d % 1000));
                }
            } catch (InterruptedException e) {
                // Nothing to do
            }
            d = System.currentTimeMillis();
        }
    }
}
