/*
 * Copyright 2006-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.batch.core.launch.support;

import org.springframework.batch.core.JobParameter;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.JobParametersIncrementer;
import org.springframework.lang.Nullable;

/**
 * This incrementer increments a "run.id" parameter of type {@link Long} from the given
 * job parameters. If the parameter does not exist, it will be initialized to 1. The
 * parameter name can be configured using {@link #setKey(String)}.
 *
 * @author Dave Syer
 * @author Mahmoud Ben Hassine
 * @author Jinho Han
 */
public class RunIdIncrementer implements JobParametersIncrementer {

	private static final String RUN_ID_KEY = "run.id";

	private String key = RUN_ID_KEY;

	/**
	 * The name of the run id in the job parameters. Defaults to "run.id".
	 * @param key the key to set
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * Increment the run.id parameter (starting with 1).
	 * @param parameters the previous job parameters
	 * @return the next job parameters with an incremented (or initialized) run.id
	 * @throws IllegalArgumentException if the previous value of run.id is invalid
	 */
	@Override
	public JobParameters getNext(@Nullable JobParameters parameters) {

		JobParameters params = (parameters == null) ? new JobParameters() : parameters;
		JobParameter<?> runIdParameter = params.getParameters().get(this.key);
		long id = 1;
		if (runIdParameter != null) {
			try {
				id = Long.parseLong(runIdParameter.getValue().toString()) + 1;
			}
			catch (NumberFormatException exception) {
				throw new IllegalArgumentException("Invalid value for parameter " + this.key, exception);
			}
		}
		return new JobParametersBuilder(params).addLong(this.key, id).toJobParameters();
	}

}
