/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TransactionManager.java 1506 2007-06-30 15:34:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.transaction;
 
/**
 * The TransactionManager interface defines the methods that
 * allow an application server to manage transaction boundaries.
 * @author Guillaume Sauthier
 */
public interface TransactionManager {

    /**
     * Create a new transaction and associate it with the current thread.
     * @throws NotSupportedException Thrown if the thread is already
     *         associated with a transaction and the Transaction Manager
     *         implementation does not support nested transactions.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void begin() throws NotSupportedException, SystemException;
    
    /**
     * Complete the transaction associated with the current thread. When
     * this method completes, the thread is no longer associated with a
     * transaction.
     * @throws RollbackException Thrown to indicate that the transaction
     *         has been rolled back rather than committed.
     * @throws HeuristicMixedException Thrown to indicate that a heuristic
     *         decision was made and that some relevant updates have been
     *         committed while others have been rolled back.
     * @throws HeuristicRollbackException Thrown to indicate that a
     *         heuristic decision was made and that all relevant updates
     *         have been rolled back.
     * @throws SecurityException Thrown to indicate that the thread is not
     *         allowed to commit the transaction.
     * @throws IllegalStateExceptionThrown if the current thread is not
     *         associated with a transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void commit() throws RollbackException,
                         HeuristicMixedException,
                         HeuristicRollbackException,
                         SecurityException,
                         IllegalStateException,
                         SystemException;
    
    /**
     * Obtain the status of the transaction associated with the current thread.
     * @return The transaction status. If no transaction is associated with
     *         the current thread, this method returns the {@link Status}.NoTransaction value.
     * @throws SystemException Thrown if the transaction manager encounters an
     *         unexpected error condition.
     */
    int getStatus() throws SystemException;
    
    /**
     * Get the transaction object that represents the transaction context of
     * the calling thread.
     * @return the {@link Transaction} object representing the transaction
     *         associated with the calling thread.
     * @throws SystemException
     */
    Transaction getTransaction() throws SystemException;
    
    /**
     * Resume the transaction context association of the calling thread
     * with the transaction represented by the supplied Transaction
     * object. When this method returns, the calling thread is associated
     * with the transaction context specified.
     * @param tobj The {@link Transaction} object that represents the
     *        transaction to be resumed.
     * @throws InvalidTransactionException Thrown if the parameter
     *         transaction object contains an invalid transaction.
     * @throws IllegalStateException Thrown if the thread is already
     *         associated with another transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void resume(Transaction tobj) throws InvalidTransactionException,
                                         IllegalStateException,
                                         SystemException;
    
    /**
     * Roll back the transaction associated with the current thread. When
     * this method completes, the thread is no longer associated with a
     * transaction.
     * @throws IllegalStateException Thrown if the current thread is not
     *         associated with a transaction.
     * @throws SecurityException Thrown to indicate that the thread is not
     *         allowed to roll back the transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void rollback() throws IllegalStateException,
                           SecurityException,
                           SystemException;
    
    /**
     * Modify the transaction associated with the current thread such that
     * the only possible outcome of the transaction is to roll back the
     * transaction.
     * @throws IllegalStateException Thrown if the current thread is not
     *         associated with a transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void setRollbackOnly() throws IllegalStateException, SystemException;
    
    /**
     * Modify the timeout value that is associated with transactions started
     * by the current thread with the begin method.<br/>
     * If an application has not called this method, the transaction service
     * uses some default value for the transaction timeout. 
     * @param seconds The value of the timeout in seconds. If the value is
     *        zero, the transaction service restores the default value. If
     *        the value is negative a SystemException is thrown.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void setTransactionTimeout(int seconds) throws SystemException;
    
    /**
     * Suspend the transaction currently associated with the calling thread
     * and return a Transaction object that represents the transaction context
     * being suspended. If the calling thread is not associated with a
     * transaction, the method returns a null object reference. When this
     * method returns, the calling thread is not associated with a transaction.
     * @return Transaction object representing the suspended transaction.
     * @throws SystemException Thrown if the transaction manager encounters an
     *         unexpected error condition.
     */
    Transaction suspend() throws SystemException;
}
