/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TransactionSynchronizationRegistry.java 1506 2007-06-30 15:34:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.transaction;

/** 
 * This interface is intended for use by system level application server
 * components such as persistence managers, resource adapters, as well
 * as EJB and Web application components. This provides the ability to
 * register synchronization objects with special ordering semantics,
 * associate resource objects with the current transaction, get the
 * transaction context of the current transaction, get current transaction
 * status, and mark the current transaction for rollback. This interface
 * is implemented by the application server by a stateless service object.
 * The same object can be used by any number of components with thread safety.<br/>
 * 
 * In standard application server environments, an instance implementing this
 * interface can be looked up by a standard name via JNDI. The standard name
 * is <code>java:comp/TransactionSynchronizationRegistry</code>.

 * @since JTA 1.1
 * @author Guillaume Sauthier
 */
public interface TransactionSynchronizationRegistry {

    /**
     * Return an opaque object to represent the transaction bound to
     * the current thread at the time this method is called. This
     * object overrides hashCode and equals to allow its use as the
     * key in a hashMap for use by the caller. If there is no
     * transaction currently active, return null.<br/>
     * 
     * This object will return the same hashCode and compare equal to
     * all other objects returned by calling this method from any
     * component executing in the same transaction context in the
     * same application server.<br/>
     * 
     * The toString method returns a String that might be usable
     * by a human reader to usefully understand the transaction
     * context. The toString result is otherwise not defined.
     * Specifically, there is no forward or backward compatibility
     * guarantee of the results of toString.<br/>
     * 
     * The object is not necessarily serializable, and has no defined
     * behavior outside the virtual machine whence it was obtained. 
     * @return an opaque object representing the transaction bound
     *         to the current thread at the time this method is called.
     * @since JTA 1.1
     */
    Object getTransactionKey();
    
    /**
     * Add or replace an object in the Map of resources being managed
     * for the transaction bound to the current thread at the time this
     * method is called. The supplied key should be of an caller-defined
     * class so as not to conflict with other users. The class of the
     * key must guarantee that the hashCode and equals methods are
     * suitable for use as keys in a map. The key and value are not
     * examined or used by the implementation. The general contract
     * of this method is that of <code>java.util.Map#put(Object, Object)</code>
     * for a Map that supports non-null keys and null values. For
     * example, if there is already an value associated with the key,
     * it is replaced by the value parameter.
     * @param key the key for the Map entry.
     * @param value the value for the Map entry.
     * @throws IllegalStateException if no transaction is active.
     * @throws NullPointerException if the parameter key is null.
     * @since JTA 1.1
     */
    void putResource(Object key, Object value);
    
    /**
     * Get an object from the Map of resources being managed for the
     * transaction bound to the current thread at the time this method
     * is called. The key should have been supplied earlier by a call
     * to putResouce in the same transaction. If the key cannot be
     * found in the current resource Map, null is returned. The general
     * contract of this method is that of <code>java.util.Map#get(Object)</code>
     * for a Map that supports non-null keys and null values. For example,
     * the returned value is null if there is no entry for the parameter
     * key or if the value associated with the key is actually null.
     * @param key the key for the Map entry.
     * @return the value associated with the key.
     * @throws IllegalStateException if no transaction is active.
     * @throws NullPointerException if the parameter key is null.
     * @since JTA 1.1
     */
    Object getResource(Object key);
    
    /**
     * Register a Synchronization instance with special ordering
     * semantics. Its beforeCompletion will be called after all
     * SessionSynchronization beforeCompletion callbacks and callbacks
     * registered directly with the Transaction, but before the
     * 2-phase commit process starts. Similarly, the afterCompletion
     * callback will be called after 2-phase commit completes but
     * before any SessionSynchronization and Transaction afterCompletion
     * callbacks.<br/>
     * 
     * The beforeCompletion callback will be invoked in the transaction
     * context of the transaction bound to the current thread at the
     * time this method is called. Allowable methods include access
     * to resources, e.g. Connectors. No access is allowed to "user components"
     * (e.g. timer services or bean methods), as these might change the
     * state of data being managed by the caller, and might change the
     * state of data that has already been flushed by another caller
     * of registerInterposedSynchronization. The general context is
     * the component context of the caller of registerInterposedSynchronization.<br/>
     * 
     * The afterCompletion callback will be invoked in an undefined
     * context. No access is permitted to "user components" as defined
     * above. Resources can be closed but no transactional work can
     * be performed with them.<br/>
     * 
     * If this method is invoked without an active transaction context,
     * an IllegalStateException is thrown.<br/>
     * 
     * If this method is invoked after the two-phase commit processing
     * has started, an IllegalStateException is thrown.
     * @param sync the Synchronization instance.
     * @throws IllegalStateException if no transaction is active.
     * @since JTA 1.1
     */
    void registerInterposedSynchronization(Synchronization sync);
    
    /**
     * Return the status of the transaction bound to the current thread
     * at the time this method is called. This is the result of
     * executing TransactionManager.getStatus() in the context of
     * the transaction bound to the current thread at the time this
     * method is called.
     * @return the status of the transaction bound to the current
     *         thread at the time this method is called.
     * @since JTA 1.1
     */
    int getTransactionStatus();
    
    /**
     * Set the rollbackOnly status of the transaction bound to the
     * current thread at the time this method is called.
     * @throws IllegalStateException if no transaction is active.
     * @since JTA 1.1
     */
    void setRollbackOnly();
    
    /**
     * Get the rollbackOnly status of the transaction bound to the
     * current thread at the time this method is called.
     * @return the rollbackOnly status.
     * @throws IllegalStateException - if no transaction is active.
     * @since JTA 1.1
     */
    boolean getRollbackOnly();
}
