/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.s3.internal.crt;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.internal.crt.CopyRequestConversionUtils;
import software.amazon.awssdk.services.s3.internal.crt.UploadPartCopyRequestIterable;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.CopyObjectResponse;
import software.amazon.awssdk.services.s3.model.CopyPartResult;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.UploadPartCopyRequest;
import software.amazon.awssdk.services.s3.model.UploadPartCopyResponse;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;

@SdkInternalApi
public final class CopyObjectHelper {
    private static final Logger log = Logger.loggerFor(S3AsyncClient.class);
    private static final long MAX_UPLOAD_PARTS = 10000L;
    private final S3AsyncClient s3AsyncClient;
    private final long partSizeInBytes;

    public CopyObjectHelper(S3AsyncClient s3AsyncClient, long partSizeInBytes) {
        this.s3AsyncClient = s3AsyncClient;
        this.partSizeInBytes = partSizeInBytes;
    }

    public CompletableFuture<CopyObjectResponse> copyObject(CopyObjectRequest copyObjectRequest) {
        CompletableFuture<CopyObjectResponse> returnFuture = new CompletableFuture<CopyObjectResponse>();
        try {
            CompletableFuture<HeadObjectResponse> headFuture = this.s3AsyncClient.headObject(CopyRequestConversionUtils.toHeadObjectRequest(copyObjectRequest));
            CompletableFutureUtils.forwardExceptionTo(returnFuture, headFuture);
            headFuture.whenComplete((headObjectResponse, throwable) -> {
                if (throwable != null) {
                    CopyObjectHelper.handleException(returnFuture, () -> "Failed to retrieve metadata from the source object", throwable);
                } else {
                    this.doCopyObject(copyObjectRequest, returnFuture, (HeadObjectResponse)((Object)headObjectResponse));
                }
            });
        }
        catch (Throwable throwable2) {
            returnFuture.completeExceptionally(throwable2);
        }
        return returnFuture;
    }

    private void doCopyObject(CopyObjectRequest copyObjectRequest, CompletableFuture<CopyObjectResponse> returnFuture, HeadObjectResponse headObjectResponse) {
        Long contentLength = headObjectResponse.contentLength();
        if (contentLength <= this.partSizeInBytes) {
            log.debug(() -> "Starting the copy as a single copy part request");
            this.copyInOneChunk(copyObjectRequest, returnFuture);
        } else {
            log.debug(() -> "Starting the copy as multipart copy request");
            this.copyInParts(copyObjectRequest, contentLength, returnFuture);
        }
    }

    private void copyInParts(CopyObjectRequest copyObjectRequest, Long contentLength, CompletableFuture<CopyObjectResponse> returnFuture) {
        CreateMultipartUploadRequest request = CopyRequestConversionUtils.toCreateMultipartUploadRequest(copyObjectRequest);
        CompletableFuture<CreateMultipartUploadResponse> createMultipartUploadFuture = this.s3AsyncClient.createMultipartUpload(request);
        CompletableFutureUtils.forwardExceptionTo(returnFuture, createMultipartUploadFuture);
        createMultipartUploadFuture.whenComplete((createMultipartUploadResponse, throwable) -> {
            if (throwable != null) {
                CopyObjectHelper.handleException(returnFuture, () -> "Failed to initiate multipart upload", throwable);
            } else {
                log.debug(() -> "Initiated new multipart upload, uploadId: " + createMultipartUploadResponse.uploadId());
                this.doCopyInParts(copyObjectRequest, contentLength, returnFuture, createMultipartUploadResponse.uploadId());
            }
        });
    }

    private int determinePartCount(long contentLength, long partSize) {
        return (int)Math.ceil((double)contentLength / (double)partSize);
    }

    private void doCopyInParts(CopyObjectRequest copyObjectRequest, Long contentLength, CompletableFuture<CopyObjectResponse> returnFuture, String uploadId) {
        long optimalPartSize = this.calculateOptimalPartSizeForCopy(contentLength);
        int partCount = this.determinePartCount(contentLength, optimalPartSize);
        log.debug(() -> String.format("Starting multipart copy with partCount: %s, optimalPartSize: %s", partCount, optimalPartSize));
        AtomicReferenceArray<CompletedPart> completedParts = new AtomicReferenceArray<CompletedPart>(partCount);
        List<CompletableFuture<CompletedPart>> futures = this.sendUploadPartCopyRequests(copyObjectRequest, contentLength, uploadId, completedParts, optimalPartSize);
        ((CompletableFuture)((CompletableFuture)CompletableFutureUtils.allOfExceptionForwarded((CompletableFuture[])futures.toArray(new CompletableFuture[0])).thenCompose(ignore -> this.completeMultipartUpload(copyObjectRequest, uploadId, completedParts))).handle(this.handleExceptionOrResponse(copyObjectRequest, returnFuture, uploadId))).exceptionally(throwable -> {
            CopyObjectHelper.handleException(returnFuture, () -> "Unexpected exception occurred", throwable);
            return null;
        });
    }

    private BiFunction<CompleteMultipartUploadResponse, Throwable, Void> handleExceptionOrResponse(CopyObjectRequest copyObjectRequest, CompletableFuture<CopyObjectResponse> returnFuture, String uploadId) {
        return (completeMultipartUploadResponse, throwable) -> {
            if (throwable != null) {
                this.cleanUpParts(copyObjectRequest, uploadId);
                CopyObjectHelper.handleException(returnFuture, () -> "Failed to send multipart copy requests.", throwable);
            } else {
                returnFuture.complete(CopyRequestConversionUtils.toCopyObjectResponse(completeMultipartUploadResponse));
            }
            return null;
        };
    }

    private CompletableFuture<CompleteMultipartUploadResponse> completeMultipartUpload(CopyObjectRequest copyObjectRequest, String uploadId, AtomicReferenceArray<CompletedPart> completedParts) {
        log.debug(() -> String.format("Sending completeMultipartUploadRequest, uploadId: %s", uploadId));
        CompletedPart[] parts = (CompletedPart[])IntStream.range(0, completedParts.length()).mapToObj(completedParts::get).toArray(CompletedPart[]::new);
        CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)((Object)CompleteMultipartUploadRequest.builder().bucket(copyObjectRequest.destinationBucket()).key(copyObjectRequest.destinationKey()).uploadId(uploadId).multipartUpload((CompletedMultipartUpload)CompletedMultipartUpload.builder().parts(parts).build()).build());
        return this.s3AsyncClient.completeMultipartUpload(completeMultipartUploadRequest);
    }

    private void cleanUpParts(CopyObjectRequest copyObjectRequest, String uploadId) {
        AbortMultipartUploadRequest abortMultipartUploadRequest = CopyRequestConversionUtils.toAbortMultipartUploadRequest(copyObjectRequest, uploadId);
        this.s3AsyncClient.abortMultipartUpload(abortMultipartUploadRequest).exceptionally(throwable -> {
            log.warn(() -> String.format("Failed to abort previous multipart upload (id: %s). You may need to call S3AsyncClient#abortMultiPartUpload to free all storage consumed by all parts. ", uploadId), throwable);
            return null;
        });
    }

    private static void handleException(CompletableFuture<CopyObjectResponse> returnFuture, Supplier<String> message, Throwable throwable) {
        Throwable cause;
        Throwable throwable2 = cause = throwable instanceof CompletionException ? throwable.getCause() : throwable;
        if (cause instanceof Error) {
            returnFuture.completeExceptionally(cause);
        } else {
            SdkClientException exception = SdkClientException.create((String)message.get(), (Throwable)cause);
            returnFuture.completeExceptionally((Throwable)exception);
        }
    }

    private List<CompletableFuture<CompletedPart>> sendUploadPartCopyRequests(CopyObjectRequest copyObjectRequest, long contentLength, String uploadId, AtomicReferenceArray<CompletedPart> completedParts, long optimalPartSize) {
        ArrayList<CompletableFuture<CompletedPart>> futures = new ArrayList<CompletableFuture<CompletedPart>>();
        UploadPartCopyRequestIterable uploadPartCopyRequests = new UploadPartCopyRequestIterable(uploadId, optimalPartSize, copyObjectRequest, contentLength);
        uploadPartCopyRequests.forEach(uploadPartCopyRequest -> this.sendIndividualUploadPartCopy(uploadId, completedParts, (List<CompletableFuture<CompletedPart>>)futures, (UploadPartCopyRequest)((Object)uploadPartCopyRequest)));
        return futures;
    }

    private void sendIndividualUploadPartCopy(String uploadId, AtomicReferenceArray<CompletedPart> completedParts, List<CompletableFuture<CompletedPart>> futures, UploadPartCopyRequest uploadPartCopyRequest) {
        Integer partNumber = uploadPartCopyRequest.partNumber();
        log.debug(() -> "Sending uploadPartCopyRequest with range: " + uploadPartCopyRequest.copySourceRange() + " uploadId: " + uploadId);
        CompletableFuture<UploadPartCopyResponse> uploadPartCopyFuture = this.s3AsyncClient.uploadPartCopy(uploadPartCopyRequest);
        CompletionStage convertFuture = uploadPartCopyFuture.thenApply(uploadPartCopyResponse -> CopyObjectHelper.convertUploadPartCopyResponse(completedParts, partNumber, uploadPartCopyResponse));
        futures.add((CompletableFuture<CompletedPart>)convertFuture);
        CompletableFutureUtils.forwardExceptionTo((CompletableFuture)convertFuture, uploadPartCopyFuture);
    }

    private static CompletedPart convertUploadPartCopyResponse(AtomicReferenceArray<CompletedPart> completedParts, Integer partNumber, UploadPartCopyResponse uploadPartCopyResponse) {
        CopyPartResult copyPartResult = uploadPartCopyResponse.copyPartResult();
        CompletedPart completedPart = CopyRequestConversionUtils.toCompletedPart(copyPartResult, partNumber);
        completedParts.set(partNumber - 1, completedPart);
        return completedPart;
    }

    private long calculateOptimalPartSizeForCopy(long contentLengthOfSource) {
        double optimalPartSize = (double)contentLengthOfSource / 10000.0;
        optimalPartSize = Math.ceil(optimalPartSize);
        return (long)Math.max(optimalPartSize, (double)this.partSizeInBytes);
    }

    private void copyInOneChunk(CopyObjectRequest copyObjectRequest, CompletableFuture<CopyObjectResponse> returnFuture) {
        CompletableFuture<CopyObjectResponse> copyObjectFuture = this.s3AsyncClient.copyObject(copyObjectRequest);
        CompletableFutureUtils.forwardExceptionTo(returnFuture, copyObjectFuture);
        CompletableFutureUtils.forwardResultTo(copyObjectFuture, returnFuture);
    }
}

