package co.kuali.coeus.s3.conv;

import co.kuali.coeus.s3.api.S3File;
import co.kuali.coeus.s3.api.S3FileMetadata;
import co.kuali.coeus.s3.api.S3FileService;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.utils.IoUtils;

import java.io.IOException;
import java.sql.*;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class ConversionServiceImpl implements ConversionService {

    private static final Logger LOG = LogManager.getLogger(ConversionServiceImpl.class);

    private String url;
    private String user;
    private String password;
    private S3FileService s3FileService;
    private int maxSkip;

    @Override
    public void convert(ConversionInfo info) {
        validateConversionInfo(info);

        final String alterSql = "ALTER TABLE " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + " add (S3_ID VARCHAR(36))";
        final String selectCountSql = "SELECT COALESCE(SUM(unprocessed_cnt), 0) unprocessed_cnt, COALESCE(SUM(processed_cnt), 0) processed_cnt, COALESCE(SUM(null_data_cnt), 0) null_data_cnt FROM (SELECT CASE WHEN S3_ID is null and " + info.getDataColumn() + " is not null THEN 1 ELSE 0 END unprocessed_cnt, CASE WHEN S3_ID is not null THEN 1 ELSE 0 END  processed_cnt, CASE WHEN " + info.getDataColumn() + " is null and S3_ID is null THEN 1 ELSE 0 END  null_data_cnt FROM " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + ") t";
        final String selectRecordSql = "SELECT " + info.getDataColumn() + info.getPkColumns().stream().collect(Collectors.joining(", ", ", ", "")) +
                (StringUtils.isNotBlank(info.getContentTypeColumn()) ? ", " + info.getContentTypeColumn() : "") +
                (StringUtils.isNotBlank(info.getFileNameColumn()) ? ", " + info.getFileNameColumn() : "") +
                (info.getAdditionalMetadataColumns() != null && !info.getAdditionalMetadataColumns().isEmpty() ? info.getAdditionalMetadataColumns().stream().collect(Collectors.joining(", ", ", ", "")) : "") +
                " FROM " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + " WHERE S3_ID is null and " + info.getDataColumn() + " is not null";
        final String updateNullDataSql = "UPDATE " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + " SET S3_ID = ?, " + info.getDataColumn() + " = null " + info.getPkColumns().stream().collect(Collectors.joining(" = ? AND ", "WHERE ", " = ?"));
        final String updateIgnoreDataSql = "UPDATE " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + " SET S3_ID = ? " + info.getPkColumns().stream().collect(Collectors.joining(" = ? AND ", "WHERE ", " = ?"));

        LOG.info("ALTER SQL: " + alterSql);
        LOG.info("SELECT COUNT SQL: " + selectCountSql);
        LOG.info("SELECT RECORD SQL: " + selectRecordSql);
        LOG.info("UPDATE NULL DATA SQL: " + updateNullDataSql);
        LOG.info("UPDATE IGNORE DATA SQL: " + updateIgnoreDataSql);

        try (Connection connection = DriverManager.getConnection(url, user, password);
             ResultSet s3MetadataResult = connection.getMetaData().getColumns(null, StringUtils.isNotBlank(info.getTargetSchema()) ? info.getTargetSchema(): null, info.getTable(), "S3_ID");
             PreparedStatement alter = connection.prepareStatement(alterSql)) {
            connection.setAutoCommit(false);

            if (!s3MetadataResult.next()) {
                LOG.info("Adding S3_ID column to " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable());
                alter.execute();
            }

            try (ResultSet dataMetadataResult = connection.getMetaData().getColumns(null, StringUtils.isNotBlank(info.getTargetSchema()) ? info.getTargetSchema(): null, info.getTable(), info.getDataColumn());
                 PreparedStatement selectCount = connection.prepareStatement(selectCountSql);
                 ResultSet countResult = selectCount.executeQuery();
                 PreparedStatement selectRecord = connection.prepareStatement(selectRecordSql);
                 PreparedStatement updateNullData = connection.prepareStatement(updateNullDataSql);
                 PreparedStatement updateIgnoreData = connection.prepareStatement(updateIgnoreDataSql)) {

                countResult.next();
                final long unprocessedCount = countResult.getLong(1);
                final long alreadyProcessedCount = countResult.getLong(2);
                final long nullDataCount = countResult.getLong(3);
                LOG.info(getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + " has " + unprocessedCount + " records to process and " + alreadyProcessedCount + " records already processed and " + nullDataCount + " records that cannot be processed because they have null data in the " + info.getDataColumn() + " column.");

                dataMetadataResult.next();
                final boolean dataColumnNullable = dataMetadataResult.getInt("NULLABLE") == DatabaseMetaData.attributeNullable;

                if (!dataColumnNullable) {
                    LOG.info(getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + "." + info.getDataColumn() + " is not nullable.  Data column will remain non-null post conversion");
                }

                try (ResultSet dataResult = selectRecord.executeQuery()) {
                    int processedCount = 1;
                    int skipped = 0;
                    while (dataResult.next()) {
                        final Blob data = dataResult.getBlob(1);
                        final List<String> pks = getResultSetValues(info.getPkColumns(), dataResult);

                        final S3File s3File = new S3File();
                        final S3FileMetadata s3FileMetadata = new S3FileMetadata();
                        s3FileMetadata.setContentLength(data.length());

                        String contentType = "";
                        if (StringUtils.isNotBlank(info.getContentTypeColumn())) {
                            contentType = StringUtils.defaultString(dataResult.getString(info.getContentTypeColumn()));
                            s3FileMetadata.setContentType(contentType);
                        }

                        String fileName = "";
                        if (StringUtils.isNotBlank(info.getFileNameColumn())) {
                            fileName = StringUtils.defaultString(dataResult.getString(info.getFileNameColumn()));
                            s3FileMetadata.setFileName(fileName);
                        }

                        List<String> am = Collections.emptyList();
                        if (info.getAdditionalMetadataColumns() != null && !info.getAdditionalMetadataColumns().isEmpty()) {
                            am = getResultSetValues(info.getAdditionalMetadataColumns(), dataResult);
                            CollectionUtils.zipMap(info.getAdditionalMetadataColumns().toArray(new String[0]), am.stream().map(StringUtils::defaultString).toArray(String[]::new)).forEach((k, v) -> s3FileMetadata.putMetadataValue("Conversion-Metadata-" + k, v));
                        }

                        CollectionUtils.zipMap(info.getPkColumns().toArray(new String[0]), pks.toArray(new String[0])).forEach((k, v) -> s3FileMetadata.putMetadataValue("Conversion-PK-" + k, v));
                        s3FileMetadata.putMetadataValue("Conversion-Date", Date.from(Instant.now()));
                        s3FileMetadata.putMetadataValue("Conversion-Target-Schema", StringUtils.defaultString(info.getTargetSchema()));
                        s3FileMetadata.putMetadataValue("Conversion-Table", info.getTable());
                        s3FileMetadata.putMetadataValue("Conversion-Data-Column", info.getDataColumn());
                        s3FileMetadata.putMetadataValue("Conversion-PK-Columns", String.join(",", info.getPkColumns()));
                        s3FileMetadata.putMetadataValue("Conversion-File-Name-Column", StringUtils.defaultString(info.getFileNameColumn()));
                        s3FileMetadata.putMetadataValue("Conversion-Content-Type-Column", StringUtils.defaultString(info.getContentTypeColumn()));
                        s3FileMetadata.putMetadataValue("Conversion-Additional-Metadata-Columns", StringUtils.defaultString(String.join(",", info.getAdditionalMetadataColumns())));

                        s3File.setFileMetaData(s3FileMetadata);
                        s3File.setByteContents(IoUtils.toByteArray(data.getBinaryStream()));

                        final String recordInfo = "Row " + processedCount + " of " + unprocessedCount + " for " + getTargetSchemaPrefix(info.getTargetSchema()) + info.getTable() + "." + info.getDataColumn() +
                                " with primary key(s) " + CollectionUtils.zipMap(info.getPkColumns().toArray(new String[0]), pks.toArray(new String[0])) +
                                (StringUtils.isNotBlank(info.getContentTypeColumn()) ? " with content type " + info.getContentTypeColumn() + "=" + contentType : "") +
                                (StringUtils.isNotBlank(info.getFileNameColumn()) ? " with file name " + info.getFileNameColumn() + "=" + fileName : "") +
                                (info.getAdditionalMetadataColumns() != null && !info.getAdditionalMetadataColumns().isEmpty() ? " with additional metadata " + CollectionUtils.zipMap(info.getAdditionalMetadataColumns().toArray(new String[0]), am.stream().map(StringUtils::defaultString).toArray(String[]::new)) : "");

                        final String s3Id;
                        try {
                            s3Id = s3FileService.createFile(s3File);
                        } catch (SdkException e) {
                            if (skipped <= maxSkip) {
                                skipped++;
                                LOG.error(recordInfo + " has failed to send to S3. Skipping file. Skip number: " + skipped + ". Max skip number " + maxSkip + ".", e);
                                continue;
                            } else {
                                LOG.error(recordInfo + " has failed to send to S3. Halting process. Skip number: " + skipped + ". Max skip number " + maxSkip + ".", e);
                                throw e;
                            }
                        }
                        final PreparedStatement update = (dataColumnNullable && info.isNullConverted()) ? updateNullData : updateIgnoreData;
                        update.setString(1, s3Id);

                        IntStream.rangeClosed(2, info.getPkColumns().size() + 1).forEach(i -> {
                            try {
                                update.setString(i, pks.get(i - 2));
                            } catch (SQLException e) {
                                throw new RuntimeException(e);
                            }
                        });

                        update.executeUpdate();
                        connection.commit();

                        LOG.info(recordInfo + " with s3Id: " + s3Id + " has been processed.");
                        processedCount++;
                    }
                }
            }
        } catch (SQLException| IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<String> getResultSetValues(List<String> columns, ResultSet resultSet) {
        return columns.stream().map(column -> {
            try {
                return resultSet.getString(column);
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }).collect(Collectors.toList());
    }

    private void validateConversionInfo(ConversionInfo info) {
        if (info == null) {
            throw new IllegalArgumentException("info is null");
        }

        if (StringUtils.isBlank(info.getTable())) {
            throw new IllegalArgumentException("table is blank");
        }

        if (StringUtils.isBlank(info.getDataColumn())) {
            throw new IllegalArgumentException("dataColumn is blank");
        }

        if (info.getPkColumns() == null) {
            throw new IllegalArgumentException("pkColumns is null");
        }

        if (info.getPkColumns().size() < 1) {
            throw new IllegalArgumentException("pkColumns must have at least one entry");
        }

        if (info.getPkColumns().stream().anyMatch(StringUtils::isBlank)) {
            throw new IllegalArgumentException("pkColumns has blank entries " + info.getPkColumns());
        }
    }

    private String getTargetSchemaPrefix(String targetSchema) {
        return StringUtils.isNotBlank(targetSchema)  ? targetSchema + "." : "";
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getUser() {
        return user;
    }

    public void setUser(String user) {
        this.user = user;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public S3FileService getS3FileService() {
        return s3FileService;
    }

    public void setS3FileService(S3FileService s3FileService) {
        this.s3FileService = s3FileService;
    }

    public int getMaxSkip() {
        return maxSkip;
    }

    public void setMaxSkip(int maxSkip) {
        this.maxSkip = maxSkip;
    }
}
