package co.kuali.coeus.s3.conv;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collector;
import java.util.stream.Collectors;


public final class CollectionUtils {

    /** private ctor. */
    private CollectionUtils() {
        throw new UnsupportedOperationException("do not call");
    }

    /**
     * Takes an array of keys and an array of values and constructs a map.  Both key and value
     * arrays must be the same length and non-null.
     *
     * @param keys the keys.  Cannot be null
     * @param values the values.  Cannot be null
     * @param <T> the key type
     * @param <U> the value type
     * @return a map.  cannot return null
     * @throws IllegalArgumentException if either argument is null or the arrays aren't the same length
     */
    public static <T, U> Map<T, U> zipMap(T[] keys, U[] values) {
        if (keys == null) {
            throw new IllegalArgumentException("keys is null");
        }

        if (values == null) {
            throw new IllegalArgumentException("values is null");
        }

        if (keys.length != values.length) {
            throw new IllegalArgumentException("Number of keys doesn't match number of values");
        }

        final Map<T, U> map = new HashMap<>();
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return map;
    }

    /**
     * Creates an entry from a key and value.
     * @param key the key. Can be null.
     * @param value the value.  Can be null.
     * @param <K> the key type
     * @param <V> the value type
     * @return a Map.Entry
     */
    public static <K, V> Map.Entry<K, V> entry(K key, V value) {
        return new AbstractMap.SimpleEntry<K, V>(key, value);
    }

    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a Map.
     * @param <K> the key type
     * @param <U> the value type
     * @return A Collector from Map.Entry to Map
     */
    public static <K, U> Collector<Map.Entry<K, U>, ?, Map<K, U>> entriesToMap() {
        return Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a ConcurrentMap.
     * @param <K> the key type
     * @param <U> the value type
     * @return A Collector from Map.Entry to ConcurrentMap
     */
    public static <K, U> Collector<Map.Entry<K, U>, ?, ConcurrentMap<K, U>> entriesToConcurrentMap() {
        return Collectors.toConcurrentMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    /**
     * This method takes a raw Map.Entry and casts it to a generic kind.
     * @param entry the entry to cast
     * @param <K> the key type
     * @param <V> the value type
     * @return the passed in entry casted
     */
    @SuppressWarnings("unchecked")
    public static <K, V> Map.Entry<K, V> unsafeCast(Map.Entry entry) {
        return (Map.Entry<K, V>) entry;
    }
}
