/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.system.globalLibraryV20.*;
import gov.grants.apply.system.universalCodesV20.CountryCodeDataType;
import gov.grants.apply.system.universalCodesV20.CountryCodeDataTypeV3;
import gov.grants.apply.system.universalCodesV20.StateCodeDataType;
import gov.grants.apply.system.universalCodesV20.StateCodeDataTypeV3;
import org.apache.commons.lang3.StringUtils;
import org.kuali.coeus.common.api.address.Addressable;
import org.kuali.coeus.common.api.country.CountryContract;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.state.StateContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.budget.S2SKeyPersonDto;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.location.S2SLocationService;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.Optional;

import static org.kuali.coeus.s2sgen.impl.util.Transformers.truncateStreetAddress;

@FormGenerator("GlobalLibraryV2_0Generator")
public class GlobalLibraryV2_0Generator {

	private static final int PRIMARY_TITLE_MAX_LENGTH = 45;

	@Autowired
    @Qualifier("s2SLocationService")
	private S2SLocationService s2SLocationService;

    @Autowired
    @Qualifier("departmentalPersonService")
    private DepartmentalPersonService departmentalPersonService;

	/**
	 * Create a CountryCodeDataType.Enum as defined in UniversalCodes 2.0 from
	 * the given country code.
	 * 
	 * @param countryCode The country code
	 * @return The CountryCodeDataType type corresponding to the given country
	 *         code
	 */
	public Optional<CountryCodeDataType.Enum> getCountryCodeDataType(String countryCode) {
		CountryContract country = s2SLocationService.getCountryFromCode(countryCode);
		if (country != null && StringUtils.isNotBlank(country.getName())) {
			final String matchValue = String.format("%s: %s", country.getAlternateCode(), country.getName());
			for (int i = 1; i <= CountryCodeDataType.Enum.table.lastInt(); i++) {
				final CountryCodeDataType.Enum countryEnum = CountryCodeDataType.Enum.forInt(i);
				if (StringUtils.equalsIgnoreCase(countryEnum.toString(), matchValue)) {
					return Optional.of(countryEnum);
				}
			}
		}
		return Optional.empty();
	}

	/**
	 * Create a CountryCodeDataTypeV3.Enum as defined in UniversalCodes 2.0 from
	 * the given country code.
	 *
	 * @param countryCode The country code
	 * @return The CountryCodeDataTypeV3 type corresponding to the given country
	 *         code, or an empty Optional if one could not be found
	 */
	public Optional<CountryCodeDataTypeV3.Enum> getCountryCodeDataTypeV3(String countryCode) {
		CountryContract country = s2SLocationService.getCountryFromCode(countryCode);
		if (country != null && StringUtils.isNotBlank(country.getNameV3())) {
			final String matchValue = String.format("%s: %s", country.getAlternateCode(), country.getNameV3());
			for (int i = 1; i <= CountryCodeDataTypeV3.Enum.table.lastInt(); i++) {
				final CountryCodeDataTypeV3.Enum countryEnum = CountryCodeDataTypeV3.Enum.forInt(i);
				if (StringUtils.equalsIgnoreCase(countryEnum.toString(), matchValue)) {
					return Optional.of(countryEnum);
				}
			}
		}
		return Optional.empty();
	}

	private static Optional<CountryCodeDataTypeV3.Enum> getMatchingCountryV3Enum(String code, String name) {
		final String matchValue = String.format("%s: %s", code, name);
		for (int i = 1; i <= CountryCodeDataTypeV3.Enum.table.lastInt(); i++) {
			final CountryCodeDataTypeV3.Enum countryEnum = CountryCodeDataTypeV3.Enum.forInt(i);
			if (StringUtils.equalsIgnoreCase(countryEnum.toString(), matchValue)) {
				return Optional.of(countryEnum);
			}
		}
		return Optional.empty();
	}

	/**
	 * Create a StateCodeDataType.Enum as defined in UniversalCodes 2.0 from the
	 * given name of the state.
	 * 
	 * @param stateCode The state code
	 * @return The StateCodeDataType type corresponding to the given State code,
	 * 		   or an empty Optional if one could not be found.
	 */
	public Optional<StateCodeDataType.Enum> getStateCodeDataType(String countryAlternateCode, String stateCode) {
		StateContract state = s2SLocationService.getStateFromCode(countryAlternateCode, stateCode);
		if (state != null && StringUtils.isNotBlank(state.getName())) {
			final String matchValue = String.format("%s: %s", stateCode, state.getName());
			for (int i = 1; i <= StateCodeDataType.Enum.table.lastInt(); i++) {
				final StateCodeDataType.Enum stateEnum = StateCodeDataType.Enum.forInt(i);
				if (StringUtils.equalsIgnoreCase(stateEnum.toString(), matchValue)) {
					return Optional.of(stateEnum);
				}
			}
		}
		return Optional.empty();
	}

	/**
	 * Create a StateCodeDataTypeV3.Enum as defined in UniversalCodes 2.0 from the
	 * given name of the state.
	 *
	 * @param stateCode The state code
	 * @return The StateCodeDataTypeV3 type corresponding to the given State code,
	 * 		   or an empty Optional if one could not be found.
	 */
	public Optional<StateCodeDataTypeV3.Enum> getStateCodeDataTypeV3(String countryAlternateCode, String stateCode) {
		StateContract state = s2SLocationService.getStateFromCode(countryAlternateCode, stateCode);
		if (state != null && StringUtils.isNotBlank(state.getNameV3())) {
			final String matchValue = String.format("%s: %s", stateCode, state.getNameV3());
			for (int i = 1; i <= StateCodeDataTypeV3.Enum.table.lastInt(); i++) {
				final StateCodeDataTypeV3.Enum stateEnum = StateCodeDataTypeV3.Enum.forInt(i);
				if (StringUtils.equalsIgnoreCase(stateEnum.toString(), matchValue)) {
					return Optional.of(stateEnum);
				}
			}
		}
		return Optional.empty();
	}

	public AddressDataType getAddressDataType(Addressable addressable) {
		final AddressDataType addressDataType = AddressDataType.Factory.newInstance();
		if (addressable != null) {
			final String street1 = addressable.getAddressLine1();
			addressDataType.setStreet1(truncateStreetAddress(street1));
			final String street2 = addressable.getAddressLine2();
			if (StringUtils.isNotBlank(street2)) {
				addressDataType.setStreet2(street2);
			}
			final String city = addressable.getCity();
			addressDataType.setCity(city);
			final String county = addressable.getCounty();
			if (StringUtils.isNotBlank(county)) {
				addressDataType.setCounty(county);
			}

			final String postalCode = addressable.getPostalCode();
			if (StringUtils.isNotBlank(postalCode)) {
				addressDataType.setZipPostalCode(postalCode);
			}
			setAddressEnums(addressDataType, addressable.getCountryCode(), addressable.getState());
		}
		return addressDataType;
	}

	private void setAddressEnums(AddressDataType address, String country, String state) {
		final Optional<CountryCodeDataType.Enum> countryCodeDataType = getCountryCodeDataType(country);
		countryCodeDataType.ifPresent(address::setCountry);

		if (StringUtils.isNotBlank(state)) {
			countryCodeDataType
					.filter(CountryCodeDataType.USA_UNITED_STATES::equals)
					.flatMap(countryEnum -> getStateCodeDataType(country, state))
					.ifPresentOrElse(address::setState, () -> address.setProvince(state));
		}
	}

	public AddressDataTypeV3 getAddressDataTypeV3(Addressable addressable) {
		final AddressDataTypeV3 addressDataType = AddressDataTypeV3.Factory.newInstance();
		if (addressable != null) {
			final String street1 = addressable.getAddressLine1();
			addressDataType.setStreet1(truncateStreetAddress(street1));
			final String street2 = addressable.getAddressLine2();
			if (StringUtils.isNotBlank(street2)) {
				addressDataType.setStreet2(street2);
			}
			final String city = addressable.getCity();
			addressDataType.setCity(city);
			final String county = addressable.getCounty();
			if (StringUtils.isNotBlank(county)) {
				addressDataType.setCounty(county);
			}

			final String postalCode = addressable.getPostalCode();
			if (StringUtils.isNotBlank(postalCode)) {
				addressDataType.setZipPostalCode(postalCode);
			}
			setAddressEnumsV3(addressDataType, addressable.getCountryCode(), addressable.getState());
		}
		return addressDataType;
	}

	private void setAddressEnumsV3(AddressDataTypeV3 address, String country, String state) {
		final Optional<CountryCodeDataTypeV3.Enum> countryCodeDataType = getCountryCodeDataTypeV3(country);
		countryCodeDataType.ifPresent(address::setCountry);

		if (StringUtils.isNotBlank(state)) {
			countryCodeDataType
					.filter(CountryCodeDataTypeV3.USA_UNITED_STATES::equals)
					.flatMap(countryEnum -> getStateCodeDataTypeV3(country, state))
					.ifPresentOrElse(address::setState, () -> address.setProvince(state));
		}
	}

	/**
	 * Create HumanNameDataType from ProposalPerson object
	 * 
	 * @param person
	 *            ProposalPerson
	 * @return HumanNameDataType corresponding to the ProposalPerson object.
	 */
	public HumanNameDataType getHumanNameDataType(ProposalPersonContract person) {
		HumanNameDataType humanName = HumanNameDataType.Factory.newInstance();
		if (person != null) {
			humanName.setFirstName(person.getFirstName());
			humanName.setLastName(person.getLastName());
			String middleName = person.getMiddleName();
			if (middleName != null && !middleName.isEmpty()) {
				humanName.setMiddleName(middleName);
			}
		}
		return humanName;
	}

	/**
	 * Create HumanNameDataType from DepartmentalPerson object
	 * 
	 * @param person
	 *            DepartmentalPerson
	 * @return HumanNameDataType corresponding to the DepartmentalPerson object
	 */
	public HumanNameDataType getHumanNameDataType(DepartmentalPersonDto person) {
		HumanNameDataType humanName = HumanNameDataType.Factory.newInstance();
		if (person != null) {
			humanName.setFirstName(person.getFirstName());
			humanName.setLastName(person.getLastName());
			String middleName = person.getMiddleName();
			if (middleName != null && !middleName.isEmpty()) {
				humanName.setMiddleName(middleName);
			}
		}
		return humanName;
	}


	/**
	 * Create a HumanNameDataType from Rolodex object
	 * 
	 * @param rolodex
	 *            Rolodex object
	 * @return HumanNameDataType corresponding to the rolodex object.
	 */
	public HumanNameDataType getHumanNameDataType(RolodexContract rolodex) {
		HumanNameDataType humanName = HumanNameDataType.Factory.newInstance();
		if (rolodex != null) {
			humanName.setFirstName(rolodex.getFirstName());
			humanName.setLastName(rolodex.getLastName());
			String middleName = rolodex.getMiddleName();
			if (middleName != null && !middleName.isEmpty()) {
				humanName.setMiddleName(middleName);
			}
		}
		return humanName;
	}

	/**
	 * Create HumanNameDataType from KeyPersonInfo object
	 * 
	 * @param keyPerson
	 *            KeyPersonInfo
	 * @return HumanNameDataType corresponding to the KeyPersonInfo object
	 */
	public HumanNameDataType getHumanNameDataType(S2SKeyPersonDto keyPerson) {
		HumanNameDataType humanName = HumanNameDataType.Factory.newInstance();
		if (keyPerson != null) {
			humanName.setFirstName(keyPerson.getFirstName());
			humanName.setLastName(keyPerson.getLastName());
			String middleName = keyPerson.getMiddleName();
			if (middleName != null && !middleName.isEmpty()) {
				humanName.setMiddleName(middleName);
			}
		}
		return humanName;
	}

	/**
	 * Create ContactPersonDataType from ProposalPerson object
	 * 
	 * @param person
	 *            Proposalperson
	 * @return ContactPersonDataType created from ProposalPerson object
	 */
	public ContactPersonDataType getContactPersonDataType(ProposalPersonContract person) {
		ContactPersonDataType contactPerson = ContactPersonDataType.Factory
				.newInstance();
		if (person != null) {
			contactPerson.setName(getHumanNameDataType(person));
			String phone = person.getOfficePhone();
			if (phone != null && !phone.isEmpty()) {
				contactPerson.setPhone(phone);
			}
			String email = person.getEmailAddress();
			if (email != null && !email.isEmpty()) {
				contactPerson.setEmail(email);
			}
			String title = person.getPrimaryTitle();
			if (title != null && !title.isEmpty()) {
				contactPerson.setTitle(title);
			}
			String fax = person.getFaxNumber();
			if (StringUtils.isNotEmpty(fax)) {
				contactPerson.setFax(fax);
			}
			contactPerson.setAddress(getAddressDataType(person));
		}
		return contactPerson;
	}

	public ContactPersonDataTypeV3 getContactPersonDataTypeV3(ProposalPersonContract person) {
		ContactPersonDataTypeV3 contactPerson = ContactPersonDataTypeV3.Factory.newInstance();
		if (person != null) {
			contactPerson.setName(getHumanNameDataType(person));
			String phone = person.getOfficePhone();
			if (phone != null && !phone.isEmpty()) {
				contactPerson.setPhone(phone);
			}
			String email = person.getEmailAddress();
			if (email != null && !email.isEmpty()) {
				contactPerson.setEmail(email);
			}
			String title = person.getPrimaryTitle();
			if (title != null && !title.isEmpty()) {
				contactPerson.setTitle(title);
			}
			String fax = person.getFaxNumber();
			if (StringUtils.isNotEmpty(fax)) {
				contactPerson.setFax(fax);
			}
			contactPerson.setAddress(getAddressDataTypeV3(person));
		}
		return contactPerson;
	}


    public ContactPersonDataType getContactPersonDataType(ProposalDevelopmentDocumentContract proposalDocument) {
		DepartmentalPersonDto person = departmentalPersonService.getContactPerson(proposalDocument);
		return getContactPersonDataType(person);
	}

	public ContactPersonDataType getContactPersonDataType(DepartmentalPersonDto person) {
		ContactPersonDataType contactPerson = ContactPersonDataType.Factory.newInstance();
        if (person != null) {
            contactPerson.setName(getHumanNameDataType(person));
            String phone = person.getOfficePhone();
            if (phone != null && !phone.isEmpty()) {
                contactPerson.setPhone(phone);
            }
            String fax = person.getFaxNumber();
            if (StringUtils.isNotEmpty(fax)) {
                contactPerson.setFax(fax);
            }
            String email = person.getEmailAddress();
            if (email != null && !email.isEmpty()) {
                contactPerson.setEmail(person.getEmailAddress());
            }
            String title = person.getPrimaryTitle();
            if (title != null && !title.isEmpty()) {
                contactPerson.setTitle(StringUtils.substring(title, 0, PRIMARY_TITLE_MAX_LENGTH));
            }
            contactPerson.setAddress(getAddressDataType(person));

        }
        return contactPerson;
    }

	public ContactPersonDataTypeV3 getContactPersonDataTypeV3(ProposalDevelopmentDocumentContract proposalDocument) {
		DepartmentalPersonDto person = departmentalPersonService.getContactPerson(proposalDocument);
		return getContactPersonDataTypeV3(person);
	}

	public ContactPersonDataTypeV3 getContactPersonDataTypeV3(DepartmentalPersonDto person) {
		ContactPersonDataTypeV3 contactPerson = ContactPersonDataTypeV3.Factory.newInstance();
		if (person != null) {
			contactPerson.setName(getHumanNameDataType(person));
			String phone = person.getOfficePhone();
			if (phone != null && !phone.isEmpty()) {
				contactPerson.setPhone(phone);
			}
			String fax = person.getFaxNumber();
			if (StringUtils.isNotEmpty(fax)) {
				contactPerson.setFax(fax);
			}
			String email = person.getEmailAddress();
			if (email != null && !email.isEmpty()) {
				contactPerson.setEmail(person.getEmailAddress());
			}
			String title = person.getPrimaryTitle();
			if (title != null && !title.isEmpty()) {
				contactPerson.setTitle(StringUtils.substring(title, 0, PRIMARY_TITLE_MAX_LENGTH));
			}
			contactPerson.setAddress(getAddressDataTypeV3(person));

		}
		return contactPerson;
	}

    public S2SLocationService getS2SLocationService() {
        return s2SLocationService;
    }

    public void setS2SLocationService(S2SLocationService s2SLocationService) {
        this.s2SLocationService = s2SLocationService;
    }

    public DepartmentalPersonService getDepartmentalPersonService() {
        return departmentalPersonService;
    }

    public void setDepartmentalPersonService(DepartmentalPersonService departmentalPersonService) {
        this.departmentalPersonService = departmentalPersonService;
    }
}
