/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.nsfCoverPage20V20.NSFCoverPage20Document;
import gov.grants.apply.forms.nsfCoverPage20V20.NSFCoverPage20Document.NSFCoverPage20;
import gov.grants.apply.forms.nsfCoverPage20V20.NSFCoverPage20Document.NSFCoverPage20.OtherInfo;
import gov.grants.apply.forms.nsfCoverPage20V20.NSFCoverPage20Document.NSFCoverPage20.PIInfo;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin1Max100DataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("NSFCoverPageV2_0Generator")
public class NSFCoverPageV2_0Generator extends NSFCoverPageBaseGenerator<NSFCoverPage20Document> implements S2SFormGeneratorPdfFillable<NSFCoverPage20Document> {

    private static final int MENTORING_PLAN = 147;
    private static final int DATA_MANAGEMENT_PLAN = 146;
    private static final int LOBBYING_ACTIVITIES_QUESTION = 11;
    private static final int FUNDING_INT_BRANCH_CAMPUS_QUESTION = 12;
    private static final int FUNDING_FOREIGN_ORG_QUESTION = 13;
    private static final int GOALI = -1;
    private static final int RAISE = -2;

    @Value("http://apply.grants.gov/forms/NSF_CoverPage_2_0-V2.0")
    private String namespace;

    @Value("NSF_CoverPage_2_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/NSF_CoverPage-V2.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/NSF_CoverPage_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("departmentalPersonService")
    protected DepartmentalPersonService departmentalPersonService;

    private NSFCoverPage20Document getNSFCoverPage20() {
        NSFCoverPage20Document nsfCoverPage20Document = NSFCoverPage20Document.Factory.newInstance();
        NSFCoverPage20 nsfCoverPage20 = NSFCoverPage20.Factory.newInstance();
        nsfCoverPage20.setFormVersion(FormVersion.v2_0.getVersion());
        setFundingOpportunityNumber(nsfCoverPage20);

        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getClosingDate() != null) {
            nsfCoverPage20.setDueDate(pdDoc.getDevelopmentProposal().getS2sOpportunity().getClosingDate());
        }
        nsfCoverPage20.setNSFUnitConsideration(getNSFUnitConsideration());
        setOtherInfo(nsfCoverPage20);
        setNsfId(nsfCoverPage20);
        AttachmentGroupMin1Max100DataType attachmentGroup = AttachmentGroupMin1Max100DataType.Factory.newInstance();
        attachmentGroup.setAttachedFileArray(getAttachedFileDataTypes());
        if (attachmentGroup.getAttachedFileArray().length > 0) {
            nsfCoverPage20.setSingleCopyDocuments(attachmentGroup);
        }
        final AttachedFileDataType dataManagementPlan = getAttachedNarrativeFile(DATA_MANAGEMENT_PLAN);
        if (dataManagementPlan != null) {
            nsfCoverPage20.setDataManagementPlan(dataManagementPlan);
        }

        final AttachedFileDataType mentoringPlan = getAttachedNarrativeFile(MENTORING_PLAN);
        if (mentoringPlan != null) {
            nsfCoverPage20.setMentoringPlan(mentoringPlan);
        }

        final AttachedFileDataType goali = getAttachedNarrativeFile(GOALI);
        if (goali != null) {
            nsfCoverPage20.setGOALILetter(goali);
        }

        final AttachedFileDataType raise = getAttachedNarrativeFile(RAISE);
        if (raise != null) {
            nsfCoverPage20.setRAISEEmail(raise);
        }
        nsfCoverPage20Document.setNSFCoverPage20(nsfCoverPage20);
        return nsfCoverPage20Document;
    }

    private void setNsfId(NSFCoverPage20 nsfCoverPage20) {
        DepartmentalPersonDto person = departmentalPersonService.getDepartmentalPerson(pdDoc);
        nsfCoverPage20.setNSFID(person.getNsfId());
    }

    protected void setFundingMechanism(NSFCoverPage20 nsfCoverPage20, String answer) {
        NSFCoverPage20.FundingMechanism.Enum fundingMechanism;
        if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.CONFERENCE.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.CONFERENCE;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.RAPID.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.RAPID;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.EAGER.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.EAGER;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.EQUIPMENT.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.EQUIPMENT;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.RAISE.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.RAISE;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.FELLOWSHIP.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.FELLOWSHIP;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.IDEAS_LAB.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.IDEAS_LAB;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.GOALI.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.GOALI;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.FASED.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.FASED;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.TRAVEL.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.TRAVEL;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.CENTER.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.CENTER;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.RESEARCH_INFRASTRUCTURE.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.RESEARCH_INFRASTRUCTURE;
        } else if (StringUtils.equalsIgnoreCase(answer, NSFCoverPage20.FundingMechanism.RESEARCH.toString())) {
            fundingMechanism = NSFCoverPage20.FundingMechanism.RESEARCH;
        }
        else {
            fundingMechanism = null;
        }
        nsfCoverPage20.setFundingMechanism(fundingMechanism);
    }

    private void setFundingOpportunityNumber(NSFCoverPage20 nsfCoverPage20) {
        nsfCoverPage20.setFundingOpportunityNumber(StringUtils.substring(pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber(), 0, PROGRAM_ANNOUNCEMENT_NUMBER_MAX_LENGTH));
    }

    private void setOtherInfo(NSFCoverPage20 nsfCoverPage20) {
        // Default funding mechanism to research to avoid triggering NSFID validation
        nsfCoverPage20.setFundingMechanism(NSFCoverPage20.FundingMechanism.RESEARCH);

        OtherInfo otherInfo = OtherInfo.Factory.newInstance();
        PIInfo pInfo = PIInfo.Factory.newInstance();
        final List<? extends AnswerContract> questionnaireAnswers = getPropDevQuestionAnswerService().getQuestionnaireAnswers(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName());
        int unansweredQuestions = questionnaireAnswers.size();

        for (AnswerContract questionnaireAnswer : questionnaireAnswers) {
            String answer = questionnaireAnswer.getAnswer();
            int questionId = questionnaireAnswer.getQuestionNumber();
            if (answer != null) {
                switch (questionId) {
                    case QUESTION_CURRENT_PI:
                        pInfo.setIsCurrentPI(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    case QUESTION_BEGIN_INVESTIGATOR:
                        otherInfo.setIsBeginInvestigator(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    case QUESTION_ACCOMPLISHMENT_RENEWAL:
                        otherInfo.setIsAccomplishmentRenewal(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    case FUNDING_MECHANISM_QUESTION:
                        setFundingMechanism(nsfCoverPage20, answer);
                        unansweredQuestions--;
                        break;
                    case LOBBYING_ACTIVITIES_QUESTION:
                        otherInfo.setIsDisclosureLobbyingActivities(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    case FUNDING_INT_BRANCH_CAMPUS_QUESTION:
                        otherInfo.setIsFundingInternationalBranch(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    case FUNDING_FOREIGN_ORG_QUESTION:
                        otherInfo.setIsFundingForeignOrganization(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        unansweredQuestions--;
                        break;
                    default:
                        break;
                }
            }
        }
        // Null OtherInfo signifies an incomplete questionnaire and should fire an audit error to that effect
        nsfCoverPage20.setOtherInfo(unansweredQuestions == 0 ? otherInfo : null);
        nsfCoverPage20.setPIInfo(pInfo);
    }

    @Override
    public NSFCoverPage20Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getNSFCoverPage20();
    }

    protected NSFCoverPage20.NSFUnitConsideration getNSFUnitConsideration() {
        NSFCoverPage20.NSFUnitConsideration nsfConsideration = NSFCoverPage20.NSFUnitConsideration.Factory.newInstance();
        nsfConsideration.setDivisionCode(pdDoc.getDevelopmentProposal().getAgencyDivisionCode());
        nsfConsideration.setProgramCode(pdDoc.getDevelopmentProposal().getAgencyProgramCode());
        return nsfConsideration;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(NSFCoverPage20Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getNSFCoverPage20().getSingleCopyDocuments() != null && form.getNSFCoverPage20().getSingleCopyDocuments().getAttachedFileList() != null) {
                final List<AttachedFileDataType> attachedFileList = form.getNSFCoverPage20().getSingleCopyDocuments().getAttachedFileList();
                for (int i = 0; i < attachedFileList.size(); i++) {
                    final AttachedFileDataType attachedFile = attachedFileList.get(i);
                    if (a.getContentId().equals(attachedFile.getFileLocation().getHref())){
                        return entry("NSF_CoverPage_2_0_P1.optionalFile1_" + i, a);
                    }
                }
            }

            final AttachedFileDataType dataManagementPlanAttachedFile = form.getNSFCoverPage20().getDataManagementPlan();
            if (dataManagementPlanAttachedFile != null && a.getContentId().equals(dataManagementPlanAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_2_0_P1.optionalFile2", a);
            }

            final AttachedFileDataType mentoringPlanAttachedFile = form.getNSFCoverPage20().getMentoringPlan();
            if (mentoringPlanAttachedFile != null && a.getContentId().equals(mentoringPlanAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_2_0_P1.optionalFile3", a);
            }

            final AttachedFileDataType goaliLetterAttachedFile = form.getNSFCoverPage20().getGOALILetter();
            if (goaliLetterAttachedFile != null && a.getContentId().equals(goaliLetterAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_2_0_P1.optionalFile4", a);
            }

            final AttachedFileDataType raiseAttachedFile = form.getNSFCoverPage20().getRAISEEmail();
            if (raiseAttachedFile != null && a.getContentId().equals(raiseAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_2_0_P1.optionalFile5", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<NSFCoverPage20Document> factory() {
        return NSFCoverPage20Document.Factory;
    }
}
