/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;


import gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument;
import gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons.Compensation;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.*;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.*;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.Equipment.EquipmentList;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.OtherDirectCosts.Others;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.OtherPersonnel.GraduateStudents;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.OtherPersonnel.PostDocAssociates;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.OtherPersonnel.SecretarialClerical;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.OtherPersonnel.UndergraduateStudents;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.ParticipantTraineeSupportCosts.Other;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget.BudgetSummary;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget.BudgetSummary.CumulativeEquipments;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget.BudgetSummary.CumulativeOtherDirect;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget.BudgetSummary.CumulativeTrainee;
import gov.grants.apply.forms.rrFedNonFedBudgetV11.RRFedNonFedBudgetDocument.RRFedNonFedBudget.BudgetSummary.CumulativeTravels;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.budget.*;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.budget.BudgetPeriodNum;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.print.GenericPrintable;
import org.kuali.coeus.s2sgen.impl.print.S2SPrintingService;
import org.kuali.coeus.s2sgen.impl.util.UnsupportedOtherCostsErrorBuilder;
import org.kuali.coeus.sys.api.model.KcFile;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;


/**
 * 
 * This class is used to generate XML Document object for grants.gov RRFedNonFedBudgetV1.1. This form is generated using XMLBean
 * API's generated by compiling RRFedNonFedBudgetV1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("RRFedNonFedBudgetV1_1Generator")
public class RRFedNonFedBudgetV1_1Generator extends RRFedNonFedBudgetBaseGenerator<RRFedNonFedBudgetDocument> {

    private static final Logger LOG = LogManager.getLogger(RRFedNonFedBudgetV1_1Generator.class);
    private static final String EXTRA_KEYPERSONS = "RRFEDNONFED_EXTRA_KEYPERSONS";
    private static final int EXTRA_KEYPERSONS_TYPE = 11;
    private static final String ADDITIONAL_EQUIPMENT_NARRATIVE_TYPE_CODE ="12";
    private static final int SUPPORTED_OTHER_COSTS = 3;
    private BudgetContract budget;

    @Value("http://apply.grants.gov/forms/RR_FedNonFedBudget-V1.1")
    private String namespace;

    @Value("RR_FedNonFedBudget-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_FedNonFedBudget-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_FedNonFedBudget-V1.1.pdf")
    private Resource pdfForm;

    @Value("169")
    private int sortIndex;

    @Autowired
    @Qualifier("s2SPrintingService")
    private S2SPrintingService s2SPrintingService;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/AdditionalEquipmentAttachmentNonFed.xsl")
    private  Resource additionalEquipmentAttachmentNonFedStyleSheet;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/ExtraKeyPersonAttachmentNonFed.xsl")
    private Resource extraKeyPersonAttachmentNonFedStyleSheet;

    /**
     * This method returns RRFedNonFedBudgetDocument object based on proposal development document which contains the informations
     * such as DUNSID,OrganizationName,BudgetType,BudgetYear and BudgetSummary.
     * 
     * @return rrFedNonFedBudgetDocument {@link XmlObject} of type RRFedNonFedBudgetDocument.
     */
    private RRFedNonFedBudgetDocument getRRFedNonFedBudget() {
        RRFedNonFedBudgetDocument rrFedNonFedBudgetDocument = RRFedNonFedBudgetDocument.Factory.newInstance();
        RRFedNonFedBudget rrFedNonFedBudget = RRFedNonFedBudget.Factory.newInstance();
        rrFedNonFedBudget.setFormVersion(FormVersion.v1_1.getVersion());
        if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
            rrFedNonFedBudget.setDUNSID(pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization().getDunsNumber());
        }
        if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
            rrFedNonFedBudget.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
        }
        rrFedNonFedBudget.setBudgetType(BudgetTypeDataType.PROJECT);
        // Set default values for mandatory fields


        validateBudgetForForm(pdDoc);
        final S2SBudgetDto budgetSummary = s2sBudgetInfoService.getBudgetInfo(pdDoc);
        final List<S2SBudgetPeriodDto> budgetPeriodList = budgetSummary.getBudgetPeriods();
        budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());

        for (S2SBudgetPeriodDto budgetPeriodData : budgetPeriodList) {
            if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
                rrFedNonFedBudget.setBudgetYear1(getBudgetYear1DataType(budgetPeriodData));
            }
            else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P2.getNum()) {
                rrFedNonFedBudget.setBudgetYear2(getBudgetYearDataType(budgetPeriodData));
            }
            else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P3.getNum()) {
                rrFedNonFedBudget.setBudgetYear3(getBudgetYearDataType(budgetPeriodData));
            }
            else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P4.getNum()) {
                rrFedNonFedBudget.setBudgetYear4(getBudgetYearDataType(budgetPeriodData));
            }
            else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P5.getNum()) {
                rrFedNonFedBudget.setBudgetYear5(getBudgetYearDataType(budgetPeriodData));
            }
        }
        for (S2SBudgetPeriodDto budgetPeriodData : budgetPeriodList) {
            if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
                rrFedNonFedBudget.setBudgetYear1(getBudgetJustificationAttachment(rrFedNonFedBudget.getBudgetYear1()));
            }
        }
        rrFedNonFedBudget.setBudgetSummary(getBudgetSummary(budgetSummary));
        rrFedNonFedBudgetDocument.setRRFedNonFedBudget(rrFedNonFedBudget);
        return rrFedNonFedBudgetDocument;
    }

    /**
     * This method gets BudgetYearDataType details like BudgetPeriodStartDate,BudgetPeriodEndDate,BudgetPeriod
     * KeyPersons,OtherPersonnel,TotalCompensation,Equipment,ParticipantTraineeSupportCosts,Travel,OtherDirectCosts
     * DirectCosts,IndirectCosts,CognizantFederalAgency,TotalCosts based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return BudgetYearDataType corresponding to the BudgetSummaryInfo object.
     */
    private BudgetYearDataType getBudgetYearDataType(S2SBudgetPeriodDto periodInfo) {

        ScaleTwoDecimal totalDirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalIndirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal directCostsTotal = ScaleTwoDecimal.ZERO;
        BudgetYearDataType budgetYear = BudgetYearDataType.Factory.newInstance();
        if (periodInfo != null) {
            budgetYear.setBudgetPeriodStartDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getStartDate()));
            budgetYear.setBudgetPeriodEndDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getEndDate()));
            gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.BudgetPeriod.Enum budgetPeriodEnum = gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.BudgetPeriod.Enum
                    .forInt(periodInfo.getBudgetPeriod());
            budgetYear.setBudgetPeriod(budgetPeriodEnum);
            budgetYear.setKeyPersons(getKeyPersons(periodInfo));
            budgetYear.setOtherPersonnel(getOtherPersonnel(periodInfo));
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalCompensation() != null) {
                summary.setFederalSummary(periodInfo.getTotalCompensation().bigDecimalValue());
            }
            if (periodInfo.getTotalCompensationCostSharing() != null) {
                if (budget.getSubmitCostSharingFlag()) {
                    summary.setNonFederalSummary(periodInfo.getTotalCompensationCostSharing().bigDecimalValue());
                    if (periodInfo.getTotalCompensation() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensation().add(
                                periodInfo.getTotalCompensationCostSharing()).bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensationCostSharing().bigDecimalValue());
                    }
                } else {
                    summary.setNonFederalSummary(BigDecimal.ZERO);
                    if (periodInfo.getTotalCompensation() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensation().bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(BigDecimal.ZERO);
                    }
                }
            }
            budgetYear.setTotalCompensation(summary);
            budgetYear.setEquipment(getEquipment(periodInfo));
            budgetYear.setTravel(getTravel(periodInfo));
            budgetYear.setParticipantTraineeSupportCosts(getParticipantTraineeSupportCosts(periodInfo));
            budgetYear.setOtherDirectCosts(getOtherDirectCosts(periodInfo));
            SummaryDataType summaryDirect = SummaryDataType.Factory.newInstance();
            if (periodInfo.getDirectCostsTotal() != null) {
                directCostsTotal = periodInfo.getDirectCostsTotal();
                summaryDirect.setFederalSummary( directCostsTotal.bigDecimalValue());
            }
            
                if (periodInfo.getTotalDirectCostSharing() != null) {
                totalDirectCostSharing = periodInfo.getTotalDirectCostSharing();
                }
                summaryDirect.setNonFederalSummary(totalDirectCostSharing.bigDecimalValue());
                if (directCostsTotal != null) {
                    summaryDirect.setTotalFedNonFedSummary(directCostsTotal.add(totalDirectCostSharing).bigDecimalValue());
                }
                else {
                    summaryDirect.setTotalFedNonFedSummary(totalDirectCostSharing.bigDecimalValue());
                }
            
            budgetYear.setDirectCosts(summaryDirect);
            IndirectCosts indirectCosts = getIndirectCosts(periodInfo);
            if (indirectCosts != null) {
                budgetYear.setIndirectCosts(indirectCosts);
            }
            budgetYear.setCognizantFederalAgency(periodInfo.getCognizantFedAgency());
            if (periodInfo.getIndirectCosts().getTotalIndirectCostSharing() != null){
            totalIndirectCostSharing = periodInfo.getIndirectCosts().getTotalIndirectCostSharing();
            }
            SummaryDataType summaryTotal = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalCosts() != null) {
                summaryTotal.setFederalSummary(periodInfo.getTotalCosts().bigDecimalValue());
            }
            summaryTotal.setNonFederalSummary(totalDirectCostSharing.bigDecimalValue().add(totalIndirectCostSharing.bigDecimalValue()));
             if (periodInfo.getTotalCosts() != null) {
                 summaryTotal.setTotalFedNonFedSummary(periodInfo.getTotalCosts().add(totalDirectCostSharing).bigDecimalValue().add(totalIndirectCostSharing.bigDecimalValue()));
             }
             else {
                 summaryTotal.setTotalFedNonFedSummary(totalDirectCostSharing.bigDecimalValue().add(totalIndirectCostSharing.bigDecimalValue()));
             }
            budgetYear.setTotalCosts(summaryTotal);
        }
        return budgetYear;
    }

    /**
     * This method gets BudgetSummary details such as
     * CumulativeTotalFundsRequestedSeniorKeyPerson,CumulativeTotalFundsRequestedOtherPersonnel
     * CumulativeTotalNoOtherPersonnel,CumulativeTotalFundsRequestedPersonnel,CumulativeEquipments,CumulativeTravels
     * CumulativeTrainee,CumulativeOtherDirect,CumulativeTotalFundsRequestedDirectCosts,CumulativeTotalFundsRequestedIndirectCost
     * CumulativeTotalFundsRequestedDirectIndirectCosts and CumulativeFee based on BudgetSummaryInfo for the RRFedNonFedBudget.
     * 
     * @param budgetSummaryData (BudgetSummaryInfo) budget summary entry.
     * @return BudgetSummary corresponding to the BudgetSummaryInfo object.
     */
    private BudgetSummary getBudgetSummary(S2SBudgetDto budgetSummaryData) {

        ScaleTwoDecimal cumTotalDirectCostSharing = ScaleTwoDecimal.ZERO;
        BudgetSummary budgetSummary = BudgetSummary.Factory.newInstance();
        SummaryDataType summarySeniorKey = SummaryDataType.Factory.newInstance();
        SummaryDataType summaryPersonnel = SummaryDataType.Factory.newInstance();
        SummaryDataType directCosts = SummaryDataType.Factory.newInstance();
        SummaryDataType directIndirectCosts = SummaryDataType.Factory.newInstance();
        if (budgetSummaryData != null) {
            if (budgetSummaryData.getCumTotalFundsForSrPersonnel() != null) {
                summarySeniorKey.setFederalSummary(budgetSummaryData.getCumTotalFundsForSrPersonnel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumTotalNonFundsForSrPersonnel() != null) {
                summarySeniorKey.setNonFederalSummary(budgetSummaryData.getCumTotalNonFundsForSrPersonnel().bigDecimalValue());

                if (budgetSummaryData.getCumTotalFundsForSrPersonnel() != null) {
                    summarySeniorKey.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalFundsForSrPersonnel().add(
                            budgetSummaryData.getCumTotalNonFundsForSrPersonnel()).bigDecimalValue());
                }
                else {
                    summarySeniorKey.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalNonFundsForSrPersonnel()
                            .bigDecimalValue());
                }

            }
            SummaryDataType summaryOtherPersonnel = SummaryDataType.Factory.newInstance();
            if (budgetSummaryData.getCumTotalFundsForOtherPersonnel() != null) {
                summaryOtherPersonnel.setFederalSummary(budgetSummaryData.getCumTotalFundsForOtherPersonnel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumTotalNonFundsForOtherPersonnel() != null) {
                summaryOtherPersonnel.setNonFederalSummary(budgetSummaryData.getCumTotalNonFundsForOtherPersonnel()
                        .bigDecimalValue());


                if (budgetSummaryData.getCumTotalFundsForOtherPersonnel() != null) {
                    summaryOtherPersonnel.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalFundsForOtherPersonnel().add(
                            budgetSummaryData.getCumTotalNonFundsForOtherPersonnel()).bigDecimalValue());
                }
                else {
                    summaryOtherPersonnel.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalNonFundsForOtherPersonnel()
                            .bigDecimalValue());
                }
            }
            budgetSummary.setCumulativeTotalFundsRequestedOtherPersonnel(summaryOtherPersonnel);
            if (budgetSummaryData.getCumNumOtherPersonnel() != null) {
                budgetSummary.setCumulativeTotalNoOtherPersonnel(budgetSummaryData.getCumNumOtherPersonnel().intValue());
            }
            if (budgetSummaryData.getCumTotalFundsForPersonnel() != null) {
                summaryPersonnel.setFederalSummary(budgetSummaryData.getCumTotalFundsForPersonnel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumTotalNonFundsForPersonnel() != null) {
                summaryPersonnel.setNonFederalSummary(budgetSummaryData.getCumTotalNonFundsForPersonnel().bigDecimalValue());
                if (budgetSummaryData.getCumTotalFundsForPersonnel() != null) {
                    summaryPersonnel.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalFundsForPersonnel().add(
                            budgetSummaryData.getCumTotalNonFundsForPersonnel()).bigDecimalValue());
                }
                else {
                    summaryPersonnel
                            .setTotalFedNonFedSummary(budgetSummaryData.getCumTotalNonFundsForPersonnel().bigDecimalValue());
                }
            }
            budgetSummary.setCumulativeEquipments(getCumulativeEquipments(budgetSummaryData));
            budgetSummary.setCumulativeTravels(getCumulativeTravels(budgetSummaryData));
            budgetSummary.setCumulativeTrainee(getCumulativeTrainee(budgetSummaryData));
            budgetSummary.setCumulativeOtherDirect(getCumulativeOtherDirect(budgetSummaryData));
            if (budgetSummaryData.getCumTotalDirectCosts() != null && budgetSummaryData.getCumTotalDirectCostSharing() != null) {
                if (budgetSummaryData.getCumFee() != null) {
                    budgetSummary.setCumulativeFee(budgetSummaryData.getCumFee().bigDecimalValue());
                }
                
                cumTotalDirectCostSharing = budgetSummaryData.getCumTotalNonFundsForPersonnel().add(budgetSummaryData.getCumEquipmentNonFunds()).add(budgetSummaryData.getCumTravelNonFund()).add(budgetSummaryData.getPartOtherCostSharing().add(
                                                        budgetSummaryData.getPartStipendCostSharing().add(budgetSummaryData.getPartTravelCostSharing().add(
                                                                        budgetSummaryData.getPartSubsistenceCostSharing().add( budgetSummaryData.getPartTuitionCostSharing())))));
                for (S2SOtherDirectCostInfoDto cumOtherDirect : budgetSummaryData.getOtherDirectCosts()) {
                      if (cumOtherDirect.getTotalOtherDirectCostSharing() != null) {
                          cumTotalDirectCostSharing= cumTotalDirectCostSharing.add(cumOtherDirect.getTotalOtherDirectCostSharing());
                      }
                }      
                directCosts.setFederalSummary(budgetSummaryData.getCumTotalDirectCosts().bigDecimalValue());
                directCosts.setNonFederalSummary(cumTotalDirectCostSharing.bigDecimalValue());
                directCosts.setTotalFedNonFedSummary((budgetSummaryData.getCumTotalDirectCosts().bigDecimalValue()).add(
                        cumTotalDirectCostSharing.bigDecimalValue()));
            }
            if (budgetSummaryData.getCumTotalIndirectCosts() != null && budgetSummaryData.getCumTotalIndirectCostSharing() != null) {
                SummaryDataType summary = SummaryDataType.Factory.newInstance();
                summary.setFederalSummary(budgetSummaryData.getCumTotalIndirectCosts().bigDecimalValue());
                summary.setNonFederalSummary(budgetSummaryData.getCumTotalIndirectCostSharing().bigDecimalValue());
                summary.setTotalFedNonFedSummary(budgetSummaryData.getCumTotalIndirectCosts().add(
                        budgetSummaryData.getCumTotalIndirectCostSharing()).bigDecimalValue());
                budgetSummary.setCumulativeTotalFundsRequestedIndirectCost(summary);
            }
            if (budgetSummaryData.getCumTotalCosts() != null && budgetSummaryData.getCumTotalDirectCostSharing() != null) {
                directIndirectCosts.setFederalSummary(budgetSummaryData.getCumTotalCosts().bigDecimalValue());                
                directIndirectCosts.setNonFederalSummary((cumTotalDirectCostSharing.bigDecimalValue()).add(budgetSummaryData.getCumTotalIndirectCostSharing().bigDecimalValue()));
                directIndirectCosts.setTotalFedNonFedSummary((budgetSummaryData.getCumTotalCosts().bigDecimalValue()).add
                        (cumTotalDirectCostSharing.bigDecimalValue()).add(budgetSummaryData.getCumTotalIndirectCostSharing().bigDecimalValue()));
            }
        }
        budgetSummary.setCumulativeTotalFundsRequestedSeniorKeyPerson(summarySeniorKey);
        budgetSummary.setCumulativeTotalFundsRequestedPersonnel(summaryPersonnel);
        budgetSummary.setCumulativeTotalFundsRequestedDirectCosts(directCosts);
        budgetSummary.setCumulativeTotalFundsRequestedDirectIndirectCosts(directIndirectCosts);
        return budgetSummary;
    }

    /**
     * This method gets CumulativeOtherDirectCost details,CumulativeMaterialAndSupplies,CumulativePublicationCosts,
     * CumulativeConsultantServices,CumulativeADPComputerServices,CumulativeSubawardConsortiumContractualCosts
     * CumulativeEquipmentFacilityRentalFees,CumulativeAlterationsAndRenovations and CumulativeOther1DirectCost based on
     * BudgetSummaryInfo for the RRFedNonFedBudget.
     * 
     * @param budgetSummaryData (BudgetSummaryInfo) budget summary entry.
     * @return CumulativeOtherDirect corresponding to the BudgetSummaryInfo object.
     */
    private CumulativeOtherDirect getCumulativeOtherDirect(S2SBudgetDto budgetSummaryData) {
        CumulativeOtherDirect cumulativeOtherDirect = CumulativeOtherDirect.Factory.newInstance();
        SummaryDataType summary = SummaryDataType.Factory.newInstance();
        if (budgetSummaryData != null) {
            for (S2SOtherDirectCostInfoDto cumOtherDirect : budgetSummaryData.getOtherDirectCosts()) {
                if (cumOtherDirect.getTotalOtherDirect() != null) {
                    summary.setFederalSummary(cumOtherDirect.getTotalOtherDirect().bigDecimalValue());
                }
                if (cumOtherDirect.getTotalOtherDirectCostSharing() != null) {
                    summary.setNonFederalSummary(cumOtherDirect.getTotalOtherDirectCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getTotalOtherDirect() != null) {
                        summary.setTotalFedNonFedSummary(cumOtherDirect.getTotalOtherDirect().add(
                                cumOtherDirect.getTotalOtherDirectCostSharing()).bigDecimalValue());
                    }
                    else {
                        summary.setTotalFedNonFedSummary(cumOtherDirect.getTotalOtherDirectCostSharing().bigDecimalValue());
                    }
                }
                TotalDataType totalMaterial = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getMaterials() != null) {
                    totalMaterial.setFederal(cumOtherDirect.getMaterials().bigDecimalValue());
                }
                if (cumOtherDirect.getMaterialsCostSharing() != null) {
                    totalMaterial.setNonFederal(cumOtherDirect.getMaterialsCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getMaterials() != null) {
                        totalMaterial.setTotalFedNonFed(cumOtherDirect.getMaterials().add(cumOtherDirect.getMaterialsCostSharing())
                                .bigDecimalValue());
                    }
                    else {
                        totalMaterial.setTotalFedNonFed(cumOtherDirect.getMaterialsCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativeMaterialAndSupplies(totalMaterial);
                TotalDataType totalPublication = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getPublications() != null) {
                    totalPublication.setFederal(cumOtherDirect.getPublications().bigDecimalValue());
                }
                if (cumOtherDirect.getPublicationsCostSharing() != null) {
                    totalPublication.setNonFederal(cumOtherDirect.getPublicationsCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getPublications() != null) {
                        totalPublication.setTotalFedNonFed(cumOtherDirect.getPublications().add(
                                cumOtherDirect.getPublicationsCostSharing()).bigDecimalValue());
                    }
                    else {
                        totalPublication.setTotalFedNonFed(cumOtherDirect.getPublicationsCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativePublicationCosts(totalPublication);
                TotalDataType totalConsultant = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getConsultants() != null) {
                    totalConsultant.setFederal(cumOtherDirect.getConsultants().bigDecimalValue());
                }
                if (cumOtherDirect.getConsultantsCostSharing() != null) {
                    totalConsultant.setNonFederal(cumOtherDirect.getConsultantsCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getConsultants() != null) {
                        totalConsultant.setTotalFedNonFed(cumOtherDirect.getConsultants().add(
                                cumOtherDirect.getConsultantsCostSharing()).bigDecimalValue());
                    }
                    else {
                        totalConsultant.setTotalFedNonFed(cumOtherDirect.getConsultantsCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativeConsultantServices(totalConsultant);
                TotalDataType totalComputer = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getComputer() != null) {
                    totalComputer.setFederal(cumOtherDirect.getComputer().bigDecimalValue());
                }
                if (cumOtherDirect.getComputerCostSharing() != null) {
                    totalComputer.setNonFederal(cumOtherDirect.getComputerCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getComputer() != null) {
                        totalComputer.setTotalFedNonFed(cumOtherDirect.getComputer().add(cumOtherDirect.getComputerCostSharing())
                                .bigDecimalValue());
                    }
                    else {
                        totalComputer.setTotalFedNonFed(cumOtherDirect.getComputerCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativeADPComputerServices(totalComputer);
                TotalDataType totalSubaward = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getSubAwards() != null) {
                    totalSubaward.setFederal(cumOtherDirect.getSubAwards().bigDecimalValue());
                }
                if (cumOtherDirect.getSubAwardsCostSharing() != null) {
                    totalSubaward.setNonFederal(cumOtherDirect.getSubAwardsCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getSubAwards() != null) {
                        totalSubaward.setTotalFedNonFed(cumOtherDirect.getSubAwards().add(cumOtherDirect.getSubAwardsCostSharing())
                                .bigDecimalValue());
                    }
                    else {
                        totalSubaward.setTotalFedNonFed(cumOtherDirect.getSubAwardsCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativeSubawardConsortiumContractualCosts(totalSubaward);
                TotalDataType totalEquipment = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getEquipRental() != null) {
                    totalEquipment.setFederal(cumOtherDirect.getEquipRental().bigDecimalValue());
                }
                if (cumOtherDirect.getEquipRentalCostSharing() != null) {
                    totalEquipment.setNonFederal(cumOtherDirect.getEquipRentalCostSharing().bigDecimalValue());
                    totalEquipment.setTotalFedNonFed(cumOtherDirect.getEquipRental()
                            .add(cumOtherDirect.getEquipRentalCostSharing()).bigDecimalValue());
                }
                cumulativeOtherDirect.setCumulativeEquipmentFacilityRentalFees(totalEquipment);
                cumulativeOtherDirect.setCumulativeEquipmentFacilityRentalFees(totalEquipment);
                TotalDataType totalAlterations = TotalDataType.Factory.newInstance();
                if (cumOtherDirect.getAlterations() != null) {
                    totalAlterations.setFederal(cumOtherDirect.getAlterations().bigDecimalValue());
                }
                if (cumOtherDirect.getAlterationsCostSharing() != null) {
                    totalAlterations.setNonFederal(cumOtherDirect.getAlterationsCostSharing().bigDecimalValue());
                    if (cumOtherDirect.getAlterations() != null) {
                        totalAlterations.setTotalFedNonFed(cumOtherDirect.getAlterations().add(
                                cumOtherDirect.getAlterationsCostSharing()).bigDecimalValue());
                    }
                    else {
                        totalAlterations.setTotalFedNonFed(cumOtherDirect.getAlterationsCostSharing().bigDecimalValue());
                    }
                }
                cumulativeOtherDirect.setCumulativeAlterationsAndRenovations(totalAlterations);

                List<S2SOtherCostDto> cvOthers = cumOtherDirect.getOtherCosts();
                if (cvOthers.size() > SUPPORTED_OTHER_COSTS) {
                    getAuditErrors().add(UnsupportedOtherCostsErrorBuilder.build(SUPPORTED_OTHER_COSTS));
                } else {
                    for (int j = 0; j < cvOthers.size(); j++) {
                        S2SOtherCostDto hmCosts = cvOthers.get(j);
                        final TotalDataType totalOther = TotalDataType.Factory.newInstance();
                        totalOther.setFederal(hmCosts.getCost().bigDecimalValue());
                        totalOther.setNonFederal(hmCosts.getCostSharing().bigDecimalValue());
                        totalOther.setTotalFedNonFed(hmCosts.getCost().add(hmCosts.getCostSharing()).bigDecimalValue());
                        if (j == 0) {
                            cumulativeOtherDirect.setCumulativeOther1DirectCost(totalOther);
                        } else if (j == 1) {
                            cumulativeOtherDirect.setCumulativeOther2DirectCost(totalOther);
                        } else {
                            cumulativeOtherDirect.setCumulativeOther3DirectCost(totalOther);
                        }
                    }
                }
            }
        }
        cumulativeOtherDirect.setCumulativeTotalFundsRequestedOtherDirectCosts(summary);
        return cumulativeOtherDirect;
    }

    /**
     * This method gets CumulativeTrainee details,
     * CumulativeTotalFundsRequestedTraineeCosts,CumulativeTraineeTuitionFeesHealthInsurance
     * CumulativeTraineeStipends,CumulativeTraineeTravel,CumulativeTraineeSubsistence,CumulativeOtherTraineeCost and
     * CumulativeNoofTrainees based on BudgetSummaryInfo for the RRFedNonFedBudget.
     * 
     * @param budgetSummaryData (BudgetSummaryInfo) budget summary entry.
     * @return CumulativeTrainee corresponding to the BudgetSummaryInfo object.
     */
    private CumulativeTrainee getCumulativeTrainee(S2SBudgetDto budgetSummaryData) {
        CumulativeTrainee cumulativeTrainee = CumulativeTrainee.Factory.newInstance();
        SummaryDataType summaryTraineeCosts = SummaryDataType.Factory.newInstance();
        if (budgetSummaryData != null) {
            if (budgetSummaryData.getPartOtherCost() != null && budgetSummaryData.getPartStipendCost() != null
                    && budgetSummaryData.getPartTravelCost() != null && budgetSummaryData.getPartSubsistence() != null
                    && budgetSummaryData.getPartTuition() != null) {
                summaryTraineeCosts.setFederalSummary(budgetSummaryData.getPartOtherCost().add(
                        budgetSummaryData.getPartStipendCost().add(
                                budgetSummaryData.getPartTravelCost().add(
                                        budgetSummaryData.getPartSubsistence().add(budgetSummaryData.getPartTuition()))))
                        .bigDecimalValue());
            }
            if (budgetSummaryData.getPartOtherCostSharing() != null && budgetSummaryData.getPartStipendCostSharing() != null
                    && budgetSummaryData.getPartTravelCostSharing() != null
                    && budgetSummaryData.getPartSubsistenceCostSharing() != null
                    && budgetSummaryData.getPartTuitionCostSharing() != null) {
                summaryTraineeCosts.setNonFederalSummary(budgetSummaryData.getPartOtherCostSharing().add(
                        budgetSummaryData.getPartStipendCostSharing().add(
                                budgetSummaryData.getPartTravelCostSharing().add(
                                        budgetSummaryData.getPartSubsistenceCostSharing().add(
                                                budgetSummaryData.getPartTuitionCostSharing())))).bigDecimalValue());
            }

            if (summaryTraineeCosts.getNonFederalSummary() != null) {
                if (summaryTraineeCosts.getFederalSummary() != null) {
                    summaryTraineeCosts.setTotalFedNonFedSummary(summaryTraineeCosts.getFederalSummary().add(
                            summaryTraineeCosts.getNonFederalSummary()));
                }

                else {
                    summaryTraineeCosts.setTotalFedNonFedSummary(summaryTraineeCosts.getNonFederalSummary());
                }
            }
            TotalDataType totalTuition = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getPartTuition() != null) {
                totalTuition.setFederal(budgetSummaryData.getPartTuition().bigDecimalValue());
            }
            if (budgetSummaryData.getPartTuitionCostSharing() != null) {
                totalTuition.setNonFederal(budgetSummaryData.getPartTuitionCostSharing().bigDecimalValue());
                if (budgetSummaryData.getPartTuition() != null) {
                    totalTuition.setTotalFedNonFed(budgetSummaryData.getPartTuition().add(
                            budgetSummaryData.getPartTuitionCostSharing()).bigDecimalValue());
                }
                else {
                    totalTuition.setTotalFedNonFed(budgetSummaryData.getPartTuitionCostSharing().bigDecimalValue());
                }
            }
            cumulativeTrainee.setCumulativeTraineeTuitionFeesHealthInsurance(totalTuition);
            TotalDataType totalStipends = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getPartStipendCost() != null) {
                totalStipends.setFederal(budgetSummaryData.getPartStipendCost().bigDecimalValue());
            }
            if (budgetSummaryData.getPartStipendCostSharing() != null) {
                totalStipends.setNonFederal(budgetSummaryData.getPartStipendCostSharing().bigDecimalValue());
                if (budgetSummaryData.getPartStipendCost() != null) {
                    totalStipends.setTotalFedNonFed(budgetSummaryData.getPartStipendCost().add(
                            budgetSummaryData.getPartStipendCostSharing()).bigDecimalValue());
                }
                else {
                    totalStipends.setTotalFedNonFed(budgetSummaryData.getPartStipendCostSharing().bigDecimalValue());
                }
            }
            cumulativeTrainee.setCumulativeTraineeStipends(totalStipends);
            TotalDataType totalTravel = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getPartTravelCost() != null) {
                totalTravel.setFederal(budgetSummaryData.getPartTravelCost().bigDecimalValue());
            }
            if (budgetSummaryData.getPartTravelCostSharing() != null) {
                totalTravel.setNonFederal(budgetSummaryData.getPartTravelCostSharing().bigDecimalValue());
                if (budgetSummaryData.getPartTravelCost() != null) {
                    totalTravel.setTotalFedNonFed(budgetSummaryData.getPartTravelCost().add(
                            budgetSummaryData.getPartTravelCostSharing()).bigDecimalValue());
                }
                else {
                    totalTravel.setTotalFedNonFed(budgetSummaryData.getPartTravelCostSharing().bigDecimalValue());
                }
            }
            cumulativeTrainee.setCumulativeTraineeTravel(totalTravel);
            TotalDataType totalSubsistence = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getPartSubsistence() != null) {
                totalSubsistence.setFederal(budgetSummaryData.getPartSubsistence().bigDecimalValue());
            }
            if (budgetSummaryData.getPartSubsistenceCostSharing() != null) {
                totalSubsistence.setNonFederal(budgetSummaryData.getPartSubsistenceCostSharing().bigDecimalValue());
                if (budgetSummaryData.getPartSubsistence() != null) {
                    totalSubsistence.setTotalFedNonFed(budgetSummaryData.getPartSubsistence().add(
                            budgetSummaryData.getPartSubsistenceCostSharing()).bigDecimalValue());
                }
                else {
                    totalSubsistence.setTotalFedNonFed(budgetSummaryData.getPartSubsistenceCostSharing().bigDecimalValue());
                }
            }
            cumulativeTrainee.setCumulativeTraineeSubsistence(totalSubsistence);
            TotalDataType totalOtherTrainee = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getPartOtherCost() != null) {
                totalOtherTrainee.setFederal(budgetSummaryData.getPartOtherCost().bigDecimalValue());
            }
            if (budgetSummaryData.getPartOtherCostSharing() != null) {
                totalOtherTrainee.setNonFederal(budgetSummaryData.getPartOtherCostSharing().bigDecimalValue());
                if (budgetSummaryData.getPartOtherCost() != null) {
                    totalOtherTrainee.setTotalFedNonFed(budgetSummaryData.getPartOtherCost().add(
                            budgetSummaryData.getPartOtherCostSharing()).bigDecimalValue());
                }
                else {
                    totalOtherTrainee.setTotalFedNonFed(budgetSummaryData.getPartOtherCostSharing().bigDecimalValue());
                }
            }
            cumulativeTrainee.setCumulativeOtherTraineeCost(totalOtherTrainee);
            cumulativeTrainee.setCumulativeNoofTrainees(budgetSummaryData.getParticipantCount());
        }
        cumulativeTrainee.setCumulativeTotalFundsRequestedTraineeCosts(summaryTraineeCosts);
        return cumulativeTrainee;
    }

    /**
     * This method gets CumulativeEquipments information CumulativeTotalFundsRequestedEquipment based on BudgetSummaryInfo for the
     * form RRFedNonFedBudget.
     * 
     * @param budgetSummaryData (BudgetSummaryInfo) budget summary entry.
     * @return CumulativeEquipments corresponding to the BudgetSummaryInfo object.
     */
    private CumulativeEquipments getCumulativeEquipments(S2SBudgetDto budgetSummaryData) {

        CumulativeEquipments cumulativeEquipments = CumulativeEquipments.Factory.newInstance();
        if (budgetSummaryData != null) {
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (budgetSummaryData.getCumEquipmentFunds() != null) {
                summary.setFederalSummary(budgetSummaryData.getCumEquipmentFunds().bigDecimalValue());
            }
            if (budgetSummaryData.getCumEquipmentNonFunds() != null) {
                summary.setNonFederalSummary(budgetSummaryData.getCumEquipmentNonFunds().bigDecimalValue());
                if (budgetSummaryData.getCumEquipmentFunds() != null) {
                    summary.setTotalFedNonFedSummary(budgetSummaryData.getCumEquipmentFunds().add(
                            budgetSummaryData.getCumEquipmentNonFunds()).bigDecimalValue());
                }
                else {
                    summary.setTotalFedNonFedSummary(budgetSummaryData.getCumEquipmentNonFunds().bigDecimalValue());
                }
            }
            cumulativeEquipments.setCumulativeTotalFundsRequestedEquipment(summary);
        }
        return cumulativeEquipments;
    }

    /**
     * This method gets CumulativeTravels details,CumulativeTotalFundsRequestedTravel,CumulativeDomesticTravelCosts and
     * CumulativeForeignTravelCosts based on BudgetSummaryInfo for the RRFedNonFedBudget.
     * 
     * @param budgetSummaryData (BudgetSummaryInfo) budget summary entry.
     * @return CumulativeTravels corresponding to the BudgetSummaryInfo object.
     */
    private CumulativeTravels getCumulativeTravels(S2SBudgetDto budgetSummaryData) {
        CumulativeTravels cumulativeTravels = CumulativeTravels.Factory.newInstance();
        SummaryDataType summary = SummaryDataType.Factory.newInstance();
        if (budgetSummaryData != null) {
            if (budgetSummaryData.getCumTravel() != null) {
                summary.setFederalSummary(budgetSummaryData.getCumTravel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumTravelNonFund() != null) {
                summary.setNonFederalSummary(budgetSummaryData.getCumTravelNonFund().bigDecimalValue());
                if (budgetSummaryData.getCumTravel() != null) {
                    summary.setTotalFedNonFedSummary(budgetSummaryData.getCumTravel().add(budgetSummaryData.getCumTravelNonFund())
                            .bigDecimalValue());
                }
                else {
                    summary.setTotalFedNonFedSummary(budgetSummaryData.getCumTravelNonFund().bigDecimalValue());
                }
            }
            TotalDataType totalDomestic = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getCumDomesticTravel() != null) {
                totalDomestic.setFederal(budgetSummaryData.getCumDomesticTravel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumDomesticTravelNonFund() != null) {
                totalDomestic.setNonFederal(budgetSummaryData.getCumDomesticTravelNonFund().bigDecimalValue());
                if (budgetSummaryData.getCumDomesticTravel() != null) {
                    totalDomestic.setTotalFedNonFed(budgetSummaryData.getCumDomesticTravel().add(
                            budgetSummaryData.getCumDomesticTravelNonFund()).bigDecimalValue());
                }
                else {
                    totalDomestic.setTotalFedNonFed(budgetSummaryData.getCumDomesticTravelNonFund().bigDecimalValue());
                }
            }
            cumulativeTravels.setCumulativeDomesticTravelCosts(totalDomestic);
            TotalDataType totalForeign = TotalDataType.Factory.newInstance();
            if (budgetSummaryData.getCumForeignTravel() != null) {
                totalForeign.setFederal(budgetSummaryData.getCumForeignTravel().bigDecimalValue());
            }
            if (budgetSummaryData.getCumForeignTravelNonFund() != null) {
                totalForeign.setNonFederal(budgetSummaryData.getCumForeignTravelNonFund().bigDecimalValue());
                if (budgetSummaryData.getCumForeignTravel() != null) {
                    totalForeign.setTotalFedNonFed(budgetSummaryData.getCumForeignTravel().add(
                            budgetSummaryData.getCumForeignTravelNonFund()).bigDecimalValue());
                }
                else {
                    totalForeign.setTotalFedNonFed(budgetSummaryData.getCumForeignTravelNonFund().bigDecimalValue());
                }
            }
            cumulativeTravels.setCumulativeForeignTravelCosts(totalForeign);
        }
        cumulativeTravels.setCumulativeTotalFundsRequestedTravel(summary);
        return cumulativeTravels;
    }

    /**
     * This method gets BudgetYear1DataType details like BudgetPeriodStartDate,BudgetPeriodEndDate,BudgetPeriod
     * KeyPersons,OtherPersonnel,TotalCompensation,Equipment,ParticipantTraineeSupportCosts,Travel,OtherDirectCosts
     * DirectCosts,IndirectCosts,CognizantFederalAgency,TotalCosts  based on BudgetPeriodInfo for
     * the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return BudgetYear1DataType corresponding to the BudgetPeriodInfo object.
     */
    private BudgetYear1DataType getBudgetYear1DataType(S2SBudgetPeriodDto periodInfo) {

        BudgetYear1DataType budgetYear = BudgetYear1DataType.Factory.newInstance();
        ScaleTwoDecimal totalDirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalIndirectCostSharing = ScaleTwoDecimal.ZERO;
        if (periodInfo != null) {
            budgetYear.setBudgetPeriodStartDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getStartDate()));
            budgetYear.setBudgetPeriodEndDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getEndDate()));
            gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.BudgetPeriod.Enum budgetPeriodEnum = gov.grants.apply.forms.rrFedNonFedBudgetV11.BudgetYearDataType.BudgetPeriod.Enum
                    .forInt(periodInfo.getBudgetPeriod());
            budgetYear.setBudgetPeriod(budgetPeriodEnum);
            budgetYear.setKeyPersons(getKeyPersons(periodInfo));
            budgetYear.setOtherPersonnel(getOtherPersonnel(periodInfo));
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalCompensation() != null) {
                summary.setFederalSummary(periodInfo.getTotalCompensation().bigDecimalValue());
            }
            if (periodInfo.getTotalCompensationCostSharing() != null) {
                if (budget.getSubmitCostSharingFlag()) {
                    summary.setNonFederalSummary(periodInfo.getTotalCompensationCostSharing().bigDecimalValue());
                    if (periodInfo.getTotalCompensation() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensation().add(
                                periodInfo.getTotalCompensationCostSharing()).bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensationCostSharing().bigDecimalValue());
                    }
                } else {
                    summary.setNonFederalSummary(BigDecimal.ZERO);
                    if (periodInfo.getTotalCompensation() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalCompensation().bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(BigDecimal.ZERO);
                    }
                }
            }
            budgetYear.setTotalCompensation(summary);
            budgetYear.setEquipment(getEquipment(periodInfo));
            budgetYear.setTravel(getTravel(periodInfo));
            budgetYear.setParticipantTraineeSupportCosts(getParticipantTraineeSupportCosts(periodInfo));
            budgetYear.setOtherDirectCosts(getOtherDirectCosts(periodInfo));
            SummaryDataType summaryDirect = SummaryDataType.Factory.newInstance();
            if (periodInfo.getDirectCostsTotal() != null) {
                summaryDirect.setFederalSummary(periodInfo.getDirectCostsTotal().bigDecimalValue());
            }
            if (periodInfo.getTotalDirectCostSharing() != null){
            totalDirectCostSharing = periodInfo.getTotalDirectCostSharing();
            }
            summaryDirect.setNonFederalSummary(totalDirectCostSharing.bigDecimalValue());
            if (periodInfo.getDirectCostsTotal() != null) {
                summaryDirect.setTotalFedNonFedSummary(periodInfo.getDirectCostsTotal().add(totalDirectCostSharing).bigDecimalValue());
            }
            else {
                summaryDirect.setTotalFedNonFedSummary(totalDirectCostSharing.bigDecimalValue());
            }
            budgetYear.setDirectCosts(summaryDirect);
            budgetYear.setIndirectCosts(getIndirectCosts(periodInfo));
            budgetYear.setCognizantFederalAgency(periodInfo.getCognizantFedAgency());
            if (periodInfo.getIndirectCosts().getTotalIndirectCostSharing() != null){
                totalIndirectCostSharing = periodInfo.getIndirectCosts().getTotalIndirectCostSharing();
            }
            SummaryDataType summaryTotal = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalCosts() != null) {
                summaryTotal.setFederalSummary(periodInfo.getTotalCosts().bigDecimalValue());
            }
            if (periodInfo.getCostSharingAmount() != null) {
                summaryTotal.setNonFederalSummary(totalDirectCostSharing.bigDecimalValue().add(totalIndirectCostSharing.bigDecimalValue()));
                if (periodInfo.getTotalCosts() != null) {
                    summaryTotal.setTotalFedNonFedSummary(periodInfo.getTotalCosts().add(totalDirectCostSharing).add(totalIndirectCostSharing)
                            .bigDecimalValue());
                }
                else {
                    summaryTotal.setTotalFedNonFedSummary(totalDirectCostSharing.bigDecimalValue().add(totalIndirectCostSharing.bigDecimalValue()));
                }
            }
            budgetYear.setTotalCosts(summaryTotal);
        }
        
        return budgetYear;
    }
    /*
     * This method gets BudgetJustificationAttachment from ProposalDevelopmentDocumentContract for the RRFedNonFedBudget.
     */
    private BudgetYear1DataType getBudgetJustificationAttachment(BudgetYear1DataType budgetYear) {
        AttachedFileDataType attachedFileDataType = null;
        budgetYear.setBudgetJustificationAttachment(null);
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == BUDGET_JUSTIFICATION_ATTACHMENT) {
                attachedFileDataType = addAttachedFileType(narrative);
                if(attachedFileDataType != null){
                    budgetYear.setBudgetJustificationAttachment(attachedFileDataType);
                    break;
                }
            }
        }
        return budgetYear;
    }

    /**
     * This method returns IndirectCosts details such as Base,CostType,FundRequested,Rate and TotalIndirectCosts in
     * BudgetYearDataType based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return IndirectCosts corresponding to the BudgetPeriodInfo object.
     */
    private IndirectCosts getIndirectCosts(S2SBudgetPeriodDto periodInfo) {

        IndirectCosts indirectCosts = null;
        if (periodInfo != null && periodInfo.getIndirectCosts() != null
                && periodInfo.getIndirectCosts().getIndirectCostDetails() != null) {

            int IndirectCostCount = 0;
            List<IndirectCosts.IndirectCost> indirectCostList = new ArrayList<>();
            for (S2SIndirectCostDetailsDto indirectCostDetails : periodInfo.getIndirectCosts().getIndirectCostDetails()) {
                IndirectCosts.IndirectCost indirectCost = IndirectCosts.IndirectCost.Factory.newInstance();
                indirectCost.setCostType(indirectCostDetails.getCostType());
                if (indirectCostDetails.getBase() != null) {
                    indirectCost.setBase(indirectCostDetails.getBase().bigDecimalValue());
                }
                if (indirectCostDetails.getRate() != null) {
                    indirectCost.setRate(indirectCostDetails.getRate().bigDecimalValue());
                }
                TotalDataType total = TotalDataType.Factory.newInstance();
                if (indirectCostDetails.getFunds() != null) {
                    total.setFederal(indirectCostDetails.getFunds().bigDecimalValue());
                }
                else{
                    total.setFederal(new BigDecimal(0.00));
                }
                if (indirectCostDetails.getCostSharing() != null) {
                    total.setNonFederal(indirectCostDetails.getCostSharing().bigDecimalValue());
                    if (indirectCostDetails.getFunds() != null) {
                        total.setTotalFedNonFed(indirectCostDetails.getFunds().add(indirectCostDetails.getCostSharing())
                                .bigDecimalValue());
                    }
                    else {
                        total.setTotalFedNonFed(indirectCostDetails.getCostSharing().bigDecimalValue());
                    }
                }
                else if(indirectCostDetails.getFunds() != null){
                    total.setTotalFedNonFed(indirectCostDetails.getFunds().bigDecimalValue());
                    total.setNonFederal(new BigDecimal(0.00));
                }
                else{
                    total.setNonFederal(new BigDecimal(0.00));
                    total.setTotalFedNonFed(new BigDecimal(0.00));
                }
                indirectCost.setFundRequested(total);
                indirectCostList.add(indirectCost);
                IndirectCostCount++;
                if (IndirectCostCount == ARRAY_LIMIT_IN_SCHEMA) {
                    LOG.warn("Stopping iteration over indirect cost details because array limit in schema is only 4");
                    break;
                }
            }
            if (IndirectCostCount > 0) {
                indirectCosts = IndirectCosts.Factory.newInstance();

                indirectCosts.setIndirectCostArray(indirectCostList.toArray(new IndirectCosts.IndirectCost[0]));
                SummaryDataType summary = SummaryDataType.Factory.newInstance();
                if (periodInfo.getIndirectCosts().getTotalIndirectCosts() != null) {
                    summary.setFederalSummary(periodInfo.getIndirectCosts().getTotalIndirectCosts().bigDecimalValue());
                }
                if (periodInfo.getIndirectCosts().getTotalIndirectCostSharing() != null) {
                    summary.setNonFederalSummary(periodInfo.getIndirectCosts().getTotalIndirectCostSharing().bigDecimalValue());
                    if (periodInfo.getIndirectCosts().getTotalIndirectCosts() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getIndirectCosts().getTotalIndirectCosts().add(
                                periodInfo.getIndirectCosts().getTotalIndirectCostSharing()).bigDecimalValue());
                    }
                    else {
                        summary.setTotalFedNonFedSummary(periodInfo.getIndirectCosts().getTotalIndirectCostSharing()
                                .bigDecimalValue());
                    }
                }
                indirectCosts.setTotalIndirectCosts(summary);
            }
        }
        return indirectCosts;
    }


    /**
     * This method gets OtherDirectCosts details such as PublicationCosts,MaterialsSupplies,ConsultantServices,
     * ADPComputerServices,SubawardConsortiumContractualCosts,EquipmentRentalFee,AlterationsRenovations and TotalOtherDirectCost in
     * BudgetYearDataType based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return OtherDirectCosts corresponding to the BudgetPeriodInfo object.
     */
    private OtherDirectCosts getOtherDirectCosts(S2SBudgetPeriodDto periodInfo) {

        OtherDirectCosts otherDirectCosts = OtherDirectCosts.Factory.newInstance();
        TotalDataType totalMaterials = TotalDataType.Factory.newInstance();
        if (periodInfo != null && periodInfo.getOtherDirectCosts() != null && periodInfo.getOtherDirectCosts().size() > 0) {
            if (periodInfo.getOtherDirectCosts().get(0).getMaterials() != null) {
                totalMaterials.setFederal(periodInfo.getOtherDirectCosts().get(0).getMaterials().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getMaterialsCostSharing() != null) {
                totalMaterials.setNonFederal(periodInfo.getOtherDirectCosts().get(0).getMaterialsCostSharing().bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getMaterials() != null) {
                    totalMaterials.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getMaterials().add(
                            periodInfo.getOtherDirectCosts().get(0).getMaterialsCostSharing()).bigDecimalValue());
                }
                else {
                    totalMaterials.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getMaterialsCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setMaterialsSupplies(totalMaterials);
            TotalDataType totalPublication = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getPublications() != null) {
                totalPublication.setFederal(periodInfo.getOtherDirectCosts().get(0).getPublications().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getPublicationsCostSharing() != null) {
                totalPublication.setNonFederal(periodInfo.getOtherDirectCosts().get(0).getPublicationsCostSharing()
                        .bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getPublications() != null) {
                    totalPublication.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getPublications().add(
                            periodInfo.getOtherDirectCosts().get(0).getPublicationsCostSharing()).bigDecimalValue());
                }
                else {
                    totalPublication.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getPublicationsCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setPublicationCosts(totalPublication);
            TotalDataType totalConsultant = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getConsultants() != null) {
                totalConsultant.setFederal(periodInfo.getOtherDirectCosts().get(0).getConsultants().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getConsultantsCostSharing() != null) {
                totalConsultant
                        .setNonFederal(periodInfo.getOtherDirectCosts().get(0).getConsultantsCostSharing().bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getConsultants() != null) {
                    totalConsultant.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getConsultants().add(
                            periodInfo.getOtherDirectCosts().get(0).getConsultantsCostSharing()).bigDecimalValue());
                }
                else {
                    totalConsultant.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getConsultantsCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setConsultantServices(totalConsultant);
            TotalDataType totalADP = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getComputer() != null) {
                totalADP.setFederal(periodInfo.getOtherDirectCosts().get(0).getComputer().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getComputerCostSharing() != null) {
                totalADP.setNonFederal(periodInfo.getOtherDirectCosts().get(0).getComputerCostSharing().bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getComputer() != null) {
                    totalADP.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getComputer().add(
                            periodInfo.getOtherDirectCosts().get(0).getComputerCostSharing()).bigDecimalValue());
                }
                else {
                    totalADP.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getComputerCostSharing().bigDecimalValue());
                }
            }
            otherDirectCosts.setADPComputerServices(totalADP);
            TotalDataType totalSubaward = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getSubAwards() != null) {
                totalSubaward.setFederal(periodInfo.getOtherDirectCosts().get(0).getSubAwards().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getSubAwardsCostSharing() != null) {
                totalSubaward.setNonFederal((periodInfo.getOtherDirectCosts().get(0).getSubAwardsCostSharing().bigDecimalValue()));
                if (periodInfo.getOtherDirectCosts().get(0).getSubAwards() != null) {
                    totalSubaward.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getSubAwards().add(
                            periodInfo.getOtherDirectCosts().get(0).getSubAwardsCostSharing()).bigDecimalValue());
                }
                else {
                    totalSubaward.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getSubAwardsCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setSubawardConsortiumContractualCosts(totalSubaward);
            TotalDataType totalEquipment = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getEquipRental() != null) {
                totalEquipment.setFederal(periodInfo.getOtherDirectCosts().get(0).getEquipRental().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getEquipRentalCostSharing() != null) {
                totalEquipment.setNonFederal(periodInfo.getOtherDirectCosts().get(0).getEquipRentalCostSharing().bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getEquipRental() != null) {
                    totalEquipment.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getEquipRental().add(
                            periodInfo.getOtherDirectCosts().get(0).getEquipRentalCostSharing()).bigDecimalValue());
                }
                else {
                    totalEquipment.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getEquipRentalCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setEquipmentRentalFee(totalEquipment);
            TotalDataType totalAlterations = TotalDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getAlterations() != null) {
                totalAlterations.setFederal(periodInfo.getOtherDirectCosts().get(0).getAlterations().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getAlterationsCostSharing() != null) {
                totalAlterations.setNonFederal(periodInfo.getOtherDirectCosts().get(0).getAlterationsCostSharing()
                        .bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getAlterations() != null) {
                    totalAlterations.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getAlterations().add(
                            periodInfo.getOtherDirectCosts().get(0).getAlterationsCostSharing()).bigDecimalValue());
                }
                else {
                    totalAlterations.setTotalFedNonFed(periodInfo.getOtherDirectCosts().get(0).getAlterationsCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setAlterationsRenovations(totalAlterations);
            otherDirectCosts.setOthers(getOthersForOtherDirectCosts(periodInfo));
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect() != null) {
                summary.setFederalSummary(periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect().bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirectCostSharing() != null) {
                summary.setNonFederalSummary(periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirectCostSharing()
                        .bigDecimalValue());
                if (periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect() != null) {
                    summary.setTotalFedNonFedSummary(periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect().add(
                            periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirectCostSharing()).bigDecimalValue());
                }
                else {
                    summary.setTotalFedNonFedSummary(periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirectCostSharing()
                            .bigDecimalValue());
                }
            }
            otherDirectCosts.setTotalOtherDirectCost(summary);
        }
        return otherDirectCosts;
    }

    private Others getOthersForOtherDirectCosts(S2SBudgetPeriodDto periodInfo) {
        final Others othersDirect = Others.Factory.newInstance();

        if (periodInfo != null && periodInfo.getOtherDirectCosts() != null) {
            for (S2SOtherDirectCostInfoDto otherDirectCostInfo : periodInfo.getOtherDirectCosts()) {
                if (CollectionUtils.isNotEmpty(otherDirectCostInfo.getOtherCosts())) {
                    for (S2SOtherCostDto oc : otherDirectCostInfo.getOtherCosts()) {
                        final Others.Other other = othersDirect.addNewOther();
                        final TotalDataType total = TotalDataType.Factory.newInstance();
                        total.setFederal(oc.getCost().bigDecimalValue());
                        total.setNonFederal(oc.getCostSharing().bigDecimalValue());
                        total.setTotalFedNonFed(oc.getCost().add(oc.getCostSharing()).bigDecimalValue());
                        other.setCost(total);
                        other.setDescription(oc.getDescription());
                    }
                }
            }
        }

        return othersDirect;
    }

    /**
     * This method gets ParticipantTraineeSupportCosts details in BudgetYearDataType such as TuitionFeeHealthInsurance
     * Stipends,Subsistence,Travel,Other,ParticipantTraineeNumber and TotalCost based on the BudgetPeriodInfo for the
     * RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return ParticipantTraineeSupportCosts corresponding to the BudgetPeriodInfo object.
     */
    private ParticipantTraineeSupportCosts getParticipantTraineeSupportCosts(S2SBudgetPeriodDto periodInfo) {

        ParticipantTraineeSupportCosts traineeSupportCosts = ParticipantTraineeSupportCosts.Factory.newInstance();
        if (periodInfo != null) {
            TotalDataType totalTution = TotalDataType.Factory.newInstance();
            if (periodInfo.getPartTuition() != null) {
                totalTution.setFederal(periodInfo.getPartTuition().bigDecimalValue());
            }
            if (periodInfo.getPartTuitionCostSharing() != null) {
                totalTution.setNonFederal(periodInfo.getPartTuitionCostSharing().bigDecimalValue());
                if (periodInfo.getPartTuition() != null) {
                    totalTution.setTotalFedNonFed(periodInfo.getPartTuition().add(periodInfo.getPartTuitionCostSharing())
                            .bigDecimalValue());
                }
                else {
                    totalTution.setTotalFedNonFed(periodInfo.getPartTuitionCostSharing().bigDecimalValue());
                }
            }
            traineeSupportCosts.setTuitionFeeHealthInsurance(totalTution);
            TotalDataType totalStipends = TotalDataType.Factory.newInstance();
            if (periodInfo.getPartStipendCost() != null) {
                totalStipends.setFederal(periodInfo.getPartStipendCost().bigDecimalValue());
            }
            if (periodInfo.getPartStipendCostSharing() != null) {
                totalStipends.setNonFederal(periodInfo.getPartStipendCostSharing().bigDecimalValue());
                if (periodInfo.getPartStipendCost() != null) {
                    totalStipends.setTotalFedNonFed(periodInfo.getPartStipendCost().add(periodInfo.getPartStipendCostSharing())
                            .bigDecimalValue());
                }
                else {
                    totalStipends.setTotalFedNonFed(periodInfo.getPartStipendCostSharing().bigDecimalValue());
                }
            }
            traineeSupportCosts.setStipends(totalStipends);
            TotalDataType totalTravel = TotalDataType.Factory.newInstance();
            if (periodInfo.getPartTravelCost() != null) {
                totalTravel.setFederal(periodInfo.getPartTravelCost().bigDecimalValue());
            }
            if (periodInfo.getPartTravelCostSharing() != null) {
                totalTravel.setNonFederal(periodInfo.getPartTravelCostSharing().bigDecimalValue());
                if (periodInfo.getPartTravelCost() != null) {
                    totalTravel.setTotalFedNonFed(periodInfo.getPartTravelCost().add(periodInfo.getPartTravelCostSharing())
                            .bigDecimalValue());
                }
                else {
                    totalTravel.setTotalFedNonFed(periodInfo.getPartTravelCostSharing().bigDecimalValue());
                }
            }
            traineeSupportCosts.setTravel(totalTravel);
            TotalDataType totalSubsistence = TotalDataType.Factory.newInstance();
            if (periodInfo.getPartSubsistence() != null) {
                totalSubsistence.setFederal(periodInfo.getPartSubsistence().bigDecimalValue());
            }
            if (periodInfo.getPartSubsistenceCostSharing() != null) {
                totalSubsistence.setNonFederal(periodInfo.getPartSubsistenceCostSharing().bigDecimalValue());
                if (periodInfo.getPartSubsistence() != null) {
                    totalSubsistence.setTotalFedNonFed(periodInfo.getPartSubsistence().add(
                            periodInfo.getPartSubsistenceCostSharing()).bigDecimalValue());
                }
                else {
                    totalSubsistence.setTotalFedNonFed(periodInfo.getPartSubsistenceCostSharing().bigDecimalValue());
                }
            }
            traineeSupportCosts.setSubsistence(totalSubsistence);
            traineeSupportCosts.setOther(getOtherPTSupportCosts(periodInfo));
            traineeSupportCosts.setParticipantTraineeNumber(periodInfo.getParticipantCount());
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            summary.setFederalSummary(periodInfo.getPartOtherCost().add(
                    periodInfo.getPartStipendCost().add(
                            periodInfo.getPartTravelCost().add(periodInfo.getPartSubsistence().add(periodInfo.getPartTuition()))))
                    .bigDecimalValue());

            summary.setNonFederalSummary(periodInfo.getPartOtherCostSharing().add(
                    periodInfo.getPartStipendCostSharing().add(
                            periodInfo.getPartTravelCostSharing().add(
                                    periodInfo.getPartSubsistenceCostSharing().add(periodInfo.getPartTuitionCostSharing()))))
                    .bigDecimalValue());
            if (summary.getNonFederalSummary() != null) {
                if (summary.getFederalSummary() != null) {
                    summary.setTotalFedNonFedSummary(summary.getFederalSummary().add(summary.getNonFederalSummary()));
                }
                else {
                    summary.setTotalFedNonFedSummary(summary.getNonFederalSummary());
                }
            }
            traineeSupportCosts.setTotalCost(summary);
        }
        return traineeSupportCosts;
    }

    /**
     * This method gets Other type description and total cost for ParticipantTraineeSupportCosts based on BudgetPeriodInfo.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return Other other participant trainee support costs corresponding to the BudgetPeriodInfo object.
     */
    private Other getOtherPTSupportCosts(S2SBudgetPeriodDto periodInfo) {

        Other other = Other.Factory.newInstance();
        other.setDescription(OTHERCOST_DESCRIPTION);
        TotalDataType total = TotalDataType.Factory.newInstance();
        if (periodInfo != null) {
            if (periodInfo.getPartOtherCost() != null) {
                total.setFederal(periodInfo.getPartOtherCost().bigDecimalValue());
            }
            if (periodInfo.getPartOtherCostSharing() != null) {
                total.setNonFederal(periodInfo.getPartOtherCostSharing().bigDecimalValue());
                if (periodInfo.getPartOtherCost() != null) {
                    total.setTotalFedNonFed(periodInfo.getPartOtherCost().add(periodInfo.getPartOtherCostSharing())
                            .bigDecimalValue());
                }
                else {
                    total.setTotalFedNonFed(periodInfo.getPartOtherCostSharing().bigDecimalValue());
                }
            }
        }
        other.setCost(total);
        return other;
    }

    /**
     * This method gets Equipment details such as EquipmentItem,FundsRequested,TotalFundForAttachedEquipment, TotalFund and
     * AdditionalEquipmentsAttachment based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return Equipment cost details corresponding to the BudgetPeriodInfo object.
     */
    private Equipment getEquipment(S2SBudgetPeriodDto periodInfo) {
        Equipment equipment = Equipment.Factory.newInstance();

        List<EquipmentList> equipmentArrayList = new ArrayList<>();
        if (periodInfo.getEquipment() != null && periodInfo.getEquipment().size() > 0) {
                SummaryDataType totalFund = SummaryDataType.Factory.newInstance();
                totalFund.setFederalSummary(BigDecimal.ZERO);
                totalFund.setNonFederalSummary(BigDecimal.ZERO);
                totalFund.setTotalFedNonFedSummary(BigDecimal.ZERO);
                for (S2SCostDto costInfo : periodInfo.getEquipment().get(0).getEquipmentList()) {
                    EquipmentList equipmentList = EquipmentList.Factory.newInstance();
                    equipmentList.setEquipmentItem(costInfo.getDescription());

                    TotalDataType fundsRequested = TotalDataType.Factory.newInstance();
                    fundsRequested.setFederal(costInfo.getCost().bigDecimalValue());
                    fundsRequested.setNonFederal(costInfo.getCostSharing().bigDecimalValue());
                    fundsRequested.setTotalFedNonFed(costInfo.getCost().add(costInfo.getCostSharing()).bigDecimalValue());

                    //  prepare the totals
                    totalFund.setFederalSummary(totalFund.getFederalSummary().add(costInfo.getCost().bigDecimalValue()));
                    totalFund.setNonFederalSummary(totalFund.getNonFederalSummary()
                            .add(costInfo.getCostSharing().bigDecimalValue()));

                    equipmentList.setFundsRequested(fundsRequested);                   
                    equipmentArrayList.add(equipmentList);
                }
                totalFund.setTotalFedNonFedSummary(totalFund.getFederalSummary().add(totalFund.getNonFederalSummary()));
                EquipmentList[] equipmentArray = equipmentArrayList.toArray(new EquipmentList[0]);
                equipment.setEquipmentListArray(equipmentArray);
                equipment.setTotalFund(totalFund);
                // Extra Exuipment Amount Setting.
                S2SEquipmentDto equipmentInfo = periodInfo.getEquipment().get(0);
                TotalDataType totalFundForExtraEquipment = TotalDataType.Factory.newInstance();
                totalFundForExtraEquipment.setFederal(equipmentInfo.getTotalExtraFund().bigDecimalValue());
                totalFundForExtraEquipment.setNonFederal(equipmentInfo.getTotalExtraNonFund().bigDecimalValue());
                if (equipmentInfo.getTotalExtraFund() != null) {
                    totalFundForExtraEquipment.setTotalFedNonFed(equipmentInfo.getTotalExtraFund().add(equipmentInfo.getTotalExtraNonFund())
                            .bigDecimalValue());
                }
                else {
                    totalFundForExtraEquipment.setTotalFedNonFed(equipmentInfo.getTotalExtraNonFund().bigDecimalValue());
                }
                equipment.setTotalFundForAttachedEquipment(totalFundForExtraEquipment);
                // Save Total Fund.
                SummaryDataType summary = SummaryDataType.Factory.newInstance();
                if (equipmentInfo.getTotalFund() != null) {
                    summary.setFederalSummary(equipmentInfo.getTotalFund().bigDecimalValue());
                }
                if (equipmentInfo.getTotalNonFund() != null) {
                    summary.setNonFederalSummary(equipmentInfo.getTotalNonFund().bigDecimalValue());
                    if (equipmentInfo.getTotalFund() != null) {
                        summary.setTotalFedNonFedSummary(equipmentInfo.getTotalFund().add(equipmentInfo.getTotalNonFund())
                                .bigDecimalValue());
                    }
                    else {
                        summary.setTotalFedNonFedSummary(equipmentInfo.getTotalNonFund().bigDecimalValue());
                    }
                }
                equipment.setTotalFund(summary);
                

        }
        NarrativeContract narrative = saveExtraEquipment(periodInfo);
        AttachedFileDataType attachedFileDataType = null;
        if(narrative!=null){
            attachedFileDataType = addAttachedFileType(narrative);
            if(attachedFileDataType != null){
                equipment.setAdditionalEquipmentsAttachment(attachedFileDataType);
            }
        }
        return equipment;
    }


    private NarrativeContract saveExtraEquipment(S2SBudgetPeriodDto periodInfo) {
        NarrativeContract narrative=null;
        List<S2SCostDto> extraEquipmentList = periodInfo.getEquipment().get(0).getExtraEquipmentList();
        if (extraEquipmentList.size() > 0) {
            AdditionalEquipmentList additionalEquipmentList = AdditionalEquipmentList.Factory
                    .newInstance();
            additionalEquipmentList.setProposalNumber(pdDoc
                    .getDevelopmentProposal().getProposalNumber());
            additionalEquipmentList.setBudgetPeriod(new BigInteger(Integer
                    .toString(periodInfo.getBudgetPeriod())));

            additionalEquipmentList
                    .setEquipmentListArray(getEquipmentListArray(extraEquipmentList));

            AdditionalEquipmentListDocument additionalEquipmentDoc = AdditionalEquipmentListDocument.Factory
                    .newInstance();
            additionalEquipmentDoc
                    .setAdditionalEquipmentList(additionalEquipmentList);
            String xmlData = additionalEquipmentDoc.xmlText();
            GenericPrintable printable = new GenericPrintable();
            printable.setXslTemplates(Collections.singletonList(additionalEquipmentAttachmentNonFedStyleSheet));
            printable.setName("");
            printable.setXml(xmlData);
            try {
                KcFile printData = s2SPrintingService
                        .print(printable);
                final String name = pdDoc.getDevelopmentProposal()
                        .getProposalNumber() + periodInfo.getBudgetPeriod()
                        + "_ADDITIONAL_EQUIPMENT";
                narrative = saveNarrative(printData.getData(),
                        ADDITIONAL_EQUIPMENT_NARRATIVE_TYPE_CODE, name + ".pdf",
                        name);
            } catch (S2SException e) {
                LOG.error(e.getMessage(), e);
            }
        }
        return narrative;
    }
    private gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList.EquipmentList[] getEquipmentListArray(
            List<S2SCostDto> extraEquipmentArrayList) {
        List<AdditionalEquipmentList.EquipmentList> additionalEquipmentListList = new ArrayList<>();
        AdditionalEquipmentList.EquipmentList equipmentList = null;
        for (S2SCostDto costInfo : extraEquipmentArrayList) {
            equipmentList = AdditionalEquipmentList.EquipmentList.Factory
                    .newInstance();
            equipmentList.setFundsRequested(costInfo.getCost()
                    .bigDecimalValue());            
            equipmentList.setNonFederal(costInfo.getCostSharing().bigDecimalValue());
            equipmentList.setTotalFedNonFed(costInfo.getCost().add(costInfo.getCostSharing()).bigDecimalValue());        
    
            equipmentList
                    .setEquipmentItem(costInfo.getDescription() != null ? costInfo
                            .getDescription()
                            : costInfo.getCategory());
            additionalEquipmentListList.add(equipmentList);
        }
        return additionalEquipmentListList
                .toArray(new gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList.EquipmentList[0]);
    }
    private NarrativeContract saveExtraKeyPersons(S2SBudgetPeriodDto periodInfo) {
        NarrativeContract extraKPNarrative = null;
        if (periodInfo.getExtraKeyPersons() != null && !periodInfo.getExtraKeyPersons().isEmpty()) {
            ExtraKeyPersonListDocument  extraKeyPersonListDocument = ExtraKeyPersonListDocument.Factory.newInstance();
            ExtraKeyPersonList extraKeyPersonList = ExtraKeyPersonList.Factory.newInstance(); 
            extraKeyPersonList.setProposalNumber(pdDoc.getDevelopmentProposal().getProposalNumber());
            extraKeyPersonList.setBudgetPeriod(new BigInteger(""+periodInfo.getBudgetPeriod()));
            extraKeyPersonList.setKeyPersonsArray(getExtraKeyPersons(periodInfo.getExtraKeyPersons()));
            extraKeyPersonListDocument.setExtraKeyPersonList(extraKeyPersonList);
            String xmlData = extraKeyPersonListDocument.xmlText();
            GenericPrintable printable = new GenericPrintable();
            printable.setXslTemplates(Collections.singletonList(extraKeyPersonAttachmentNonFedStyleSheet));
            printable.setName("");
            printable.setXml(xmlData);
            try {
                KcFile printData = s2SPrintingService.print(printable);
                final String name = pdDoc.getDevelopmentProposal().getProposalNumber() +"_" + periodInfo.getBudgetPeriod() + "_" + EXTRA_KEYPERSONS;
                extraKPNarrative = saveNarrative(printData.getData(), "" + EXTRA_KEYPERSONS_TYPE, name + ".pdf", name);
            } catch (S2SException e) {
                LOG.error(e.getMessage(), e);
            }
        }
        return extraKPNarrative;
    }
    private gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons[] getExtraKeyPersons(List<S2SKeyPersonDto> keyPersonList) {
        List<gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons> keypersonslist = new ArrayList<>();
        for(S2SKeyPersonDto keyPersonInfo : keyPersonList){
            gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons keyPerson = gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons.Factory.newInstance();
            keyPerson.setFirstName(keyPersonInfo.getFirstName());
            keyPerson.setMiddleName(keyPersonInfo.getMiddleName());
            keyPerson.setLastName(keyPersonInfo.getLastName());
            keyPerson.setProjectRole(keyPersonInfo.getRole());
            keyPerson.setCompensation(getExtraKeyPersonCompensation(keyPersonInfo));
            keypersonslist.add(keyPerson);
        }
        return keypersonslist.toArray(new gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons[0]);
    }
    private Compensation getExtraKeyPersonCompensation(S2SKeyPersonDto keyPersonInfo) {
        ExtraKeyPersonList.KeyPersons.Compensation compensation = ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons.Compensation.Factory.newInstance();
        compensation.setAcademicMonths(keyPersonInfo.getCompensation().getAcademicMonths().bigDecimalValue());
        compensation.setCalendarMonths(keyPersonInfo.getCompensation().getCalendarMonths().bigDecimalValue());
        compensation.setSummerMonths(keyPersonInfo.getCompensation().getSummerMonths().bigDecimalValue());

         compensation.setBaseSalary(keyPersonInfo.getCompensation().getBaseSalary().bigDecimalValue());
        compensation.setFringeBenefits(keyPersonInfo.getCompensation().getFringe().add(keyPersonInfo.getCompensation().getFringeCostSharing() != null ? keyPersonInfo.getCompensation().getFringeCostSharing() : ScaleTwoDecimal.ZERO).bigDecimalValue());
        compensation.setFundsRequested(keyPersonInfo.getCompensation().getFundsRequested().bigDecimalValue());
        compensation.setRequestedSalary(keyPersonInfo.getCompensation().getRequestedSalary().add(keyPersonInfo.getCompensation().getCostSharingAmount() != null ? keyPersonInfo.getCompensation().getCostSharingAmount() : ScaleTwoDecimal.ZERO).bigDecimalValue());
        compensation.setNonFederal(keyPersonInfo.getCompensation().getNonFundsRequested().bigDecimalValue());
        if (keyPersonInfo.getCompensation().getFundsRequested() != null){
            compensation.setTotalFedNonFed(keyPersonInfo.getCompensation().getFundsRequested().add(keyPersonInfo.getCompensation().getNonFundsRequested()).bigDecimalValue());
        }else {
            compensation.setTotalFedNonFed(keyPersonInfo.getCompensation().getNonFundsRequested().bigDecimalValue());
        }
        
        return compensation;
    }

    /**
     * This method gets Travel cost information including DomesticTravelCost,ForeignTravelCost and TotalTravelCost in the
     * BudgetYearDataType based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return Travel cost details corresponding to the BudgetPeriodInfo object.
     */
    private Travel getTravel(S2SBudgetPeriodDto periodInfo) {

        Travel travel = Travel.Factory.newInstance();
        if (periodInfo != null) {
            TotalDataType total = TotalDataType.Factory.newInstance();
            if (periodInfo.getDomesticTravelCost() != null) {
                total.setFederal(periodInfo.getDomesticTravelCost().bigDecimalValue());
            }
            if (periodInfo.getDomesticTravelCostSharing() != null) {
                total.setNonFederal(periodInfo.getDomesticTravelCostSharing().bigDecimalValue());
                if (periodInfo.getDomesticTravelCost() != null) {
                    total.setTotalFedNonFed(periodInfo.getDomesticTravelCost().add(periodInfo.getDomesticTravelCostSharing())
                            .bigDecimalValue());
                }
                else {
                    total.setTotalFedNonFed(periodInfo.getDomesticTravelCostSharing().bigDecimalValue());
                }
            }
            travel.setDomesticTravelCost(total);
            TotalDataType totalForeign = TotalDataType.Factory.newInstance();
            if (periodInfo.getForeignTravelCost() != null) {
                totalForeign.setFederal(periodInfo.getForeignTravelCost().bigDecimalValue());
            }
            if (periodInfo.getForeignTravelCostSharing() != null) {
                totalForeign.setNonFederal(periodInfo.getForeignTravelCostSharing().bigDecimalValue());
                if (periodInfo.getForeignTravelCost() != null) {
                    totalForeign.setTotalFedNonFed(periodInfo.getForeignTravelCost().add(periodInfo.getForeignTravelCostSharing())
                            .bigDecimalValue());
                }
                else {
                    totalForeign.setTotalFedNonFed(periodInfo.getForeignTravelCostSharing().bigDecimalValue());
                }
            }
            travel.setForeignTravelCost(totalForeign);
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalTravelCost() != null) {
                summary.setFederalSummary(periodInfo.getTotalTravelCost().bigDecimalValue());
            }
            if (periodInfo.getTotalTravelCostSharing() != null) {
                summary.setNonFederalSummary(periodInfo.getTotalTravelCostSharing().bigDecimalValue());
                if (periodInfo.getTotalTravelCost() != null) {
                    summary.setTotalFedNonFedSummary(periodInfo.getTotalTravelCost().add(periodInfo.getTotalTravelCostSharing())
                            .bigDecimalValue());
                }
                else {
                    summary.setTotalFedNonFedSummary(periodInfo.getTotalTravelCostSharing().bigDecimalValue());
                }
            }
            travel.setTotalTravelCost(summary);
        }
        return travel;
    }

    /**
     * This method gets OtherPersonnel informations like PostDocAssociates,GraduateStudents,UndergraduateStudents
     * SecretarialClerical based on PersonnelType and also gets NumberOfPersonnel, ProjectRole,Compensation
     * OtherPersonnelTotalNumber and TotalOtherPersonnelFund based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return OtherPersonnel details corresponding to the BudgetPeriodInfo object.
     */
    private OtherPersonnel getOtherPersonnel(S2SBudgetPeriodDto periodInfo) {
        OtherPersonnel otherPersonnel = OtherPersonnel.Factory.newInstance();
        int OtherpersonalCount = 0;
        List<OtherPersonnelDataType> otherPersonnelList = new ArrayList<>();

        if (periodInfo != null) {
            if (periodInfo.getOtherPersonnel() != null) {
                for (S2SOtherPersonnelDto otherPersonnelInfo : periodInfo.getOtherPersonnel()) {
                    if (OTHERPERSONNEL_POSTDOC.equals(otherPersonnelInfo.getPersonnelType())) {
                        otherPersonnel.setPostDocAssociates(getPostDocAssociates(otherPersonnelInfo));
                    }
                    else if (OTHERPERSONNEL_GRADUATE.equals(otherPersonnelInfo.getPersonnelType())) {
                        otherPersonnel.setGraduateStudents(getGraduateStudents(otherPersonnelInfo));
                    }
                    else if (OTHERPERSONNEL_UNDERGRADUATE.equals(otherPersonnelInfo.getPersonnelType())) {
                        otherPersonnel.setUndergraduateStudents(getUndergraduateStudents(otherPersonnelInfo));
                    }
                    else if (OTHERPERSONNEL_SECRETARIAL.equals(otherPersonnelInfo.getPersonnelType())) {
                        otherPersonnel.setSecretarialClerical(getSecretarialClerical(otherPersonnelInfo));
                    }
                    else if (OtherpersonalCount < OTHERPERSONNEL_MAX_ALLOWED) {// Max allowed is 6
                        OtherPersonnelDataType otherPersonnelDataType = OtherPersonnelDataType.Factory.newInstance();
                        otherPersonnelDataType.setNumberOfPersonnel(otherPersonnelInfo.getNumberPersonnel());
                        otherPersonnelDataType.setProjectRole(otherPersonnelInfo.getRole());
                        otherPersonnelDataType.setCompensation(getSectBCompensationDataType(otherPersonnelInfo.getCompensation()));
                        otherPersonnelList.add(otherPersonnelDataType);
                        OtherpersonalCount++;
                    }
                }
                OtherPersonnelDataType[] otherPersonnelDataTypeArray = otherPersonnelList.toArray(new OtherPersonnelDataType[0]);
                otherPersonnel.setOtherArray(otherPersonnelDataTypeArray);

                if (periodInfo.getOtherPersonnelTotalNumber() != null) {
                    otherPersonnel.setOtherPersonnelTotalNumber(periodInfo.getOtherPersonnelTotalNumber().intValue());
                }
            }
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalOtherPersonnelFunds() != null) {
                summary.setFederalSummary(periodInfo.getTotalOtherPersonnelFunds().bigDecimalValue());
            }
            if (periodInfo.getTotalOtherPersonnelNonFunds() != null) {
                if (budget.getSubmitCostSharingFlag()) {
                    summary.setNonFederalSummary(periodInfo.getTotalOtherPersonnelNonFunds().bigDecimalValue());  
                    if (periodInfo.getTotalOtherPersonnelFunds() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalOtherPersonnelFunds().add(
                                periodInfo.getTotalOtherPersonnelNonFunds()).bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalOtherPersonnelNonFunds().bigDecimalValue());
                    }
                } else {
                    summary.setNonFederalSummary(BigDecimal.ZERO);
                    if (periodInfo.getTotalOtherPersonnelFunds() != null) {
                        summary.setTotalFedNonFedSummary(periodInfo.getTotalOtherPersonnelFunds().bigDecimalValue());
                    } else {
                        summary.setTotalFedNonFedSummary(BigDecimal.ZERO);
                    }
                }
            }
            otherPersonnel.setTotalOtherPersonnelFund(summary);
        }
        return otherPersonnel;
    }

    /**
     * This method gets the PostDocAssociates details,ProjectRole, NumberOfPersonnel,Compensation based on OtherPersonnelInfo for
     * the RRFedNonFedBudget,if it is a PostDocAssociates type.
     * 
     * @param otherPersonnel (OtherPersonnelInfo) other personnel entry.
     * @return PostDocAssociates details corresponding to the OtherPersonnelInfo object.
     */
    private PostDocAssociates getPostDocAssociates(S2SOtherPersonnelDto otherPersonnel) {

        PostDocAssociates postDoc = PostDocAssociates.Factory.newInstance();
        if (otherPersonnel != null) {
            postDoc.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
            postDoc.setProjectRole(otherPersonnel.getRole());
            postDoc.setCompensation(getSectBCompensationDataType(otherPersonnel.getCompensation()));
        }
        return postDoc;
    }

    /**
     * This method gets the GraduateStudents details,ProjectRole, NumberOfPersonnel,Compensation based on OtherPersonnelInfo for the
     * RRFedNonFedBudget, if it is a GraduateStudents type.
     * 
     * @param otherPersonnel (OtherPersonnelInfo) other personnel entry.
     * @return GraduateStudents details corresponding to the OtherPersonnelInfo object.
     */
    private GraduateStudents getGraduateStudents(S2SOtherPersonnelDto otherPersonnel) {

        GraduateStudents graduate = GraduateStudents.Factory.newInstance();
        if (otherPersonnel != null) {
            graduate.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
            graduate.setProjectRole(otherPersonnel.getRole());
            graduate.setCompensation(getSectBCompensationDataType(otherPersonnel.getCompensation()));
        }
        return graduate;
    }

    /**
     * This method is to get the UndergraduateStudents details,ProjectRole, NumberOfPersonnel,Compensation based on
     * OtherPersonnelInfo for the RRFedNonFedBudget,if it is a UndergraduateStudents type.
     * 
     * @param otherPersonnel (OtherPersonnelInfo) other personnel entry.
     * @return UndergraduateStudents details corresponding to the OtherPersonnelInfo object.
     */
    private UndergraduateStudents getUndergraduateStudents(S2SOtherPersonnelDto otherPersonnel) {

        UndergraduateStudents undergraduate = UndergraduateStudents.Factory.newInstance();
        if (otherPersonnel != null) {
            undergraduate.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
            undergraduate.setProjectRole(otherPersonnel.getRole());
            undergraduate.setCompensation(getSectBCompensationDataType(otherPersonnel.getCompensation()));
        }
        return undergraduate;
    }

    /**
     * This method is to get the SecretarialClerical details,ProjectRole, NumberOfPersonnel,Compensation based on OtherPersonnelInfo
     * for the RRFedNonFedBudget,if it is a SecretarialClerical type.
     * 
     * @param otherPersonnel (OtherPersonnelInfo) other personnel entry.
     * @return SecretarialClerical details corresponding to the OtherPersonnelInfo object.
     */
    private SecretarialClerical getSecretarialClerical(S2SOtherPersonnelDto otherPersonnel) {

        SecretarialClerical secretarial = SecretarialClerical.Factory.newInstance();
        if (otherPersonnel != null) {
            secretarial.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
            secretarial.setProjectRole(otherPersonnel.getRole());
            secretarial.setCompensation(getSectBCompensationDataType(otherPersonnel.getCompensation()));
        }
        return secretarial;
    }

    /**
     * This method gets SectBCompensationDataType details AcademicMonths,CalendarMonths,FringeBenefits
     * FundsRequested,SummerMonths,and RequestedSalary based on KeyPersonInfo for the OtherPersonnel.
     * 
     * @param compensation (CompensationInfo) compensation entry.
     * @return SectBCompensationDataType corresponding to the CompensationInfo object.
     */
    private SectBCompensationDataType getSectBCompensationDataType(S2SCompensationDto compensation) {

        SectBCompensationDataType sectBCompensation = SectBCompensationDataType.Factory.newInstance();
        if (compensation != null) {
            if (compensation.getAcademicMonths() != null) {
                sectBCompensation.setAcademicMonths(compensation.getAcademicMonths().bigDecimalValue());
            }
            if (compensation.getCalendarMonths() != null) {
                sectBCompensation.setCalendarMonths(compensation.getCalendarMonths().bigDecimalValue());
            }
            if (compensation.getSummerMonths() != null) {
                sectBCompensation.setSummerMonths(compensation.getSummerMonths().bigDecimalValue());
            }
            if (compensation.getFringe() != null) {
                sectBCompensation.setFringeBenefits(compensation.getFringe().add(compensation.getFringeCostSharing() != null ? compensation.getFringeCostSharing() : ScaleTwoDecimal.ZERO).bigDecimalValue());
            }
            if (compensation.getRequestedSalary() != null) {
                sectBCompensation.setRequestedSalary(compensation.getRequestedSalary().add(compensation.getCostSharingAmount() != null ? compensation.getCostSharingAmount() : ScaleTwoDecimal.ZERO).bigDecimalValue());
            }
            TotalDataType totalDataType = TotalDataType.Factory.newInstance();
            if (compensation.getFundsRequested() != null) {
                totalDataType.setFederal(compensation.getFundsRequested().bigDecimalValue());
            }
            if (compensation.getNonFundsRequested() != null) {
                if (budget.getSubmitCostSharingFlag()) {
                    totalDataType.setNonFederal(compensation.getNonFundsRequested().bigDecimalValue());
                    if (compensation.getFundsRequested() != null && compensation.getNonFundsRequested() != null) {
                        totalDataType.setTotalFedNonFed(compensation.getFundsRequested().add(compensation.getNonFundsRequested())
                                .bigDecimalValue());
                    }
                } else {
                    totalDataType.setNonFederal(BigDecimal.ZERO);
                    if (compensation.getFundsRequested() != null && compensation.getNonFundsRequested() != null) {
                        totalDataType.setTotalFedNonFed(compensation.getFundsRequested().bigDecimalValue());
                    }
                }
            }
            sectBCompensation.setOtherTotal(totalDataType);
        }
        return sectBCompensation;
    }

    /**
     * This method gets KeyPersons details such as Name,ProjectRole,Compensation,TotalFundForAttachedKeyPersons
     * TotalFundForKeyPersons and AttachedKeyPersons based on BudgetPeriodInfo for the RRFedNonFedBudget.
     * 
     * @param periodInfo (BudgetPeriodInfo) budget period entry.
     * @return KeyPersons details corresponding to the BudgetPeriodInfo object.
     */
    private KeyPersons getKeyPersons(S2SBudgetPeriodDto periodInfo) {
        KeyPersons keyPersons = KeyPersons.Factory.newInstance();
        if (periodInfo != null) {
            if (periodInfo.getKeyPersons() != null) {
                List<KeyPersonDataType> keyPersonList = new ArrayList<>();
                int keyPersonCount = 0;
                for (S2SKeyPersonDto keyPerson : periodInfo.getKeyPersons()) {
                  if(keyPerson.getRole().equals(NID_PD_PI) || hasPersonnelBudget(keyPerson,periodInfo.getBudgetPeriod())){
                    KeyPersonDataType keyPersonDataType = KeyPersonDataType.Factory.newInstance();
                    keyPersonDataType.setName(globLibV20Generator.getHumanNameDataType(keyPerson));
                    if(keyPerson.getKeyPersonRole()!=null){
                        keyPersonDataType.setProjectRole(keyPerson.getKeyPersonRole());
                    }
                    else {
                        keyPersonDataType.setProjectRole(keyPerson.getRole());
                    }
                    keyPersonDataType.setCompensation(getCompensation(keyPerson, periodInfo.getBudgetPeriod()));
                    keyPersonList.add(keyPersonDataType);
                    keyPersonCount++;
                    LOG.info("keyPersonCount:" + keyPersonCount);
                }
                }
                keyPersons.setKeyPersonArray(keyPersonList.toArray(new KeyPersonDataType[0]));
            }
            SummaryDataType summary = SummaryDataType.Factory.newInstance();
            if (periodInfo.getTotalFundsKeyPersons() != null) {
                summary.setFederalSummary(periodInfo.getTotalFundsKeyPersons().bigDecimalValue());
            }
            if (periodInfo.getTotalNonFundsKeyPersons() != null) {
                summary.setNonFederalSummary(periodInfo.getTotalNonFundsKeyPersons().bigDecimalValue());
                if (periodInfo.getTotalFundsKeyPersons() != null) {
                    summary.setTotalFedNonFedSummary(periodInfo.getTotalFundsKeyPersons().add(
                            periodInfo.getTotalNonFundsKeyPersons()).bigDecimalValue());
                }
                else {
                    summary.setTotalFedNonFedSummary(periodInfo.getTotalNonFundsKeyPersons().bigDecimalValue());
                }
            }
            keyPersons.setTotalFundForKeyPersons(summary);
            SummaryDataType summaryAttachedKey = SummaryDataType.Factory.newInstance();

            BigDecimal totalFederalSummary = BigDecimal.ZERO;
            BigDecimal totalNonFederalSummary = BigDecimal.ZERO;
            for(S2SKeyPersonDto keyPersonInfo : periodInfo.getExtraKeyPersons()){
                totalFederalSummary = totalFederalSummary.add(keyPersonInfo.getCompensation().getFundsRequested().bigDecimalValue());
                totalNonFederalSummary = totalNonFederalSummary.add(keyPersonInfo.getCompensation().getNonFundsRequested().bigDecimalValue());
            }
            summaryAttachedKey.setFederalSummary(totalFederalSummary);
            summaryAttachedKey.setNonFederalSummary(totalNonFederalSummary);
            summaryAttachedKey.setTotalFedNonFedSummary(totalFederalSummary.add(totalNonFederalSummary));
            keyPersons.setTotalFundForAttachedKeyPersons(summaryAttachedKey);
        }
        NarrativeContract extraKeyPersonNarr = saveExtraKeyPersons(periodInfo);
        AttachedFileDataType attachedFileDataType = null;
        if(extraKeyPersonNarr!=null){
            attachedFileDataType = addAttachedFileType(extraKeyPersonNarr);
            if(attachedFileDataType != null){
                keyPersons.setAttachedKeyPersons(attachedFileDataType);
            }
        }
        return keyPersons;
    }

    /**
     * This method gets KeyPersonCompensationDataType informations such as AcademicMonths,CalendarMonths,FringeBenefits
     * SummerMonths,RequestedSalary,FundsRequested and BaseSalary based on KeyPersonInfo for the keyPerson.
     * 
     * @param keyPerson (KeyPersonInfo) key person entry.
     * @return KeyPersonCompensationDataType corresponding to the KeyPersonInfo object.
     */
    private KeyPersonCompensationDataType getCompensation(S2SKeyPersonDto keyPerson, int budgetPeriod) {

        KeyPersonCompensationDataType keyPersonCompensation = KeyPersonCompensationDataType.Factory.newInstance();
        ScaleTwoDecimal baseSalaryByPeriod;
        if (keyPerson != null) {
            if (keyPerson.getCompensation().getAcademicMonths() != null) {
                keyPersonCompensation.setAcademicMonths(keyPerson.getCompensation().getAcademicMonths().bigDecimalValue());
            }
            if (keyPerson.getCompensation().getCalendarMonths() != null) {
                keyPersonCompensation.setCalendarMonths(keyPerson.getCompensation().getCalendarMonths().bigDecimalValue());
            }
            if (keyPerson.getCompensation().getSummerMonths() != null) {
                keyPersonCompensation.setSummerMonths(keyPerson.getCompensation().getSummerMonths().bigDecimalValue());
            }
            if (keyPerson.getCompensation().getFringe() != null) {
                keyPersonCompensation.setFringeBenefits(keyPerson.getCompensation().getFringe().add(keyPerson.getCompensation().getFringeCostSharing() != null ? keyPerson.getCompensation().getFringeCostSharing() : ScaleTwoDecimal.ZERO).bigDecimalValue());
            }
            if (keyPerson.getCompensation().getRequestedSalary() != null) {
                keyPersonCompensation.setRequestedSalary(keyPerson.getCompensation().getRequestedSalary().add(keyPerson.getCompensation().getCostSharingAmount() != null ? keyPerson.getCompensation().getCostSharingAmount() : ScaleTwoDecimal.ZERO).bigDecimalValue());
            }
            TotalDataType totalDataType = TotalDataType.Factory.newInstance();
            if (keyPerson.getCompensation().getFundsRequested() != null) {
                totalDataType.setFederal(keyPerson.getCompensation().getFundsRequested().bigDecimalValue());
            }
            if (keyPerson.getCompensation().getNonFundsRequested() != null) {
                totalDataType.setNonFederal(keyPerson.getCompensation().getNonFundsRequested().bigDecimalValue());
            }
            if (keyPerson.getCompensation().getFundsRequested() != null && keyPerson.getCompensation().getNonFundsRequested() != null) {
                totalDataType.setTotalFedNonFed(keyPerson.getCompensation().getFundsRequested().add(keyPerson.getCompensation().getNonFundsRequested())
                        .bigDecimalValue());
            }
            keyPersonCompensation.setTotal(totalDataType);
            if (pdDoc.getDevelopmentProposal().getBudgets() != null) {
                baseSalaryByPeriod = keyPerson.getCompensation().getBaseSalary();
                if (baseSalaryByPeriod != null) {
                    keyPersonCompensation.setBaseSalary(baseSalaryByPeriod.bigDecimalValue());
                }
            }
            else {
                if (keyPerson.getCompensation().getBaseSalary() != null) {
                    keyPersonCompensation.setBaseSalary(keyPerson.getCompensation().getBaseSalary().bigDecimalValue());
                }
            }
        }
        return keyPersonCompensation;
    }

    /**
     * This method creates {@link XmlObject} of type {@link RRFedNonFedBudgetDocument} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public RRFedNonFedBudgetDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getRRFedNonFedBudget();
    }

    public S2SPrintingService getS2SPrintingService() {
        return s2SPrintingService;
    }

    public void setS2SPrintingService(S2SPrintingService s2SPrintingService) {
        this.s2SPrintingService = s2SPrintingService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    public Resource getAdditionalEquipmentAttachmentNonFedStyleSheet() {
        return additionalEquipmentAttachmentNonFedStyleSheet;
    }

    public void setAdditionalEquipmentAttachmentNonFedStyleSheet(Resource additionalEquipmentAttachmentNonFedStyleSheet) {
        this.additionalEquipmentAttachmentNonFedStyleSheet = additionalEquipmentAttachmentNonFedStyleSheet;
    }

    public Resource getExtraKeyPersonAttachmentNonFedStyleSheet() {
        return extraKeyPersonAttachmentNonFedStyleSheet;
    }

    public void setExtraKeyPersonAttachmentNonFedStyleSheet(Resource extraKeyPersonAttachmentNonFedStyleSheet) {
        this.extraKeyPersonAttachmentNonFedStyleSheet = extraKeyPersonAttachmentNonFedStyleSheet;
    }

    @Override
    public Attachments getMappedAttachments(RRFedNonFedBudgetDocument form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            final BudgetYear1DataType budget1 = form.getRRFedNonFedBudget().getBudgetYear1();
            if (budget1 != null) {
                final AttachedFileDataType budgetJustificationAttachment = budget1.getBudgetJustificationAttachment();
                if (budgetJustificationAttachment != null && a.getContentId().equals(budgetJustificationAttachment.getFileLocation().getHref())) {
                    return entry("RR_FedNonFedBudget_P4.optionalFile", a);
                }

                final KeyPersons keyPersons = budget1.getKeyPersons();
                if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[0].P1.optionalFile", a);
                }

                final Equipment equipment = budget1.getEquipment();
                if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[0].P2.optionalFile", a);
                }
            }

            final BudgetYearDataType budget2 = form.getRRFedNonFedBudget().getBudgetYear2();
            if (budget2 != null) {
                final KeyPersons keyPersons = budget2.getKeyPersons();
                if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[1].P1.optionalFile", a);
                }

                final Equipment equipment = budget2.getEquipment();
                if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[1].P2.optionalFile", a);
                }
            }

            final BudgetYearDataType budget3 = form.getRRFedNonFedBudget().getBudgetYear3();
            if (budget3 != null) {
                final KeyPersons keyPersons = budget3.getKeyPersons();
                if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[2].P1.optionalFile", a);
                }

                final Equipment equipment = budget3.getEquipment();
                if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[2].P2.optionalFile", a);
                }
            }

            final BudgetYearDataType budget4 = form.getRRFedNonFedBudget().getBudgetYear4();
            if (budget4 != null) {
                final KeyPersons keyPersons = budget4.getKeyPersons();
                if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[3].P1.optionalFile", a);
                }

                final Equipment equipment = budget4.getEquipment();
                if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[3].P2.optionalFile", a);
                }
            }

            final BudgetYearDataType budget5 = form.getRRFedNonFedBudget().getBudgetYear5();
            if (budget5 != null) {
                final KeyPersons keyPersons = budget5.getKeyPersons();
                if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[4].P1.optionalFile", a);
                }

                final Equipment equipment = budget5.getEquipment();
                if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                    return entry("RR_FedNonFedBudget_P1.BudgetYear[4].P2.optionalFile", a);
                }
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<RRFedNonFedBudgetDocument> factory() {
        return RRFedNonFedBudgetDocument.Factory;
    }
}
