/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.budget;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.coeus.common.api.org.OrganizationRepositoryService;
import org.kuali.coeus.common.api.person.KcPersonContract;
import org.kuali.coeus.common.api.person.KcPersonRepositoryService;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.common.api.sponsor.hierarchy.SponsorHierarchyService;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.common.budget.api.core.category.BudgetCategoryMapContract;
import org.kuali.coeus.common.budget.api.core.category.BudgetCategoryMapService;
import org.kuali.coeus.common.budget.api.core.category.BudgetCategoryMappingContract;
import org.kuali.coeus.common.budget.api.modular.ModularBudgetService;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemCalculatedAmountContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetRateAndBaseContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.common.budget.api.personnel.*;
import org.kuali.coeus.common.budget.api.rate.RateClassContract;
import org.kuali.coeus.common.budget.api.rate.RateClassType;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.budget.modular.BudgetModularContract;
import org.kuali.coeus.propdev.api.budget.modular.BudgetModularIdcContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.s2sgen.api.budget.*;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.person.S2SProposalPersonService;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.kuali.coeus.s2sgen.api.budget.S2SOtherCostDto.*;

@Component("s2SBudgetInfoService")
public class S2SBudgetInfoServiceImpl implements S2SBudgetInfoService, S2SBudgetIndirectCostsService {

    private static final String CUM_OTHER_COSTS_1 = "Other 1";
    private static final String CUM_OTHER_COSTS_2 = "Other 2";
    private static final String CUM_OTHER_COSTS_3 = "Other 3";
    private static final String VALUE_UNKNOWN = "Unknown";
    private static final int MAX_KEY_PERSON_COUNT = 8;
    private static final String CATEGORY_TYPE_OTHER_DIRECT_COST = "O";
    private static final String LASALARIES = "LASALARIES";
    private static final String PERSONNEL_TYPE_GRAD = "Grad";
    private static final String PERSONNEL_TYPE_POSTDOC = "PostDoc";
    private static final String PERSONNEL_TYPE_UNDERGRAD = "UnderGrad";
    private static final String PERSONNEL_TYPE_SEC = "Sec";
    private static final String PERSONNEL_TYPE_OTHER = "Other";
    private static final String PERSONNEL_TYPE_OTHER_PROFESSIONALS = "Other Professionals";
    private static final String PERSONNEL_TYPE_ALLOCATED_ADMIN_SUPPORT = "Allocated Admin Support";
    private static final String ROLE_GRADUATE_STUDENTS = "Graduate Students";
    private static final String ROLE_POST_DOCTORAL_ASSOCIATES = "Post Doctoral Associates";
    private static final String ROLE_GRADUATE_UNDERGRADUATE_STUDENTS = "Undergraduate Students";
    private static final String ROLE_GRADUATE_SECRETARIAL_OR_CLERICAL = "Secretarial / Clerical";
    private static final String ROLE_GRADUATE_OTHER = "Other";
    private static final String ROLE_GRADUATE_OTHER_PROFESSIONALS = "Other Professionals";
    private static final String ROLE_GRADUATE_ALLOCATED_ADMIN_SUPPORT = "Allocated Admin Support";
    private static final String TARGET_CATEGORY_CODE_01 = "01";

    private static final String DESCRIPTION_LA = "LA ";
    private static final String KEYPERSON_CO_PD_PI = "CO-PD/PI";
    private static final String NID_PD_PI = "PD/PI";
    private static final String NID_CO_PD_PI = "CO-INVESTIGATOR";
    private static final String KEYPERSON_OTHER = "Other (Specify)";
    private static final String SPONSOR = "S2S";
    private static final String PRINCIPAL_INVESTIGATOR_ROLE = "PD/PI";
    private static final BigDecimal POINT_ZERO_ONE = new ScaleTwoDecimal(0.01).bigDecimalValue();
    private static final Logger LOGGER = LogManager.getLogger(S2SBudgetInfoServiceImpl.class);

    @Autowired
    @Qualifier("budgetCategoryMapService")
    private BudgetCategoryMapService budgetCategoryMapService;

    @Autowired
    @Qualifier("s2SConfigurationService")
    private S2SConfigurationService s2SConfigurationService;

    @Autowired
    @Qualifier("budgetPersonSalaryService")
    private BudgetPersonSalaryService budgetPersonSalaryService;

    @Autowired
    @Qualifier("organizationRepositoryService")
    private OrganizationRepositoryService organizationRepositoryService;

    @Autowired
    @Qualifier("sponsorHierarchyService")
    private SponsorHierarchyService sponsorHierarchyService;

    @Autowired
    @Qualifier("kcPersonRepositoryService")
    private KcPersonRepositoryService kcPersonRepositoryService;

    @Autowired
    @Qualifier("tbnPersonService")
    private TbnPersonService tbnPersonService;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    @Autowired
    @Qualifier("s2SDateTimeService")
    private S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("s2SProposalPersonService")
    private S2SProposalPersonService s2SProposalPersonService;

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    private S2SCommonBudgetService s2SCommonBudgetService;

    @Autowired
    @Qualifier("modularBudgetService")
    private ModularBudgetService modularBudgetService;

    @Autowired
    @Qualifier("s2SBudgetCategoryMapService")
    private S2SBudgetCategoryMapService s2SBudgetCategoryMapService;

    @Override
    public S2SBudgetDto getBudgetInfo(ProposalDevelopmentDocumentContract pdDoc) {
        Map<String, S2SOtherCostDto> cumulativeOtherCostsGrouping = new HashMap<>();


        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
        final S2SBudgetDto budgetSummaryInfo = new S2SBudgetDto();
        if (budget == null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("ICDB: budgetSummaryInfo is null");
            }
            return budgetSummaryInfo;
        }

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("ICDB: calling getBudgetPeriods");
        }

        final List<S2SBudgetPeriodDto> budgetPeriodInfos = getBudgetPeriods(pdDoc);

        budgetSummaryInfo.setBudgetPeriods(budgetPeriodInfos);
        budgetSummaryInfo.setCumTotalCosts(budget.getTotalCost());
        budgetSummaryInfo.setCumTotalIndirectCosts(budget.getTotalIndirectCost());
        budgetSummaryInfo.setCumTotalDirectCosts(budget.getTotalDirectCost());
        if (budget.getSubmitCostSharingFlag()) {
            budgetSummaryInfo.setCumTotalCostSharing(budget.getCostSharingAmount());
        }

        ScaleTwoDecimal totalDirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalIndirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal lineItemCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal lineItemCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fringeCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fringeCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetDetailsCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetDetailsCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetDetailsFringeCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetDetailsFringeCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totPersFunds = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totPersNonFunds = ScaleTwoDecimal.ZERO;
        String budgetCategoryTypePersonnel = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_BUDGET_CATEGORY_TYPE_PERSONNEL);
        String rateTypeSupportStaffSalaries = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_TYPE_SUPPORT_STAFF_SALARIES);
        String rateTypeAdministrativeSalaries = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_TYPE_ADMINISTRATIVE_SALARIES);

        for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
            for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                totalDirectCostSharing = totalDirectCostSharing.add(lineItem.getCostSharingAmount());
                if (lineItem
                        .getBudgetCategory()
                        .getBudgetCategoryType()
                        .getCode()
                        .equals(budgetCategoryTypePersonnel)) {
                    lineItemCost = lineItemCost.add(lineItem.getLineItemCost());
                    if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                        lineItemCostSharingAmount = lineItemCostSharingAmount.add(lineItem.getCostSharingAmount());
                    }
                }

                for (BudgetLineItemCalculatedAmountContract lineItemCalAmt : lineItem.getBudgetLineItemCalculatedAmounts()) {
                    if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                        if (lineItemCalAmt.getRateClass().getRateClassType().getCode().equals(RateClassType.OVERHEAD.getRateClassType())) {
                            totalIndirectCostSharing = totalIndirectCostSharing.add(lineItemCalAmt.getCalculatedCostSharing());
                        } else {
                            totalDirectCostSharing = totalDirectCostSharing.add(lineItemCalAmt.getCalculatedCostSharing());
                        }
                    }
                    if ((lineItemCalAmt.getRateClass().getCode().equals(
                            s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_CODE_EMPLOYEE_BENEFITS)) && !lineItemCalAmt.getRateTypeCode()
                            .equals(rateTypeSupportStaffSalaries))
                            || (lineItemCalAmt.getRateClass().getCode().equals(
                            s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_CODE_VACATION)) && !lineItemCalAmt.getRateTypeCode()
                            .equals(rateTypeAdministrativeSalaries))) {
                        if (lineItem
                                .getBudgetCategory()
                                .getBudgetCategoryType()
                                .getCode()
                                .equals(budgetCategoryTypePersonnel)) {
                            fringeCost = fringeCost.add(lineItemCalAmt.getCalculatedCost());
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                fringeCostSharingAmount = fringeCostSharingAmount.add(lineItemCalAmt.getCalculatedCostSharing());
                            }
                        }
                    }
                    if (lineItemCalAmt
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_LAB_ALLOCATION_SALARIES))) {
                        budgetDetailsCost = budgetDetailsCost.add(lineItemCalAmt.getCalculatedCost());
                        if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                            budgetDetailsCostSharingAmount = budgetDetailsCostSharingAmount.add(lineItemCalAmt
                                    .getCalculatedCostSharing());
                        }
                    }
                    if ((lineItemCalAmt
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_EMPLOYEE_BENEFITS)) && lineItemCalAmt.getRateTypeCode()
                            .equals(rateTypeSupportStaffSalaries))
                            || (lineItemCalAmt
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_VACATION)) && lineItemCalAmt.getRateTypeCode()
                            .equals(rateTypeAdministrativeSalaries))) {
                        budgetDetailsFringeCost = budgetDetailsFringeCost.add(lineItemCalAmt.getCalculatedCost());
                        if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                            budgetDetailsFringeCostSharingAmount = budgetDetailsFringeCostSharingAmount.add(lineItemCalAmt
                                    .getCalculatedCostSharing());
                        }
                    }

                }
            }
        }
        if (budget.getSubmitCostSharingFlag()) {
            budgetSummaryInfo.setCumTotalDirectCostSharing(totalDirectCostSharing);
            budgetSummaryInfo.setCumTotalIndirectCostSharing(totalIndirectCostSharing);
        }
        totPersFunds = totPersFunds.add(lineItemCost).add(fringeCost).add(budgetDetailsCost).add(budgetDetailsFringeCost);
        totPersNonFunds = totPersNonFunds.add(lineItemCostSharingAmount).add(fringeCostSharingAmount)
                .add(budgetDetailsCostSharingAmount).add(budgetDetailsFringeCostSharingAmount);

        budgetSummaryInfo.setCumTotalNonFundsForPersonnel(totPersNonFunds);
        budgetSummaryInfo.setCumTotalFundsForPersonnel(totPersFunds);

        S2SOtherDirectCostInfoDto otherDirectCostInfo;
        List<S2SOtherDirectCostInfoDto> cvOtherDirectCost;

        ScaleTwoDecimal cumAlterations = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumConsultants = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumMaterials = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPubs = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumSubAward = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumComputer = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumEquipRental = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumAll = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType1 = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType2 = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType3 = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartStipends = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartSubsistence = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartTuition = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartOther = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartTravel = ScaleTwoDecimal.ZERO;
        int cumParticipantCount = 0;
        ScaleTwoDecimal cumAlterationsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumConsultantsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumMaterialsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPubsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumSubAwardCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumComputerCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumEquipRentalCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumAllCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType1CostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType2CostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumOtherType3CostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartStipendsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartSubsistenceCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartTuitionCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartOtherCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumPartTravelCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal participantTotalCostSharing = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal totalDomesticTravel = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalForeignTravel = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalDomesticTravelNonFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalForeignTravelNonFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumTotalEquipFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal cumTotalEquipNonFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totCountOtherPersonnel = ScaleTwoDecimal.ZERO;

        for (S2SBudgetPeriodDto budgetPeriodInfo : budgetPeriodInfos) {
            cvOtherDirectCost = budgetPeriodInfo.getOtherDirectCosts();
            otherDirectCostInfo = cvOtherDirectCost.get(0);
            cumAlterations = cumAlterations.add(otherDirectCostInfo.getAlterations());
            cumConsultants = cumConsultants.add(otherDirectCostInfo.getConsultants());
            cumMaterials = cumMaterials.add(otherDirectCostInfo.getMaterials());
            cumPubs = cumPubs.add(otherDirectCostInfo.getPublications());
            cumSubAward = cumSubAward.add(otherDirectCostInfo.getSubAwards());
            cumComputer = cumComputer.add(otherDirectCostInfo.getComputer());
            cumEquipRental = cumEquipRental.add(otherDirectCostInfo.getEquipRental());
            cumAll = cumAll.add(otherDirectCostInfo.getTotalOtherDirect());

            cumPartStipends = cumPartStipends.add(otherDirectCostInfo.getPartStipends() == null ? ScaleTwoDecimal.ZERO
                    : otherDirectCostInfo.getPartStipends());
            cumPartTravel = cumPartTravel.add(otherDirectCostInfo.getPartTravel() == null ? ScaleTwoDecimal.ZERO
                    : otherDirectCostInfo.getPartTravel());
            cumPartSubsistence = cumPartSubsistence.add(otherDirectCostInfo.getPartSubsistence() == null ? ScaleTwoDecimal.ZERO
                    : otherDirectCostInfo.getPartSubsistence());
            cumPartTuition = cumPartTuition.add(otherDirectCostInfo.getPartTuition() == null ? ScaleTwoDecimal.ZERO
                    : otherDirectCostInfo.getPartTuition());
            cumPartOther = cumPartOther.add(otherDirectCostInfo.getPartOther() == null ? ScaleTwoDecimal.ZERO
                    : otherDirectCostInfo.getPartOther());
            cumParticipantCount = cumParticipantCount + otherDirectCostInfo.getParticipantTotalCount();
            if (budget.getSubmitCostSharingFlag()) {
                cumAlterationsCostSharing = cumAlterationsCostSharing.add(otherDirectCostInfo.getAlterationsCostSharing());
                cumConsultantsCostSharing = cumConsultantsCostSharing.add(otherDirectCostInfo.getConsultantsCostSharing());
                cumMaterialsCostSharing = cumMaterialsCostSharing.add(otherDirectCostInfo.getMaterialsCostSharing());
                cumPubsCostSharing = cumPubsCostSharing.add(otherDirectCostInfo.getPublicationsCostSharing());
                cumSubAwardCostSharing = cumSubAwardCostSharing.add(otherDirectCostInfo.getSubAwardsCostSharing());
                cumComputerCostSharing = cumComputerCostSharing.add(otherDirectCostInfo.getComputerCostSharing());
                cumEquipRentalCostSharing = cumEquipRentalCostSharing.add(otherDirectCostInfo.getEquipRentalCostSharing());
                cumAllCostSharing = cumAllCostSharing.add(otherDirectCostInfo.getTotalOtherDirectCostSharing());

                cumPartStipendsCostSharing = cumPartStipendsCostSharing
                        .add(otherDirectCostInfo.getPartStipendsCostSharing() == null ? ScaleTwoDecimal.ZERO
                                : otherDirectCostInfo.getPartStipendsCostSharing());
                cumPartTravelCostSharing = cumPartTravelCostSharing
                        .add(otherDirectCostInfo.getPartTravelCostSharing() == null ? ScaleTwoDecimal.ZERO : otherDirectCostInfo
                                .getPartTravelCostSharing());
                cumPartSubsistenceCostSharing = cumPartSubsistenceCostSharing.add(otherDirectCostInfo
                        .getPartSubsistenceCostSharing() == null ? ScaleTwoDecimal.ZERO : otherDirectCostInfo
                        .getPartSubsistenceCostSharing());
                cumPartTuitionCostSharing = cumPartTuitionCostSharing
                        .add(otherDirectCostInfo.getPartTuitionCostSharing() == null ? ScaleTwoDecimal.ZERO : otherDirectCostInfo
                                .getPartTuitionCostSharing());

                cumPartOtherCostSharing = cumPartOtherCostSharing
                        .add(otherDirectCostInfo.getPartOtherCostSharing() == null ? ScaleTwoDecimal.ZERO : otherDirectCostInfo
                                .getPartOtherCostSharing());
            } else {
                cumAlterationsCostSharing = cumAlterationsCostSharing.add(ScaleTwoDecimal.ZERO);
                cumConsultantsCostSharing = cumConsultantsCostSharing.add(ScaleTwoDecimal.ZERO);
                cumMaterialsCostSharing = cumMaterialsCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPubsCostSharing = cumPubsCostSharing.add(ScaleTwoDecimal.ZERO);
                cumSubAwardCostSharing = cumSubAwardCostSharing.add(ScaleTwoDecimal.ZERO);
                cumComputerCostSharing = cumComputerCostSharing.add(ScaleTwoDecimal.ZERO);
                cumEquipRentalCostSharing = cumEquipRentalCostSharing.add(ScaleTwoDecimal.ZERO);
                cumAllCostSharing = cumAllCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPartStipendsCostSharing = cumPartStipendsCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPartTravelCostSharing = cumPartTravelCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPartSubsistenceCostSharing = cumPartSubsistenceCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPartTuitionCostSharing = cumPartTuitionCostSharing.add(ScaleTwoDecimal.ZERO);
                cumPartOtherCostSharing = cumPartOtherCostSharing.add(ScaleTwoDecimal.ZERO);
            }
            totalDomesticTravel = totalDomesticTravel.add(budgetPeriodInfo.getDomesticTravelCost());
            totalForeignTravel = totalForeignTravel.add(budgetPeriodInfo.getForeignTravelCost());
            totalDomesticTravelNonFund = totalDomesticTravelNonFund.add(budgetPeriodInfo.getDomesticTravelCostSharing());
            totalForeignTravelNonFund = totalForeignTravelNonFund.add(budgetPeriodInfo.getForeignTravelCostSharing());
            totCountOtherPersonnel = totCountOtherPersonnel.add(budgetPeriodInfo.getOtherPersonnelTotalNumber());

            for (S2SEquipmentDto equipmentInfo : budgetPeriodInfo.getEquipment()) {
                cumTotalEquipFund = cumTotalEquipFund.add(equipmentInfo.getTotalFund());
                cumTotalEquipNonFund = cumTotalEquipNonFund.add(equipmentInfo.getTotalNonFund());
            }

            for (int i = 0; i < otherDirectCostInfo.getOtherCosts().size(); i++) {
                final S2SOtherCostDto oc = otherDirectCostInfo.getOtherCosts().get(i);
                String key = oc.getDescription();
                if (cumulativeOtherCostsGrouping.get(key) == null) {
                    S2SOtherCostDto cumulativeCost = new S2SOtherCostDto();
                    cumulativeCost.setDescription(key);
                    cumulativeCost.setCost(oc.getCost());
                    cumulativeCost.setCostSharing(oc.getCostSharing());
                    cumulativeOtherCostsGrouping.put(key, cumulativeCost);
                } else {
                    S2SOtherCostDto cumulativeCost = cumulativeOtherCostsGrouping.get(key);
                    cumulativeCost.setCost(cumulativeCost.getCost().add(oc.getCost()));
                    cumulativeCost.setCostSharing(cumulativeCost.getCostSharing().add(oc.getCostSharing()));
                    cumulativeOtherCostsGrouping.put(key, cumulativeCost);
                }
            }
        }


        budgetSummaryInfo.setCumDomesticTravelNonFund(totalDomesticTravelNonFund);
        budgetSummaryInfo.setCumForeignTravelNonFund(totalForeignTravelNonFund);
        budgetSummaryInfo.setCumTravelNonFund(totalDomesticTravelNonFund.add(totalForeignTravelNonFund));
        budgetSummaryInfo.setCumDomesticTravel(totalDomesticTravel);
        budgetSummaryInfo.setCumForeignTravel(totalForeignTravel);
        budgetSummaryInfo.setCumTravel(totalDomesticTravel.add(totalForeignTravel));
        budgetSummaryInfo.setPartOtherCost(cumPartOther);
        budgetSummaryInfo.setPartStipendCost(cumPartStipends);
        budgetSummaryInfo.setPartTravelCost(cumPartTravel);
        budgetSummaryInfo.setPartSubsistence(cumPartSubsistence);
        budgetSummaryInfo.setPartTuition(cumPartTuition);
        budgetSummaryInfo.setParticipantCount(cumParticipantCount);

        if (budget.getSubmitCostSharingFlag()) {
            // add costSaring for fedNonFedBudget report
            budgetSummaryInfo.setPartOtherCostSharing(cumPartOtherCostSharing);
            budgetSummaryInfo.setPartStipendCostSharing(cumPartStipendsCostSharing);
            budgetSummaryInfo.setPartTravelCostSharing(cumPartTravelCostSharing);
            budgetSummaryInfo.setPartSubsistenceCostSharing(cumPartSubsistenceCostSharing);
            budgetSummaryInfo.setPartTuitionCostSharing(cumPartTuitionCostSharing);
        }

        S2SOtherDirectCostInfoDto summaryOtherDirectCostInfo = new S2SOtherDirectCostInfoDto();
        summaryOtherDirectCostInfo.setAlterations(cumAlterations);
        summaryOtherDirectCostInfo.setComputer(cumComputer);
        summaryOtherDirectCostInfo.setConsultants(cumConsultants);
        summaryOtherDirectCostInfo.setMaterials(cumMaterials);
        summaryOtherDirectCostInfo.setPublications(cumPubs);
        summaryOtherDirectCostInfo.setSubAwards(cumSubAward);
        summaryOtherDirectCostInfo.setEquipRental(cumEquipRental);
        summaryOtherDirectCostInfo.setTotalOtherDirect(cumAll);

        summaryOtherDirectCostInfo.setPartStipends(cumPartStipends);
        summaryOtherDirectCostInfo.setPartTravel(cumPartTravel);
        summaryOtherDirectCostInfo.setPartSubsistence(cumPartSubsistence);
        summaryOtherDirectCostInfo.setPartTuition(cumPartTuition);
        summaryOtherDirectCostInfo.setPartOther(cumPartOther);
        summaryOtherDirectCostInfo.setParticipantTotal(cumPartStipends.add(cumPartTravel.add(cumPartOther.add(cumPartSubsistence
                .add(cumPartTravel)))));
        summaryOtherDirectCostInfo.setParticipantTotalCount(cumParticipantCount);
        // start add costSaring for fedNonFedBudget report
        summaryOtherDirectCostInfo.setAlterationsCostSharing(cumAlterationsCostSharing);
        summaryOtherDirectCostInfo.setComputerCostSharing(cumComputerCostSharing);
        summaryOtherDirectCostInfo.setConsultantsCostSharing(cumConsultantsCostSharing);
        summaryOtherDirectCostInfo.setMaterialsCostSharing(cumMaterialsCostSharing);
        summaryOtherDirectCostInfo.setPublicationsCostSharing(cumPubsCostSharing);
        summaryOtherDirectCostInfo.setSubAwardsCostSharing(cumSubAwardCostSharing);
        summaryOtherDirectCostInfo.setEquipRentalCostSharing(cumEquipRentalCostSharing);
        summaryOtherDirectCostInfo.setTotalOtherDirectCostSharing(cumAllCostSharing);

        summaryOtherDirectCostInfo.setPartStipendsCostSharing(cumPartStipendsCostSharing);
        summaryOtherDirectCostInfo.setPartTravelCostSharing(cumPartTravelCostSharing);
        summaryOtherDirectCostInfo.setPartTuitionCostSharing(cumPartTuitionCostSharing);
        summaryOtherDirectCostInfo.setPartSubsistenceCostSharing(cumPartSubsistenceCostSharing);
        summaryOtherDirectCostInfo.setPartOtherCostSharing(cumPartOtherCostSharing);

        participantTotalCostSharing = participantTotalCostSharing.add(cumPartStipendsCostSharing);
        participantTotalCostSharing = participantTotalCostSharing.add(cumPartTravelCostSharing);
        participantTotalCostSharing = participantTotalCostSharing.add(cumPartOtherCostSharing);
        participantTotalCostSharing = participantTotalCostSharing.add(cumPartTuitionCostSharing);
        participantTotalCostSharing = participantTotalCostSharing.add(cumPartSubsistenceCostSharing);
        summaryOtherDirectCostInfo.setParticipantTotalCostSharing(participantTotalCostSharing);

        List<S2SOtherCostDto> otherCosts = new ArrayList<>();
        S2SOtherCostDto otherDirectCost = cumulativeOtherCostsGrouping.get(OTHER_DIRECT_COSTS);
        S2SOtherCostDto humanFetalTissueCosts = cumulativeOtherCostsGrouping.get(HUMAN_FETAL_TISSUE_COSTS);
        S2SOtherCostDto dataManagementAndSharingCosts = cumulativeOtherCostsGrouping.get(DATA_MANAGEMENT_AND_SHARING_COSTS);
        S2SOtherCostDto allOtherCosts = cumulativeOtherCostsGrouping.get(ALL_OTHER_COSTS);

        if (otherDirectCost != null && otherDirectCost.getCost().isNonZero()) {
            otherCosts.add(cumulativeOtherCostsGrouping.get(OTHER_DIRECT_COSTS));
        }
        if (humanFetalTissueCosts != null && humanFetalTissueCosts.getCost().isNonZero()) {
            otherCosts.add(cumulativeOtherCostsGrouping.get(HUMAN_FETAL_TISSUE_COSTS));
        }
        if (dataManagementAndSharingCosts != null && dataManagementAndSharingCosts.getCost().isNonZero()) {
            otherCosts.add(cumulativeOtherCostsGrouping.get(DATA_MANAGEMENT_AND_SHARING_COSTS));
        }
        if (allOtherCosts != null && allOtherCosts.getCost().isNonZero()) {
            otherCosts.add(cumulativeOtherCostsGrouping.get(ALL_OTHER_COSTS));
        }
        summaryOtherDirectCostInfo.setOtherCosts(otherCosts);

        List<S2SOtherDirectCostInfoDto> cvCumOtherDirectCost = new ArrayList<>(); // all
        // periods
        cvCumOtherDirectCost.add(summaryOtherDirectCostInfo);
        budgetSummaryInfo.setOtherDirectCosts(cvCumOtherDirectCost);

        budgetSummaryInfo.setCumEquipmentFunds(cumTotalEquipFund);
        budgetSummaryInfo.setCumEquipmentNonFunds(cumTotalEquipNonFund);

        // hardcoded
        budgetSummaryInfo.setCumFee(ScaleTwoDecimal.ZERO);

        ScaleTwoDecimal totSrFunds = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totSrNonFunds = ScaleTwoDecimal.ZERO;

        for (S2SBudgetPeriodDto budgetPeriodInfo : budgetPeriodInfos) {
            totSrFunds = totSrFunds.add(budgetPeriodInfo.getTotalFundsKeyPersons());
            totSrNonFunds = totSrNonFunds.add(budgetPeriodInfo.getTotalNonFundsKeyPersons());
        }

        budgetSummaryInfo.setCumTotalFundsForSrPersonnel(totSrFunds);
        budgetSummaryInfo.setCumTotalNonFundsForSrPersonnel(totSrNonFunds);

        // other personnel - funds for other personnel will be difference
        // between total person funds and senior person funds
        budgetSummaryInfo.setCumTotalFundsForOtherPersonnel(totPersFunds.subtract(totSrFunds));
        budgetSummaryInfo.setCumTotalNonFundsForOtherPersonnel(totPersNonFunds.subtract(totSrNonFunds));
        budgetSummaryInfo.setCumNumOtherPersonnel(totCountOtherPersonnel);

        budgetSummaryInfo.setCumProjectIncome(budgetPeriodInfos.stream()
                .map(S2SBudgetPeriodDto::getTotalProjectIncome)
                .reduce(ScaleTwoDecimal.ZERO, ScaleTwoDecimal::add));

        return budgetSummaryInfo;
    }

    private List<S2SBudgetPeriodDto> getBudgetPeriods(ProposalDevelopmentDocumentContract pdDoc) {
        List<S2SBudgetPeriodDto> budgetPeriods = new ArrayList<>();
        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());

        if (budget == null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("ICDB: budget is null ");
            }
            return budgetPeriods;
        }

        if (LOGGER.isDebugEnabled()) {
            var bp = budget.getBudgetPeriods();
            var count = bp == null ? "null" : String.valueOf(bp.size());
            LOGGER.debug("ICDB: Processing " + count + " budget periods. budgetId = " + budget.getBudgetId());
        }

        for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
            S2SBudgetPeriodDto bpData = new S2SBudgetPeriodDto();
            ScaleTwoDecimal totalCostSharing = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalDirectCostSharing = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalIndirectCostSharing = ScaleTwoDecimal.ZERO;
            bpData.setLineItemCount(budgetPeriod.getBudgetLineItems().size());
            if (budget.getSubmitCostSharingFlag()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    if (lineItem.getSubmitCostSharingFlag()) {
                        totalCostSharing = totalCostSharing.add(lineItem.getCostSharingAmount());
                    }
                    for (BudgetLineItemCalculatedAmountContract lineItemCalculatedAmt : lineItem.getBudgetLineItemCalculatedAmounts()) {
                        if (lineItem.getSubmitCostSharingFlag()) {
                            if (lineItemCalculatedAmt.getRateClass().getRateClassType().getCode()
                                    .equals(RateClassType.OVERHEAD.getRateClassType())) {
                                totalIndirectCostSharing = totalIndirectCostSharing.add(lineItemCalculatedAmt
                                        .getCalculatedCostSharing());
                            } else {
                                totalDirectCostSharing = totalDirectCostSharing.add(lineItemCalculatedAmt
                                        .getCalculatedCostSharing());
                            }
                        }
                    }
                }
                totalDirectCostSharing = totalDirectCostSharing.add(totalCostSharing);
            }


            bpData.setBudgetPeriod(budgetPeriod.getBudgetPeriod());
            bpData.setStartDate(new java.sql.Date(budgetPeriod.getStartDate().getTime()));
            bpData.setEndDate(new java.sql.Date(budgetPeriod.getEndDate().getTime()));
            bpData.setTotalCosts(budgetPeriod.getTotalCost());
            bpData.setDirectCostsTotal(budgetPeriod.getTotalDirectCost());
            bpData.setTotalIndirectCost(budgetPeriod.getTotalIndirectCost());
            bpData.setCostSharingAmount(budgetPeriod.getCostSharingAmount());
            if (budget.getSubmitCostSharingFlag()) {
                bpData.setTotalDirectCostSharing(totalDirectCostSharing);
                bpData.setTotalIndirectCostSharing(totalIndirectCostSharing);
            }
            bpData.setCognizantFedAgency(getCognizantFedAgency(pdDoc.getDevelopmentProposal()));

            List<? extends BudgetModularContract> modularAmounts = modularBudgetService.getModularBudgetAmounts(budget);

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("ICDB: Calling getIndirectCosts. budgetId = " + budget.getBudgetId() + ", budgetPeriodId = " + budgetPeriod.getBudgetPeriodId() + ", modularAmounts = " + modularAmounts);
            }

            bpData.setIndirectCosts(getIndirectCosts(budget, budgetPeriod, modularAmounts));
            bpData.setEquipment(getEquipment(budgetPeriod, budget));
            bpData.setOtherDirectCosts(getOtherDirectCosts(budgetPeriod, budget));
            if (bpData.getOtherDirectCosts().size() > 0) {
                S2SOtherDirectCostInfoDto otherCostInfo = bpData.getOtherDirectCosts().get(0);
                bpData.setDomesticTravelCost(otherCostInfo.getDomTravel());

                bpData.setForeignTravelCost(otherCostInfo.getForeignTravel());
                bpData.setTotalTravelCost(otherCostInfo.getTotTravel());
                if (budget.getSubmitCostSharingFlag()) {
                    // add costSaring for fedNonFedBudget report
                    bpData.setDomesticTravelCostSharing(otherCostInfo.getDomTravelCostSharing());
                    bpData.setForeignTravelCostSharing(otherCostInfo.getForeignTravelCostSharing());
                    bpData.setTotalTravelCostSharing(otherCostInfo.getTotTravelCostSharing());
                }
                // participants
                bpData.setPartOtherCost(otherCostInfo.getPartOther());
                bpData.setPartStipendCost(otherCostInfo.getPartStipends());
                bpData.setPartTravelCost(otherCostInfo.getPartTravel());
                bpData.setPartSubsistence(otherCostInfo.getPartSubsistence());
                bpData.setPartTuition(otherCostInfo.getPartTuition());
                bpData.setParticipantCount(otherCostInfo.getParticipantTotalCount());
                if (budget.getSubmitCostSharingFlag()) {
                    bpData.setPartOtherCostSharing(otherCostInfo.getPartOtherCostSharing());
                    bpData.setPartStipendCostSharing(otherCostInfo.getPartStipendsCostSharing());
                    bpData.setPartTravelCostSharing(otherCostInfo.getPartTravelCostSharing());
                    bpData.setPartTuitionCostSharing(otherCostInfo.getPartTuitionCostSharing());
                    bpData.setPartSubsistenceCostSharing(otherCostInfo.getPartSubsistenceCostSharing());
                }
            }

            final List<S2SKeyPersonDto> keyPersons = getKeyPersons(budgetPeriod, pdDoc, budget);
            final List<S2SKeyPersonDto> nKeyPersons = getNKeyPersons(keyPersons, MAX_KEY_PERSON_COUNT);
            final List<S2SKeyPersonDto> extraPersons = keyPersons.stream()
                    .filter(person -> hasPersonnelBudget(budgetPeriod, person))
                    .filter(seniorKp -> !nKeyPersons.contains(seniorKp))
                    .collect(Collectors.toList());

            bpData.setKeyPersons(nKeyPersons);
            bpData.setExtraKeyPersons(extraPersons);

            ScaleTwoDecimal totalKeyPersonSum = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalKeyPersonSumCostSharing = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalAttKeyPersonSum = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalAttKeyPersonSumCostSharing = ScaleTwoDecimal.ZERO;
            for (S2SKeyPersonDto keyPerson : nKeyPersons) {
                totalKeyPersonSum = totalKeyPersonSum.add(keyPerson.getCompensation().getFundsRequested());
                if (budget.getSubmitCostSharingFlag()) {
                    totalKeyPersonSumCostSharing = totalKeyPersonSumCostSharing.add(keyPerson.getCompensation().getNonFundsRequested());
                }
            }

            for (S2SKeyPersonDto keyPerson : extraPersons) {
                totalAttKeyPersonSum = totalAttKeyPersonSum.add(keyPerson.getCompensation().getFundsRequested());
                if (budget.getSubmitCostSharingFlag()) {
                    // start add costSaring for fedNonFedBudget report
                    totalAttKeyPersonSumCostSharing = totalAttKeyPersonSumCostSharing.add(keyPerson.getCompensation().getNonFundsRequested());
                }
            }
            bpData.setTotalFundsKeyPersons(totalKeyPersonSum.add(totalAttKeyPersonSum));
            bpData.setTotalFundsAttachedKeyPersons(totalAttKeyPersonSum);
            bpData.setTotalNonFundsKeyPersons(totalKeyPersonSumCostSharing.add(totalAttKeyPersonSumCostSharing));
            bpData.setTotalNonFundsAttachedKeyPersons(totalAttKeyPersonSumCostSharing);

            List<S2SOtherPersonnelDto> otherPersonnel = getOtherPersonnel(budgetPeriod, keyPersons);
            bpData.setOtherPersonnel(otherPersonnel);
            ScaleTwoDecimal otherPersonnelCount = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal otherPersonnelTotalFunds = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal otherPersonnelTotalNonFunds = ScaleTwoDecimal.ZERO;

            for (S2SOtherPersonnelDto otherPersonnelInfo : otherPersonnel) {
                otherPersonnelCount = otherPersonnelCount.add(new ScaleTwoDecimal(otherPersonnelInfo.getNumberPersonnel()));
                otherPersonnelTotalFunds = otherPersonnelTotalFunds.add(otherPersonnelInfo.getCompensation().getFundsRequested());
                otherPersonnelTotalNonFunds = otherPersonnelTotalNonFunds.add(otherPersonnelInfo.getCompensation()
                        .getNonFundsRequested());
            }
            bpData.setTotalOtherPersonnelFunds(otherPersonnelTotalFunds);
            bpData.setOtherPersonnelTotalNumber(otherPersonnelCount);
            bpData.setTotalCompensation(otherPersonnelTotalFunds.add(totalKeyPersonSum).add(totalAttKeyPersonSum));
            bpData.setTotalOtherPersonnelNonFunds(otherPersonnelTotalNonFunds);
            bpData.setTotalCompensationCostSharing(otherPersonnelTotalNonFunds.add(totalKeyPersonSumCostSharing).add(
                    totalAttKeyPersonSumCostSharing));
            bpData.setProjectIncomes(getProjectIncomes(budget, budgetPeriod.getBudgetPeriod()));
            bpData.setTotalProjectIncome(bpData.getProjectIncomes().stream()
                    .map(S2SProjectIncomeDto::getProjectIncome)
                    .reduce(ScaleTwoDecimal.ZERO, ScaleTwoDecimal::add));
            bpData.setProjectIncomesSummary(bpData.getProjectIncomes().stream()
                    .map(S2SProjectIncomeDto::getDescription)
                    .collect(Collectors.joining(";")));

            budgetPeriods.add(bpData);
        }
        return budgetPeriods;
    }

    protected String getCognizantFedAgency(DevelopmentProposalContract developmentProposal) {
        StringBuilder fedAgency = new StringBuilder();
        ProposalSiteContract applicantOrganization = developmentProposal.getApplicantOrganization();
        if (applicantOrganization != null && applicantOrganization.getOrganization() != null
                && applicantOrganization.getOrganization().getCognizantAuditor() != null) {
            fedAgency.append(organizationRepositoryService.getCognizantFedAgency(applicantOrganization.getOrganization()));
        }
        if (fedAgency.toString().length() == 0) {
            fedAgency.append(VALUE_UNKNOWN);
        }
        return fedAgency.toString();
    }

    private List<S2SProjectIncomeDto> getProjectIncomes(final ProposalDevelopmentBudgetExtContract budget, final int periodNumber) {
        return budget.getBudgetProjectIncomes().stream()
                .filter(projectIncome -> projectIncome.getBudgetPeriodNumber() == periodNumber)
                .map(projectIncome -> {
                    S2SProjectIncomeDto incomeDto = new S2SProjectIncomeDto();
                    incomeDto.setPeriodNumber(periodNumber);
                    incomeDto.setDescription(projectIncome.getDescription());
                    incomeDto.setProjectIncome(projectIncome.getProjectIncome());
                    return incomeDto;
                })
                .collect(Collectors.toList());
    }

    @Override
    public S2SIndirectCostDto getIndirectCosts(BudgetContract budget, BudgetPeriodContract budgetPeriod, List<? extends BudgetModularContract> modularAmounts) {
        List<S2SIndirectCostDetailsDto> indirectCostDetailList = new ArrayList<>();
        S2SIndirectCostDetailsDto indirectCostDetails;
        ScaleTwoDecimal baseCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal baseCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal calculatedCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal calculatedCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal appliedRate = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalIndirectCosts = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalIndirectCostSharing = ScaleTwoDecimal.ZERO;

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("ICDB: In getIndirectCosts()");
        }

        String description = "";
        boolean firstLoop = true;

        if (budget.getModularBudgetFlag()) {

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("ICDB: Processing modular budget...");
            }

            BudgetModularContract modularBudget = modularBudgetService.getModularBudgetForPeriod(modularAmounts, budgetPeriod);
            for (BudgetModularIdcContract budgetModularIdc : modularBudget.getBudgetModularIdcs()) {
                if (firstLoop) {
                    appliedRate = appliedRate.add(budgetModularIdc.getIdcRate());
                    description = budgetModularIdc.getDescription();
                    firstLoop = false;
                }
                baseCost = baseCost.add(budgetModularIdc.getIdcBase());
                calculatedCost = calculatedCost.add(budgetModularIdc.getFundsRequested());
            }
            indirectCostDetails = new S2SIndirectCostDetailsDto();
            indirectCostDetails.setBase(baseCost);
            indirectCostDetails.setBaseCostSharing(baseCostSharing);
            indirectCostDetails.setCostSharing(calculatedCostSharing);
            indirectCostDetails.setCostType(description);
            indirectCostDetails.setFunds(calculatedCost);
            indirectCostDetails.setRate(appliedRate);
            indirectCostDetailList.add(indirectCostDetails);
            totalIndirectCosts = totalIndirectCosts.add(calculatedCost);
            totalIndirectCostSharing = totalIndirectCostSharing.add(calculatedCostSharing);
        } else {
            Map<String, S2SIndirectCostDetailsDto> costDetailsMap = new HashMap<>();

            if (LOGGER.isDebugEnabled()) {
                int size = budgetPeriod.getBudgetLineItems() == null ? 0 : budgetPeriod.getBudgetLineItems().size();
                LOGGER.debug("ICDB: Processing " + size + " Line Items");
            }

            for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {


                if (LOGGER.isDebugEnabled()) {
                    int size = lineItem.getBudgetRateAndBaseList() == null ? 0 : lineItem.getBudgetRateAndBaseList().size();
                    LOGGER.debug("ICDB: Processing " + size + "Rate and Bases for Period = " + budgetPeriod.getBudgetPeriod());
                }


                for (BudgetRateAndBaseContract rateBase : lineItem.getBudgetRateAndBaseList()) {
                    RateClassContract rateClass = rateBase.getRateClass();

                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("ICDB: Rate Class = " + rateClass.toString());
                    }

                    if (rateClass.getRateClassType().getCode().equals(RateClassType.OVERHEAD.getRateClassType())) {

                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("ICDB: Processing OVERHEAD rate class. Rate Class = " + rateClass);
                        }

                        String rateClassCode = rateClass.getCode();
                        String rateTypeCode = rateBase.getRateTypeCode();
                        appliedRate = rateBase.getAppliedRate();
                        String key = rateClassCode + "-" + rateTypeCode + "-" + appliedRate;
                        boolean applyRateFlag = getApplyRateFlagForRateBase(rateBase, lineItem.getBudgetLineItemCalculatedAmounts());
                        if (costDetailsMap.get(key) == null) {
                            indirectCostDetails = new S2SIndirectCostDetailsDto();
                            indirectCostDetails.setBase(rateBase.getBaseCost() == null ? ScaleTwoDecimal.ZERO : applyRateFlag ? rateBase
                                    .getBaseCost() : ScaleTwoDecimal.ZERO);
                            indirectCostDetails.setBaseCostSharing(rateBase.getBaseCostSharing() == null ? ScaleTwoDecimal.ZERO
                                    : rateBase.getBaseCostSharing());
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                indirectCostDetails.setCostSharing(rateBase.getCalculatedCostSharing() == null ? ScaleTwoDecimal.ZERO
                                        : rateBase.getCalculatedCostSharing());
                            }
                            indirectCostDetails.setCostType(rateClass.getDescription());
                            indirectCostDetails.setFunds(rateBase.getCalculatedCost() == null ? ScaleTwoDecimal.ZERO : rateBase
                                    .getCalculatedCost());
                            indirectCostDetails.setRate(appliedRate);
                        } else {
                            indirectCostDetails = costDetailsMap.get(key);
                            baseCost = indirectCostDetails.getBase().add(
                                    rateBase.getBaseCost() == null ? ScaleTwoDecimal.ZERO : applyRateFlag ? rateBase
                                            .getBaseCost() : ScaleTwoDecimal.ZERO);
                            baseCostSharing = indirectCostDetails.getBaseCostSharing().add(
                                    rateBase.getBaseCostSharing() == null ? ScaleTwoDecimal.ZERO : rateBase.getBaseCostSharing());
                            calculatedCost = indirectCostDetails.getFunds().add(
                                    rateBase.getCalculatedCost() == null ? ScaleTwoDecimal.ZERO : rateBase.getCalculatedCost());
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                calculatedCostSharing = indirectCostDetails.getCostSharing().add(
                                        rateBase.getCalculatedCostSharing() == null ? ScaleTwoDecimal.ZERO : rateBase
                                                .getCalculatedCostSharing());
                            } else if (!lineItem.getSubmitCostSharingFlag() && budget.getSubmitCostSharingFlag()) {
                                calculatedCostSharing = indirectCostDetails.getCostSharing();
                            }
                            indirectCostDetails.setBase(baseCost);
                            indirectCostDetails.setBaseCostSharing(baseCostSharing);
                            indirectCostDetails.setCostSharing(calculatedCostSharing);
                            indirectCostDetails.setFunds(calculatedCost);
                        }
                        costDetailsMap.put(key, indirectCostDetails);

                        indirectCostDetailList = new ArrayList<>(costDetailsMap.values());


                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("ICDB: Updated Details List = " + indirectCostDetailList);
                        }

                        totalIndirectCosts = totalIndirectCosts.add(rateBase.getCalculatedCost() == null ? ScaleTwoDecimal.ZERO
                                : rateBase.getCalculatedCost());
                        if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                            totalIndirectCostSharing = totalIndirectCostSharing
                                    .add(rateBase.getCalculatedCostSharing() == null ? ScaleTwoDecimal.ZERO : rateBase
                                            .getCalculatedCostSharing());
                        }
                    }
                }
            }
        }
        S2SIndirectCostDto indirectCostInfo = new S2SIndirectCostDto();
        indirectCostInfo.setIndirectCostDetails(indirectCostDetailList);
        indirectCostInfo.setTotalIndirectCosts(totalIndirectCosts);
        indirectCostInfo.setTotalIndirectCostSharing(totalIndirectCostSharing);
        return indirectCostInfo;
    }

    protected List<S2SEquipmentDto> getEquipment(BudgetPeriodContract budgetPeriod, BudgetContract budget) {
        List<S2SCostDto> cvExtraEquipment = new ArrayList<>();
        S2SCostDto equipCostInfo;
        List<? extends BudgetCategoryMapContract> budgetCategoryMapList = s2SBudgetCategoryMapService.getBudgetCategoryMapList(new ArrayList<>(), new ArrayList<>());

        ScaleTwoDecimal totalEquipFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalExtraEquipFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalEquipNonFund = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalExtraEquipNonFund = ScaleTwoDecimal.ZERO;
        Map<String, S2SCostDto> costInfoMap = new HashMap<>();
        List<S2SCostDto> costInfos = new ArrayList<>();
        for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
            for (BudgetCategoryMapContract budgetCategoryMap : budgetCategoryMapList) {
                equipCostInfo = new S2SCostDto();
                for (BudgetCategoryMappingContract budgetCategoryMapping : budgetCategoryMap.getBudgetCategoryMappings()) {
                    if (lineItem.getBudgetCategory().getCode().equals(budgetCategoryMapping.getBudgetCategoryCode())
                            && budgetCategoryMapping.getTargetCategoryCode().equals(s2SConfigurationService.getValueAsString(
                            ConfigurationConstants.S2SBUDGET_TARGET_CATEGORY_CODE_EQUIPMENT_COST))
                            && budgetCategoryMapping.getMappingName().equals(SPONSOR)) {
                        equipCostInfo.setCategory(budgetCategoryMap.getDescription());
                        equipCostInfo.setCategoryType(budgetCategoryMap.getCategoryType());
                        if (lineItem.getLineItemDescription() != null)
                            equipCostInfo.setDescription(lineItem.getLineItemDescription());
                        else
                            equipCostInfo.setDescription(lineItem.getCostElementBO().getDescription());
                        totalEquipFund = totalEquipFund.add(lineItem.getLineItemCost());
                        if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                            totalEquipNonFund = totalEquipNonFund.add(lineItem.getCostSharingAmount());
                        }

                        String key = budgetCategoryMap.getCategoryType() + "-" + lineItem.getLineItemDescription();
                        if (costInfoMap.get(key) == null) {
                            equipCostInfo = new S2SCostDto();
                            equipCostInfo.setCategory(budgetCategoryMap.getDescription());
                            equipCostInfo.setCategoryType(budgetCategoryMap.getCategoryType());
                            if (lineItem.getLineItemDescription() != null)
                                equipCostInfo.setDescription(lineItem.getLineItemDescription());
                            else
                                equipCostInfo.setDescription(lineItem.getCostElementBO().getDescription());
                            equipCostInfo.setCost(lineItem.getLineItemCost());
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                equipCostInfo.setCostSharing(lineItem.getCostSharingAmount());
                            }

                            costInfos.add(equipCostInfo);
                        } else {
                            equipCostInfo = costInfoMap.get(key);
                            equipCostInfo.setCost(equipCostInfo.getCost().add(lineItem.getLineItemCost()));
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                equipCostInfo.setCostSharing(equipCostInfo.getCostSharing().add(lineItem.getCostSharingAmount()));
                            }
                        }
                        if (StringUtils.isNotEmpty(lineItem.getLineItemDescription())) {
                            costInfoMap.put(key, equipCostInfo);
                        }
                    }
                }
            }
        }
        S2SEquipmentDto equipmentInfo = new S2SEquipmentDto();

        if (costInfos.size() > 10) {
            for (int j = costInfos.size() - 1; j > 9; j--) {
                cvExtraEquipment.add(costInfos.get(j));
                S2SCostDto extraCostInfo = costInfos.get(j);
                totalExtraEquipFund = totalExtraEquipFund.add(extraCostInfo.getCost());
                totalExtraEquipNonFund = totalExtraEquipNonFund.add(extraCostInfo.getCostSharing());
                costInfos.remove(j);
            }
            Collections.reverse(cvExtraEquipment);
            equipmentInfo.setExtraEquipmentList(cvExtraEquipment);
            equipmentInfo.setTotalExtraNonFund(totalExtraEquipNonFund);
        }
        equipmentInfo.setTotalExtraFund(totalExtraEquipFund);

        equipmentInfo.setEquipmentList(costInfos);
        equipmentInfo.setTotalFund(totalEquipFund);
        equipmentInfo.setTotalNonFund(totalEquipNonFund);

        List<S2SEquipmentDto> equipmentInfos = new ArrayList<>();
        equipmentInfos.add(equipmentInfo);
        return equipmentInfos;
    }

    protected List<S2SOtherDirectCostInfoDto> getOtherDirectCosts(BudgetPeriodContract budgetPeriod, BudgetContract budget) {
        S2SOtherDirectCostInfoDto otherDirectCostInfo = new S2SOtherDirectCostInfoDto();

        List<S2SCostDto> costInfoList = new ArrayList<>();
        List<String> filterTargetCategoryCodes = new ArrayList<>();
        filterTargetCategoryCodes.add(s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_TARGET_CATEGORY_CODE_EQUIPMENT_COST));
        List<String> filterCategoryTypes = new ArrayList<>();
        filterCategoryTypes.add(s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_FILTER_CATEGORY_TYPE_PERSONNEL));
        List<? extends BudgetCategoryMapContract> budgetCategoryMapList = s2SBudgetCategoryMapService.getBudgetCategoryMapList(filterTargetCategoryCodes, filterCategoryTypes);

        boolean recordAdded;
        for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
            for (BudgetCategoryMapContract budgetCategoryMap : budgetCategoryMapList) {
                recordAdded = false;
                for (BudgetCategoryMappingContract budgetCategoryMapping : budgetCategoryMap.getBudgetCategoryMappings()) {
                    if (lineItem.getBudgetCategory().getCode().equals(budgetCategoryMapping.getBudgetCategoryCode())) {
                        S2SCostDto costInfo = new S2SCostDto();
                        costInfo.setCost(lineItem.getLineItemCost());
                        if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                            costInfo.setCostSharing(lineItem.getCostSharingAmount());
                        }
                        costInfo.setCategory(budgetCategoryMap.getDescription());
                        costInfo.setCategoryType(budgetCategoryMap.getCategoryType());
                        if (lineItem.getQuantity() != null) {
                            costInfo.setQuantity(lineItem.getQuantity());
                        }
                        costInfoList.add(costInfo);
                        recordAdded = true;
                        break;
                    }
                }
                if (recordAdded) {
                    break;
                }
            }

            S2SCostDto lineItemcostInfo = new S2SCostDto();
            lineItemcostInfo.setCategory(OTHER_DIRECT_COSTS);
            lineItemcostInfo.setCategoryType(CATEGORY_TYPE_OTHER_DIRECT_COST);
            lineItemcostInfo.setQuantity(1);

            ScaleTwoDecimal totalCost = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal totalCostSharing = ScaleTwoDecimal.ZERO;
            for (BudgetLineItemCalculatedAmountContract lineItemCalculatedAmt : lineItem.getBudgetLineItemCalculatedAmounts()) {
                if (lineItemCalculatedAmt
                        .getRateClass()
                        .getRateClassType().getCode()
                        .equals(s2SConfigurationService.getValueAsString(
                                ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_SALARIES_MS))) {
                    totalCost = totalCost.add(lineItemCalculatedAmt.getCalculatedCost());
                    if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                        totalCostSharing = totalCostSharing.add(lineItemCalculatedAmt.getCalculatedCostSharing());
                    }
                }
            }
            lineItemcostInfo.setCost(totalCost);
            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                lineItemcostInfo.setCostSharing(totalCostSharing);
            }
            String description = DESCRIPTION_LA + totalCost + ";";
            lineItemcostInfo.setDescription(description);
            costInfoList.add(lineItemcostInfo);
        }

        ScaleTwoDecimal totalOtherDirect = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalTravelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalParticipantCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalOtherDirectCostSharing = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal totalTravelCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalParticipantCostSharing = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal materialCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal materialCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal consultantCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal consultantCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal publicationCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal publicationCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal computerCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal computerCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal alterationsCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal alterationsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal subContractCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal subContractCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal equipmentRentalCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal equipmentRentalCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal domesticTravelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal domesticTravelCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal foreignTravelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal foreignTravelCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partStipendsCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partStipendsCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partTravelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partTravelCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partTuitionCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partTuitionCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partSubsistenceCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partSubsistenceCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partOtherCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal partOtherCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherDirectCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherDirectCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherCostSharing = ScaleTwoDecimal.ZERO;
        Optional<ScaleTwoDecimal> otherDirectHumanFetalTissueCost = Optional.empty();
        Optional<ScaleTwoDecimal> otherDirectHumanFetalTissueCostSharing = Optional.empty();
        Optional<ScaleTwoDecimal> otherDataManagementAndSharingCosts = Optional.empty();
        Optional<ScaleTwoDecimal> otherDataManagementAndSharingCostsSharing = Optional.empty();

        for (S2SCostDto costInfo : costInfoList) {
            if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_MATERIALS_AND_SUPPLIES_CATEGORY))) {
                materialCost = materialCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    materialCostSharing = materialCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_CONSULTANT_COSTS_CATEGORY))) {
                consultantCost = consultantCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    consultantCostSharing = consultantCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PUBLICATION_COSTS_CATEGORY))) {
                publicationCost = publicationCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    publicationCostSharing = publicationCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_COMPUTER_SERVICES_CATEGORY))) {
                computerCost = computerCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    computerCostSharing = computerCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_ALTERATIONS_CATEGORY))) {
                alterationsCost = alterationsCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    alterationsCostSharing = alterationsCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_SUBCONTRACT_CATEGORY))) {
                subContractCost = subContractCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    subContractCostSharing = subContractCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_EQUIPMENT_RENTAL_CATEGORY))) {
                equipmentRentalCost = equipmentRentalCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    equipmentRentalCostSharing = equipmentRentalCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_DOMESTIC_TRAVEL_CATEGORY))) {
                domesticTravelCost = domesticTravelCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    domesticTravelCostSharing = domesticTravelCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_FOREIGN_TRAVEL_CATEGORY))) {
                foreignTravelCost = foreignTravelCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    foreignTravelCostSharing = foreignTravelCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PARTICIPANT_STIPENDS_CATEGORY))) {
                partStipendsCost = partStipendsCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    partStipendsCostSharing = partStipendsCostSharing.add(costInfo.getCostSharing());
                    totalParticipantCostSharing = totalParticipantCostSharing.add(costInfo.getCostSharing());
                }
                totalParticipantCost = totalParticipantCost.add(costInfo.getCost());
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PARTICIPANT_TRAVEL_CATEGORY))) {
                partTravelCost = partTravelCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    partTravelCostSharing = partTravelCostSharing.add(costInfo.getCostSharing());
                    totalParticipantCostSharing = totalParticipantCostSharing.add(costInfo.getCostSharing());
                }
                totalParticipantCost = totalParticipantCost.add(costInfo.getCost());
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PARTICIPANT_TUITION_CATEGORY))) {
                partTuitionCost = partTuitionCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    partTuitionCostSharing = partTuitionCostSharing.add(costInfo.getCostSharing());
                    totalParticipantCostSharing = totalParticipantCostSharing.add(costInfo.getCostSharing());
                }
                totalParticipantCost = totalParticipantCost.add(costInfo.getCost());
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PARTICIPANT_SUBSISTENCE_CATEGORY))) {
                partSubsistenceCost = partSubsistenceCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    partSubsistenceCostSharing = partSubsistenceCostSharing.add(costInfo.getCostSharing());
                    totalParticipantCostSharing = totalParticipantCostSharing.add(costInfo.getCostSharing());
                }
                totalParticipantCost = totalParticipantCost.add(costInfo.getCost());
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_PARTICIPANT_OTHER_CATEGORY))) {
                partOtherCost = partOtherCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    partOtherCostSharing = partOtherCostSharing.add(costInfo.getCostSharing());
                    totalParticipantCostSharing = totalParticipantCostSharing.add(costInfo.getCostSharing());
                }
                totalParticipantCost = totalParticipantCost.add(costInfo.getCost());
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_OTHER_DIRECT_COSTS_CATEGORY))) {
                otherDirectCost = otherDirectCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    otherDirectCostSharing = otherDirectCostSharing.add(costInfo.getCostSharing());
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_OTHER_DIRECT_HUMAN_FETAL_TISSUE_COSTS_CATEGORY))) {
                otherDirectHumanFetalTissueCost = otherDirectHumanFetalTissueCost
                        .map(v -> v.add(costInfo.getCost()))
                        .or(() -> Optional.of(costInfo.getCost()));

                if (budget.getSubmitCostSharingFlag()) {
                    otherDirectHumanFetalTissueCostSharing = otherDirectHumanFetalTissueCostSharing
                            .map(v -> v.add(costInfo.getCostSharing()))
                            .or(() -> Optional.of(costInfo.getCostSharing()));
                }
            } else if (costInfo.getCategory().equals(s2SConfigurationService.getValueAsString(
                    ConfigurationConstants.S2SBUDGET_OTHER_DIRECT_DATA_MANAGEMENT_AND_SHARING_COSTS_CATEGORY))) {
                otherDataManagementAndSharingCosts = otherDataManagementAndSharingCosts
                        .map(v -> v.add(costInfo.getCost()))
                        .or(() -> Optional.of(costInfo.getCost()));
                if (budget.getSubmitCostSharingFlag()) {
                    otherDataManagementAndSharingCostsSharing = otherDataManagementAndSharingCostsSharing
                            .map(v -> v.add(costInfo.getCostSharing()))
                            .or(() -> Optional.of(costInfo.getCostSharing()));
                }
            } else {
                otherCost = otherCost.add(costInfo.getCost());
                if (budget.getSubmitCostSharingFlag()) {
                    otherCostSharing = otherCostSharing.add(costInfo.getCostSharing());
                }
            }
        }

        otherDirectCostInfo.setMaterials(materialCost);
        otherDirectCostInfo.setMaterialsCostSharing(materialCostSharing);
        otherDirectCostInfo.setConsultants(consultantCost);
        otherDirectCostInfo.setConsultantsCostSharing(consultantCostSharing);
        otherDirectCostInfo.setPublications(publicationCost);
        otherDirectCostInfo.setPublicationsCostSharing(publicationCostSharing);
        otherDirectCostInfo.setComputer(computerCost);
        otherDirectCostInfo.setComputerCostSharing(computerCostSharing);
        otherDirectCostInfo.setAlterations(alterationsCost);
        otherDirectCostInfo.setAlterationsCostSharing(alterationsCostSharing);
        otherDirectCostInfo.setSubAwards(subContractCost);
        otherDirectCostInfo.setSubAwardsCostSharing(subContractCostSharing);
        otherDirectCostInfo.setEquipRental(equipmentRentalCost);
        otherDirectCostInfo.setEquipRentalCostSharing(equipmentRentalCostSharing);
        otherDirectCostInfo.setDomTravel(domesticTravelCost);
        otherDirectCostInfo.setDomTravelCostSharing(domesticTravelCostSharing);
        otherDirectCostInfo.setForeignTravel(foreignTravelCost);
        otherDirectCostInfo.setForeignTravelCostSharing(foreignTravelCostSharing);
        otherDirectCostInfo.setPartStipends(partStipendsCost);
        otherDirectCostInfo.setPartStipendsCostSharing(partStipendsCostSharing);
        otherDirectCostInfo.setPartTravel(partTravelCost);
        otherDirectCostInfo.setPartTravelCostSharing(partTravelCostSharing);
        otherDirectCostInfo.setPartTuition(partTuitionCost);
        otherDirectCostInfo.setPartTuitionCostSharing(partTuitionCostSharing);
        otherDirectCostInfo.setPartSubsistence(partSubsistenceCost);
        otherDirectCostInfo.setPartSubsistenceCostSharing(partSubsistenceCostSharing);
        otherDirectCostInfo.setPartOther(partOtherCost);
        otherDirectCostInfo.setPartOtherCostSharing(partOtherCostSharing);
        otherDirectCostInfo.setParticipantTotal(totalParticipantCost);
        otherDirectCostInfo.setParticipantTotalCostSharing(totalParticipantCostSharing);
        otherDirectCostInfo.setParticipantTotalCount(budgetPeriod.getNumberOfParticipants() == null ? 0 : budgetPeriod.getNumberOfParticipants());

        totalOtherDirect = totalOtherDirect.add(materialCost);
        totalOtherDirect = totalOtherDirect.add(consultantCost);
        totalOtherDirect = totalOtherDirect.add(publicationCost);
        totalOtherDirect = totalOtherDirect.add(computerCost);
        totalOtherDirect = totalOtherDirect.add(alterationsCost);
        if (otherDirectHumanFetalTissueCost.isPresent()) {
            totalOtherDirect = totalOtherDirect.add(otherDirectHumanFetalTissueCost.get());
        }
        if (otherDataManagementAndSharingCosts.isPresent()) {
            totalOtherDirect = totalOtherDirect.add(otherDataManagementAndSharingCosts.get());
        }
        totalOtherDirect = totalOtherDirect.add(subContractCost);
        totalOtherDirect = totalOtherDirect.add(equipmentRentalCost);
        totalOtherDirect = totalOtherDirect.add(otherDirectCost);
        totalOtherDirect = totalOtherDirect.add(otherCost);

        totalTravelCost = totalTravelCost.add(domesticTravelCost);
        totalTravelCost = totalTravelCost.add(foreignTravelCost);
        if (budget.getSubmitCostSharingFlag()) {
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(materialCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(consultantCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(publicationCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(computerCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(alterationsCostSharing);
            if (otherDirectHumanFetalTissueCostSharing.isPresent()) {
                totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(otherDirectHumanFetalTissueCostSharing.get());
            }
            if (otherDataManagementAndSharingCostsSharing.isPresent()) {
                totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(otherDataManagementAndSharingCostsSharing.get());
            }
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(subContractCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(equipmentRentalCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(otherDirectCostSharing);
            totalOtherDirectCostSharing = totalOtherDirectCostSharing.add(otherCostSharing);

            totalTravelCostSharing = totalTravelCostSharing.add(domesticTravelCostSharing);
            totalTravelCostSharing = totalTravelCostSharing.add(foreignTravelCostSharing);
        }
        otherDirectCostInfo.setTotalOtherDirect(totalOtherDirect);
        otherDirectCostInfo.setTotalOtherDirectCostSharing(totalOtherDirectCostSharing);
        otherDirectCostInfo.setTotTravel(totalTravelCost);
        otherDirectCostInfo.setTotTravelCostSharing(totalTravelCostSharing);

        final List<S2SOtherCostDto> otherCostDetails = new ArrayList<>();

        final S2SOtherCostDto hmOtherDirectCostDetails = new S2SOtherCostDto();
        hmOtherDirectCostDetails.setCost(otherDirectCost);
        hmOtherDirectCostDetails.setDescription(OTHER_DIRECT_COSTS);
        hmOtherDirectCostDetails.setCostSharing(otherDirectCostSharing);
        otherCostDetails.add(hmOtherDirectCostDetails);

        if (otherDirectHumanFetalTissueCost.isPresent()) {
            final S2SOtherCostDto hftOtherDirectCostDetails = new S2SOtherCostDto();
            hftOtherDirectCostDetails.setCost(otherDirectHumanFetalTissueCost.get());
            hftOtherDirectCostDetails.setDescription(HUMAN_FETAL_TISSUE_COSTS);
            hftOtherDirectCostDetails.setCostSharing(otherDirectHumanFetalTissueCostSharing.orElse(ScaleTwoDecimal.ZERO));
            otherCostDetails.add(hftOtherDirectCostDetails);
        }

        if (otherDataManagementAndSharingCosts.isPresent()) {
            final S2SOtherCostDto dmsOtherDirectionCostDetails = new S2SOtherCostDto();
            dmsOtherDirectionCostDetails.setCost(otherDataManagementAndSharingCosts.get());
            dmsOtherDirectionCostDetails.setDescription(DATA_MANAGEMENT_AND_SHARING_COSTS);
            dmsOtherDirectionCostDetails.setCostSharing(otherDataManagementAndSharingCostsSharing.orElse(ScaleTwoDecimal.ZERO));
            otherCostDetails.add(dmsOtherDirectionCostDetails);
        }

        final S2SOtherCostDto hmOtherCostDetails = new S2SOtherCostDto();
        hmOtherCostDetails.setCost(otherCost);
        hmOtherCostDetails.setDescription(ALL_OTHER_COSTS);
        hmOtherCostDetails.setCostSharing(otherCostSharing);
        otherCostDetails.add(hmOtherCostDetails);

        otherDirectCostInfo.setOtherCosts(otherCostDetails);
        List<S2SOtherDirectCostInfoDto> otherDirectCosts = new ArrayList<>();
        otherDirectCosts.add(otherDirectCostInfo);
        return otherDirectCosts;
    }

    protected List<S2SKeyPersonDto> getKeyPersons(BudgetPeriodContract budgetPeriod, ProposalDevelopmentDocumentContract pdDoc, BudgetContract budget) {
        List<S2SKeyPersonDto> keyPersons = new ArrayList<>();
        S2SKeyPersonDto keyPerson = new S2SKeyPersonDto();
        ProposalPersonContract principalInvestigator = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);

        if (principalInvestigator != null) {

            keyPerson.setPersonId(principalInvestigator.getPersonId());
            keyPerson.setRolodexId(principalInvestigator.getRolodexId());
            keyPerson.setFirstName(principalInvestigator.getFirstName() == null ? FieldValueConstants.VALUE_UNKNOWN
                    : principalInvestigator.getFirstName());
            keyPerson.setLastName(principalInvestigator.getLastName() == null ? FieldValueConstants.VALUE_UNKNOWN : principalInvestigator
                    .getLastName());
            keyPerson.setMiddleName(principalInvestigator.getMiddleName());
            keyPerson.setRole(PRINCIPAL_INVESTIGATOR_ROLE);
            addUniqueKeyPerson(keyPersons, keyPerson);
        }

        List<? extends BudgetCategoryMappingContract> seniorBudgetCategories = budgetCategoryMapService.findCatMappingByTargetAndMappingName(TARGET_CATEGORY_CODE_01, SPONSOR);
        for (ProposalPersonContract coInvestigator : s2SProposalPersonService.getCoInvestigators(pdDoc)) {

            keyPerson = new S2SKeyPersonDto();
            keyPerson.setPersonId(coInvestigator.getPersonId());
            keyPerson.setRolodexId(coInvestigator.getRolodexId());
            keyPerson.setFirstName(coInvestigator.getFirstName() == null ? FieldValueConstants.VALUE_UNKNOWN : coInvestigator
                    .getFirstName());
            keyPerson.setLastName(coInvestigator.getLastName() == null ? FieldValueConstants.VALUE_UNKNOWN : coInvestigator.getLastName());
            keyPerson.setMiddleName(coInvestigator.getMiddleName());

            if (sponsorHierarchyService.isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                if (coInvestigator.isMultiplePi()) {
                    keyPerson.setRole(NID_PD_PI);
                } else {
                    keyPerson.setRole(NID_CO_PD_PI);
                }
            } else {
                keyPerson.setRole(KEYPERSON_CO_PD_PI);
            }
            boolean isEmployeeMpi = keyPerson.getRole().equals(NID_PD_PI) && keyPerson.getPersonId() != null;
            if (isEmployeeMpi || hasSeniorPersonnelBudget(budgetPeriod, keyPerson, seniorBudgetCategories)) {
                addUniqueKeyPerson(keyPersons, keyPerson);
            }
        }

        for (ProposalPersonContract propPerson : s2SProposalPersonService.getKeyPersons(pdDoc)) {

            keyPerson = new S2SKeyPersonDto();
            keyPerson.setPersonId(propPerson.getPersonId());
            keyPerson.setRolodexId(propPerson.getRolodexId());
            keyPerson.setFirstName(propPerson.getFirstName() == null ? FieldValueConstants.VALUE_UNKNOWN : propPerson.getFirstName());
            keyPerson.setLastName(propPerson.getLastName() == null ? FieldValueConstants.VALUE_UNKNOWN : propPerson.getLastName());
            keyPerson.setMiddleName(propPerson.getMiddleName());

            keyPerson.setRole(getBudgetPersonRoleOther());
            keyPerson.setKeyPersonRole(propPerson.getProjectRole());
            if (hasSeniorPersonnelBudget(budgetPeriod, keyPerson, seniorBudgetCategories)) {
                addUniqueKeyPerson(keyPersons, keyPerson);
            }
        }

        for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
            for (BudgetPersonnelDetailsContract budgetPersonnelDetails : lineItem.getBudgetPersonnelDetailsList()) {
                if (isSeniorLineItem(seniorBudgetCategories, lineItem.getBudgetCategory().getCode())) {
                    if (budgetPersonnelDetails.getBudgetPerson().getNonEmployeeFlag()) {
                        if (budgetPersonnelDetails.getBudgetPerson().getRolodexId() != null) {
                            RolodexContract rolodexPerson = rolodexService.getRolodex(budgetPersonnelDetails.getBudgetPerson().getRolodexId());
                            keyPerson = new S2SKeyPersonDto();
                            keyPerson.setRolodexId(rolodexPerson.getRolodexId());
                            keyPerson.setFirstName(rolodexPerson.getFirstName() == null ? FieldValueConstants.VALUE_UNKNOWN
                                    : rolodexPerson.getFirstName());
                            keyPerson.setLastName(rolodexPerson.getLastName() == null ? FieldValueConstants.VALUE_UNKNOWN : rolodexPerson
                                    .getLastName());
                            keyPerson.setMiddleName(rolodexPerson.getMiddleName());
                            keyPerson.setRole(StringUtils.isNotBlank(rolodexPerson.getTitle()) ? rolodexPerson.getTitle()
                                    : getBudgetPersonRoleOther());

                            addUniqueKeyPerson(keyPersons, keyPerson);
                        } else if (StringUtils.isNotBlank(budgetPersonnelDetails.getBudgetPerson().getTbnId())) {
                            TbnPersonContract tbnPerson = tbnPersonService.getTbnPerson(budgetPersonnelDetails.getBudgetPerson().getTbnId());
                            if (tbnPerson != null) {
                                keyPerson = new S2SKeyPersonDto();
                                keyPerson.setTbn(true);
                                keyPerson.setTbnName(budgetPersonnelDetails.getBudgetPerson().getPersonName());
                                final List<String> tbnNames = Stream.of(budgetPersonnelDetails.getBudgetPerson().getPersonName().split(" "))
                                        .filter(Objects::nonNull)
                                        .map(s -> s.replace("-", ""))
                                        .map(String::trim)
                                        .filter(StringUtils::isNotBlank)
                                        .collect(Collectors.toList());
                                final String firstName = tbnNames.size() >= 1 ? tbnNames.get(0) : FieldValueConstants.VALUE_UNKNOWN;
                                final String middleName = tbnNames.size() >= 3 ? tbnNames.get(1) : " ";
                                final String lastName;
                                if (tbnNames.size() == 2) {
                                    final String l = tbnNames.get(1);
                                    lastName = StringUtils.isNotBlank(l) ? l : FieldValueConstants.VALUE_UNKNOWN;
                                } else if (tbnNames.size() >= 3) {
                                    final String l = String.join(" ", tbnNames.subList(2, tbnNames.size()));
                                    lastName = StringUtils.isNotBlank(l) ? l : FieldValueConstants.VALUE_UNKNOWN;
                                } else {
                                    lastName = FieldValueConstants.VALUE_UNKNOWN;
                                }

                                keyPerson.setPersonId(tbnPerson.getId());
                                keyPerson.setFirstName(firstName);
                                keyPerson.setMiddleName(middleName);
                                keyPerson.setLastName(lastName);
                                keyPerson.setRole(tbnPerson.getPersonName());

                                addUniqueKeyPerson(keyPersons, keyPerson);
                            }
                        }
                    } else {

                        KcPersonContract kcPerson = kcPersonRepositoryService.findKcPersonByPersonId(budgetPersonnelDetails.getBudgetPerson().getPersonId());
                        if (kcPerson != null) {
                            keyPerson = new S2SKeyPersonDto();
                            keyPerson.setPersonId(kcPerson.getPersonId());
                            keyPerson.setFirstName(kcPerson.getFirstName() == null ? FieldValueConstants.VALUE_UNKNOWN : kcPerson
                                    .getFirstName());
                            keyPerson.setLastName(kcPerson.getLastName() == null ? FieldValueConstants.VALUE_UNKNOWN : kcPerson
                                    .getLastName());
                            keyPerson.setMiddleName(kcPerson.getMiddleName());
                            keyPerson.setRole(getBudgetPersonRoleOther());

                            addUniqueKeyPerson(keyPersons, keyPerson);
                        }
                    }
                }
            }
        }

        for (S2SKeyPersonDto keyPersonInfo : keyPersons) {
            setCompensation(keyPersonInfo, budgetPeriod, budget);
        }

        return keyPersons;
    }

    private Boolean hasPersonnelBudget(BudgetPeriodContract budgetPeriod, S2SKeyPersonDto keyPerson) {
        return budgetPeriod.getBudgetLineItems().stream()
                .anyMatch(lineItem -> lineItem.getBudgetPersonnelDetailsList().stream()
                        .anyMatch(personnelDetails -> isSamePerson(keyPerson, personnelDetails))
                );
    }

    private boolean hasSeniorPersonnelBudget(BudgetPeriodContract budgetPeriod, S2SKeyPersonDto keyPerson, List<? extends BudgetCategoryMappingContract> budgetCategoryList) {
        return budgetPeriod.getBudgetLineItems().stream()
                .anyMatch(lineItem -> lineItem.getBudgetPersonnelDetailsList().stream()
                        .anyMatch(personnelDetails -> isSamePerson(keyPerson, personnelDetails) && isSeniorLineItem(budgetCategoryList, lineItem.getBudgetCategory().getCode()))
                );
    }

    private boolean canBudgetLineItemCostSharingInclude(BudgetContract budget, BudgetLineItemContract lineItem) {
        return budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag();
    }

    private boolean isSamePerson(S2SKeyPersonDto keyPerson, BudgetPersonnelDetailsContract budgetPersonnelDetails) {
        return budgetPersonnelDetails.getPersonId().equals(keyPerson.getPersonId()) ||
                (keyPerson.getRolodexId() != null && budgetPersonnelDetails.getPersonId().equals(keyPerson.getRolodexId().toString()));
    }

    private String getBudgetPersonRoleOther() {
        return KEYPERSON_OTHER;
    }

    private boolean isSeniorLineItem(List<? extends BudgetCategoryMappingContract> budgetCategoryList, String budgetCategoryCode) {
        boolean isSeniorLineItem = false;
        for (BudgetCategoryMappingContract categoryMapping : budgetCategoryList) {
            if (categoryMapping.getBudgetCategoryCode().equals(budgetCategoryCode)) {
                isSeniorLineItem = true;
            }
        }
        return isSeniorLineItem;
    }

    protected List<S2SKeyPersonDto> getNKeyPersons(List<S2SKeyPersonDto> keyPersons, int n) {
        return keyPersons.stream().limit(n).collect(Collectors.toList());
    }

    protected List<S2SOtherPersonnelDto> getOtherPersonnel(BudgetPeriodContract budgetPeriod, List<S2SKeyPersonDto> keyPersons) {
        List<S2SOtherPersonnelDto> cvOtherPersonnel = new ArrayList<>();
        cvOtherPersonnel.add(getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_GRADUATES),
                PERSONNEL_TYPE_GRAD,
                ROLE_GRADUATE_STUDENTS,
                keyPersons));
        cvOtherPersonnel.add(getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_POSTDOCS),
                PERSONNEL_TYPE_POSTDOC,
                ROLE_POST_DOCTORAL_ASSOCIATES,
                keyPersons));
        cvOtherPersonnel.add(getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_UNDERGRADS),
                PERSONNEL_TYPE_UNDERGRAD,
                ROLE_GRADUATE_UNDERGRADUATE_STUDENTS,
                keyPersons));
        cvOtherPersonnel.add(getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_SECRETARIAL),
                PERSONNEL_TYPE_SEC,
                ROLE_GRADUATE_SECRETARIAL_OR_CLERICAL,
                keyPersons));
        S2SOtherPersonnelDto other = getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_OTHER),
                PERSONNEL_TYPE_OTHER,
                ROLE_GRADUATE_OTHER,
                keyPersons);
        S2SOtherPersonnelDto otherProfessional = getOtherPersonnelDetails(
                budgetPeriod,
                s2SConfigurationService.getValueAsString(ConfigurationConstants.S2SBUDGET_CATEGORY_01_OTHER_PROFS),
                PERSONNEL_TYPE_OTHER_PROFESSIONALS,
                ROLE_GRADUATE_OTHER_PROFESSIONALS,
                keyPersons);
        S2SOtherPersonnelDto allocatedAdminSupport = getOtherPersonnelDetails(
                budgetPeriod,
                LASALARIES,
                PERSONNEL_TYPE_ALLOCATED_ADMIN_SUPPORT,
                ROLE_GRADUATE_ALLOCATED_ADMIN_SUPPORT,
                keyPersons);
        if (other.getNumberPersonnel() > 0) {
            cvOtherPersonnel.add(other);
        }
        if (otherProfessional.getNumberPersonnel() > 0) {
            cvOtherPersonnel.add(otherProfessional);
        }
        if (allocatedAdminSupport.getNumberPersonnel() > 0) {
            cvOtherPersonnel.add(allocatedAdminSupport);
        }
        return cvOtherPersonnel;
    }

    protected S2SOtherPersonnelDto getOtherPersonnelDetails(BudgetPeriodContract budgetPeriod, String category, String personnelType,
                                                            String role, List<S2SKeyPersonDto> keyPersons) {
        S2SOtherPersonnelDto otherPersonnelInfo = new S2SOtherPersonnelDto();

        int count = 0;
        ScaleTwoDecimal salaryRequested = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal salaryCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal lineItemCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal lineItemCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal mrLaCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal mrLaCostSharingAmount = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal fringeCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fringeCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal mrLaFringeCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal mrLaFringeCostSharingAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetLineItemFringeCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal budgetLineItemFringeCostSharingAmount = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal bdSalary = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal bdFringe = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal bdFunds;
        ScaleTwoDecimal bdSalaryCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal bdFringeCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal bdNonFunds;

        BigDecimal academicMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        BigDecimal summerMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        BigDecimal calendarMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        BigDecimal cycleMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);

        BigDecimal numberOfMonths;
        String rateTypeSupportStaffSalaries = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_TYPE_SUPPORT_STAFF_SALARIES);
        String rateClassCodeEmployeeBenefits = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_CLASS_CODE_EMPLOYEE_BENEFITS);
        String rateClassCodeVacation = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_CLASS_CODE_VACATION);
        String rateTypeAdministrativeSalaries = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_RATE_TYPE_ADMINISTRATIVE_SALARIES);
        Set<String> personJobCodes = new HashSet<>();

        // Calculate the salary and fringe for category
        // LASALARIES
        if (category.equalsIgnoreCase(LASALARIES)) {
            for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                // Caluclate LA for rate class type Y
                for (BudgetLineItemCalculatedAmountContract lineItemCalculatedAmount : lineItem
                        .getBudgetLineItemCalculatedAmounts()) {
                    if (lineItemCalculatedAmount
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_LAB_ALLOCATION_SALARIES))) {
                        mrLaCost = mrLaCost.add(lineItemCalculatedAmount.getCalculatedCost());
                        if (lineItem.getSubmitCostSharingFlag()) {
                            mrLaCostSharingAmount = mrLaCostSharingAmount.add(lineItemCalculatedAmount
                                    .getCalculatedCostSharing());
                        }
                    }

                    // Calculate the fringe
                    if ((lineItemCalculatedAmount
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_EMPLOYEE_BENEFITS)) && lineItemCalculatedAmount
                            .getRateTypeCode().equals(
                                    rateTypeSupportStaffSalaries))
                            || (lineItemCalculatedAmount
                            .getRateClass()
                            .getRateClassType().getCode()
                            .equals(s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_VACATION)) && lineItemCalculatedAmount
                            .getRateTypeCode().equals(
                                    rateTypeAdministrativeSalaries))) {
                        mrLaFringeCost = mrLaFringeCost.add(lineItemCalculatedAmount.getCalculatedCost());
                        if (lineItem.getSubmitCostSharingFlag()) {
                            mrLaFringeCostSharingAmount = mrLaFringeCostSharingAmount.add(lineItemCalculatedAmount
                                    .getCalculatedCostSharing());
                        }
                    }
                }
            }
        } else {
            for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                List<? extends BudgetCategoryMappingContract> budgetCategoryList = budgetCategoryMapService.findCatMappingByTargetAndMappingName(category, SPONSOR);

                for (BudgetCategoryMappingContract categoryMapping : budgetCategoryList) {
                    if (categoryMapping.getBudgetCategoryCode().equals(lineItem.getBudgetCategory().getCode())) {
                        List<? extends BudgetPersonnelDetailsContract> lineItemPersonDetails = lineItem.getBudgetPersonnelDetailsList();
                        if (!lineItemPersonDetails.isEmpty()) {
                            for (BudgetPersonnelDetailsContract personDetails : lineItemPersonDetails) {
                                if (categoryMapping.getBudgetCategoryCode().equals(lineItem.getBudgetCategory().getCode())) {
                                    // get sum of salary of other personnel, but exclude key persons
                                    if (keyPersons.stream().noneMatch(keyPersonDto -> isSamePerson(keyPersonDto, personDetails))) {
                                        salaryRequested = salaryRequested.add(personDetails.getSalaryRequested());
                                        if (lineItem.getSubmitCostSharingFlag()) {
                                            salaryCostSharing = salaryCostSharing.add(personDetails.getCostSharingAmount());
                                        }
                                        numberOfMonths = s2SDateTimeService.getNumberOfMonths(personDetails.getStartDate(),
                                                personDetails.getEndDate()).bigDecimalValue();
                                        if (personDetails.getPeriodTypeCode().equals(
                                                s2SConfigurationService.getValueAsString(
                                                        ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_ACADEMIC_MONTHS))) {
                                            academicMonths = getPersonEffortMonths(academicMonths, numberOfMonths, personDetails);
                                        } else if (personDetails.getPeriodTypeCode().equals(
                                                s2SConfigurationService.getValueAsString(
                                                        ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_SUMMER_MONTHS))) {
                                            summerMonths = getPersonEffortMonths(summerMonths, numberOfMonths, personDetails);

                                        } else if (personDetails.getPeriodTypeCode().equals(
                                                s2SConfigurationService.getValueAsString(
                                                        ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_CALENDAR_MONTHS))) {
                                            calendarMonths = getPersonEffortMonths(calendarMonths, numberOfMonths, personDetails);
                                        } else if (personDetails.getPeriodTypeCode().equals(
                                                s2SConfigurationService.getValueAsString(
                                                        ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_CYCLE_MONTHS))) {
                                            cycleMonths = cycleMonths.add(personDetails.getPercentEffort().bigDecimalValue().multiply(numberOfMonths)
                                                    .multiply(POINT_ZERO_ONE));
                                        }
                                        // Get total count of unique
                                        // personId+jobCode combination for those
                                        // persons who are part of
                                        // BudgetPersonnelDetails but are not
                                        // proposal persons
                                        String key;
                                        if (isTbn(personDetails.getBudgetPerson())) {
                                            key = personDetails.getPersonId() + "$tbn$" + personDetails.getBudgetPerson().getPersonName() + personDetails.getJobCode();

                                            if (personDetails.getBudgetPerson().getHierarchyProposalNumber() != null) {
                                                key += "$pd$" + personDetails.getBudgetPerson().getHierarchyProposalNumber();
                                            }
                                        } else {
                                            key = personDetails.getPersonId() + personDetails.getJobCode();
                                        }

                                        personJobCodes.add(key);

                                        // Calculate the fringe cost
                                        for (BudgetPersonnelCalculatedAmountContract personCalculatedAmount : personDetails
                                                .getBudgetPersonnelCalculatedAmounts()) {
                                            if ((personCalculatedAmount.getRateClass().getCode().equals(
                                                    rateClassCodeEmployeeBenefits) && !personCalculatedAmount
                                                    .getRateTypeCode().equals(
                                                            rateTypeSupportStaffSalaries))
                                                    || (personCalculatedAmount.getRateClass().getCode().equals(
                                                    rateClassCodeVacation) && !personCalculatedAmount
                                                    .getRateTypeCode().equals(
                                                            rateTypeAdministrativeSalaries))) {
                                                fringeCost = fringeCost.add(personCalculatedAmount.getCalculatedCost());
                                                if (lineItem.getSubmitCostSharingFlag()) {
                                                    fringeCostSharingAmount = fringeCostSharingAmount.add(personCalculatedAmount
                                                            .getCalculatedCostSharing());
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        } else {
                            // personExist is false. No person found for the line
                            // item.
                            // get costs for this budget category that do not have
                            // persons attached to the cost element
                            lineItemCost = lineItemCost.add(lineItem.getLineItemCost());
                            if (lineItem.getSubmitCostSharingFlag()) {
                                lineItemCostSharingAmount = lineItemCostSharingAmount.add(lineItem.getCostSharingAmount());
                            }
                            count = lineItem.getQuantity();
                            for (BudgetLineItemCalculatedAmountContract lineItemCalculatedAmount : lineItem
                                    .getBudgetLineItemCalculatedAmounts()) {

                                // Calculate fringe cost
                                if (lineItemCalculatedAmount.getRateClass().getRateClassType().getCode().equalsIgnoreCase("E")) {
                                    fringeCost = fringeCost.add(lineItemCalculatedAmount.getCalculatedCost());
                                }
                                if ((lineItemCalculatedAmount.getRateClass().getCode().equals(
                                        rateClassCodeEmployeeBenefits) && !lineItemCalculatedAmount
                                        .getRateTypeCode().equals(
                                                rateTypeSupportStaffSalaries))
                                        || (lineItemCalculatedAmount.getRateClass().getCode().equals(
                                        rateClassCodeVacation) && !lineItemCalculatedAmount
                                        .getRateTypeCode().equals(
                                                rateTypeAdministrativeSalaries))) {
                                    if (lineItem.getSubmitCostSharingFlag()) {
                                        fringeCostSharingAmount = fringeCostSharingAmount.add(lineItemCalculatedAmount
                                                .getCalculatedCostSharing());
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        // Set the salary amounts
        bdSalary = bdSalary.add(salaryRequested).add(lineItemCost).add(mrLaCost);
        bdSalaryCostSharing = bdSalaryCostSharing.add(salaryCostSharing).add(lineItemCostSharingAmount).add(mrLaCostSharingAmount);

        // Set the fringe costs
        bdFringe = bdFringe.add(fringeCost).add(budgetLineItemFringeCost).add(mrLaFringeCost);
        bdFringeCostSharing = bdFringeCostSharing.add(fringeCostSharingAmount).add(budgetLineItemFringeCostSharingAmount)
                .add(mrLaFringeCostSharingAmount);

        bdNonFunds = bdSalaryCostSharing.add(bdFringeCostSharing);
        bdFunds = bdSalary.add(bdFringe);

        otherPersonnelInfo.setNumberPersonnel(personJobCodes.isEmpty() ? count : personJobCodes.size());
        otherPersonnelInfo.setPersonnelType(personnelType);
        otherPersonnelInfo.setRole(role);

        S2SCompensationDto compensationInfo = otherPersonnelInfo.getCompensation();
        // not sure that we need base salary
        compensationInfo.setBaseSalary(ScaleTwoDecimal.ZERO);
        compensationInfo.setFringe(bdFringe);
        compensationInfo.setFundsRequested(bdFunds);
        compensationInfo.setRequestedSalary(bdSalary);
        compensationInfo.setSummerMonths(new ScaleTwoDecimal(summerMonths));
        compensationInfo.setAcademicMonths(new ScaleTwoDecimal(academicMonths));
        compensationInfo.setCalendarMonths(new ScaleTwoDecimal(calendarMonths));

        // start add costSaring for fedNonFedBudget report
        compensationInfo.setFringeCostSharing(bdFringeCostSharing);
        compensationInfo.setNonFundsRequested(bdNonFunds);
        compensationInfo.setCostSharingAmount(bdSalaryCostSharing);
        // end add costSaring for fedNonFedBudget report

        otherPersonnelInfo.setCompensation(compensationInfo);
        return otherPersonnelInfo;
    }

    protected boolean getApplyRateFlagForRateBase(BudgetRateAndBaseContract rateBase, List<? extends BudgetLineItemCalculatedAmountContract> budgetLineItemCalculatedAmounts) {
        for (BudgetLineItemCalculatedAmountContract lineItemCalculatedAmount : budgetLineItemCalculatedAmounts) {
            if (lineItemCalculatedAmount.getBudgetLineItemId().equals(rateBase.getBudgetLineItemId()) && rateBase.getRateClassCode().equals(lineItemCalculatedAmount.getRateClass().getCode())) {
                return lineItemCalculatedAmount.getApplyRateFlag();
            }
        }
        return true;
    }

    protected void setCompensation(S2SKeyPersonDto keyPerson, BudgetPeriodContract budgetPeriod, BudgetContract budget) {
        S2SCompensationDto compensationInfo = keyPerson.getCompensation();
        BigDecimal summerMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        BigDecimal academicMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        BigDecimal calendarMonths = BigDecimal.ZERO.setScale(2, RoundingMode.HALF_UP);
        ScaleTwoDecimal totalSal = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fringe = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal baseAmount = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal totalSalCostSharing = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fringeCostSharing = ScaleTwoDecimal.ZERO;
        BigDecimal numberOfMonths;
        String budgetCatagoryCodePersonnel = s2SConfigurationService.getValueAsString(
                ConfigurationConstants.S2SBUDGET_BUDGET_CATEGORY_CODE_PERSONNEL);

        for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {

            for (BudgetPersonnelDetailsContract personDetails : lineItem.getBudgetPersonnelDetailsList()) {
                if (keyPersonEqualsBudgetPerson(keyPerson, personDetails)) {
                    numberOfMonths = s2SDateTimeService.getNumberOfMonths(personDetails.getStartDate(), personDetails.getEndDate()).bigDecimalValue();
                    if (personDetails.getPeriodTypeCode().equals(
                            s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_ACADEMIC_MONTHS))) {
                        academicMonths = getPersonEffortMonths(academicMonths, numberOfMonths, personDetails);
                    } else if (personDetails.getPeriodTypeCode().equals(
                            s2SConfigurationService.getValueAsString(
                                    ConfigurationConstants.S2SBUDGET_PERIOD_TYPE_SUMMER_MONTHS))) {
                        summerMonths = getPersonEffortMonths(summerMonths, numberOfMonths, personDetails);
                    } else {
                        if (StringUtils.isNotBlank(personDetails.getBudgetPerson().getTbnId())) {
                            if (lineItem.getBudgetCategory()
                                    .getCode().equals(budgetCatagoryCodePersonnel)) {
                                calendarMonths = getPersonEffortMonths(calendarMonths, numberOfMonths, personDetails);
                            }
                        } else {
                            calendarMonths = getPersonEffortMonths(calendarMonths, numberOfMonths, personDetails);
                        }
                    }
                    if (StringUtils.isNotBlank(personDetails.getBudgetPerson().getTbnId())) {
                        if (lineItem.getBudgetCategory()
                                .getCode().equals(budgetCatagoryCodePersonnel)) {
                            totalSal = totalSal.add(personDetails.getSalaryRequested());
                        }
                    } else {
                        totalSal = totalSal.add(personDetails.getSalaryRequested());
                    }
                    if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                        if (StringUtils.isNotBlank(personDetails.getBudgetPerson().getTbnId())) {
                            if (lineItem.getBudgetCategory()
                                    .getCode().equals(budgetCatagoryCodePersonnel)) {
                                totalSalCostSharing = totalSalCostSharing.add(personDetails.getCostSharingAmount());
                            }
                        } else {
                            totalSalCostSharing = totalSalCostSharing.add(personDetails.getCostSharingAmount());
                        }
                    }
                    for (BudgetPersonnelCalculatedAmountContract personCalculatedAmt : personDetails.getBudgetPersonnelCalculatedAmounts()) {
                        if ((personCalculatedAmt
                                .getRateClass()
                                .getRateClassType().getCode()
                                .equals(s2SConfigurationService.getValueAsString(
                                        ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_EMPLOYEE_BENEFITS)) && !personCalculatedAmt
                                .getRateTypeCode().equals(
                                        s2SConfigurationService.getValueAsString(
                                                ConfigurationConstants.S2SBUDGET_RATE_TYPE_SUPPORT_STAFF_SALARIES)))
                                || (personCalculatedAmt
                                .getRateClass()
                                .getRateClassType().getCode()
                                .equals(s2SConfigurationService.getValueAsString(
                                        ConfigurationConstants.S2SBUDGET_RATE_CLASS_TYPE_VACATION)) && !personCalculatedAmt
                                .getRateTypeCode().equals(
                                        s2SConfigurationService.getValueAsString(
                                                ConfigurationConstants.S2SBUDGET_RATE_TYPE_ADMINISTRATIVE_SALARIES)))) {
                            if (StringUtils.isNotBlank(personDetails.getBudgetPerson().getTbnId())) {
                                if (lineItem.getBudgetCategory()
                                        .getCode().equals(budgetCatagoryCodePersonnel)) {
                                    fringe = fringe.add(personCalculatedAmt.getCalculatedCost());
                                }
                            } else {
                                fringe = fringe.add(personCalculatedAmt.getCalculatedCost());
                            }
                            if (canBudgetLineItemCostSharingInclude(budget, lineItem)) {
                                if (StringUtils.isNotBlank(personDetails.getBudgetPerson().getTbnId())) {
                                    if (lineItem.getBudgetCategory()
                                            .getCode().equals(budgetCatagoryCodePersonnel)) {
                                        fringeCostSharing = fringeCostSharing.add(personCalculatedAmt.getCalculatedCostSharing());
                                    }
                                } else {
                                    fringeCostSharing = fringeCostSharing.add(personCalculatedAmt.getCalculatedCostSharing());
                                }
                            }
                        }
                    }
                    BudgetPersonContract budgetPerson = personDetails.getBudgetPerson();
                    if (budgetPerson != null) {
                        ScaleTwoDecimal baseSalaryByPeriod = getBaseSalaryByPeriod(budget.getBudgetId(), budgetPeriod.getBudgetPeriod(), keyPerson);
                        if (baseSalaryByPeriod != null) {
                            baseAmount = baseSalaryByPeriod;
                        } else if (budgetPerson.getCalculationBase() != null) {
                            baseAmount = budgetPerson.getCalculationBase();
                        }
                    }
                }

            }
        }
        compensationInfo.setAcademicMonths(new ScaleTwoDecimal(academicMonths));
        compensationInfo.setCalendarMonths(new ScaleTwoDecimal(calendarMonths));
        compensationInfo.setSummerMonths(new ScaleTwoDecimal(summerMonths));
        compensationInfo.setRequestedSalary(totalSal);
        compensationInfo.setBaseSalary(baseAmount);
        compensationInfo.setCostSharingAmount(totalSalCostSharing);
        compensationInfo.setFringe(fringe);
        compensationInfo.setFundsRequested(totalSal.add(fringe));
        compensationInfo.setFringeCostSharing(fringeCostSharing);
        compensationInfo.setNonFundsRequested(totalSalCostSharing.add(fringeCostSharing));
    }

    private ScaleTwoDecimal getBaseSalaryByPeriod(Long budgetId, int budgetPeriod, S2SKeyPersonDto person) {
        return budgetPersonSalaryService.findBaseSalaryForFirstPeriod(budgetId, person.getPersonId() != null ? person.getPersonId() : person.getRolodexId().toString(), budgetPeriod);
    }

    private boolean keyPersonEqualsBudgetPerson(S2SKeyPersonDto keyPersonInfo, BudgetPersonnelDetailsContract budgetPersonnelDetails) {
        boolean equal = false;
        if (keyPersonInfo != null && budgetPersonnelDetails != null) {
            String budgetPersonId = budgetPersonnelDetails.getPersonId();
            if ((keyPersonInfo.getPersonId() != null && keyPersonInfo.getPersonId().equals(budgetPersonId) && !keyPersonInfo.isTbn())
                    || (keyPersonInfo.getRolodexId() != null && keyPersonInfo.getRolodexId().toString().equals(budgetPersonId))
                    || (keyPersonInfo.getPersonId() != null && keyPersonInfo.getPersonId().equals(budgetPersonId) && keyPersonInfo.isTbn() && keyPersonInfo.getTbnName() != null && keyPersonInfo.getTbnName().equals(budgetPersonnelDetails.getBudgetPerson().getPersonName()))) {
                equal = true;
            }
        }
        return equal;
    }

    private void addUniqueKeyPerson(List<S2SKeyPersonDto> keyPersons, S2SKeyPersonDto keyPerson) {
        if (keyPersons.stream().noneMatch(p -> matches(p, keyPerson))) {
            keyPersons.add(keyPerson);
        }
    }

    private boolean matches(S2SKeyPersonDto keyPerson1, S2SKeyPersonDto keyPerson2) {
        final boolean personIdMatches = StringUtils.isNotBlank(keyPerson1.getPersonId()) && keyPerson1.getPersonId().equals(keyPerson2.getPersonId()) && !keyPerson1.isTbn() && keyPerson1.isTbn() == keyPerson2.isTbn();
        final boolean tbnIdMatches = StringUtils.isNotBlank(keyPerson1.getPersonId()) && keyPerson1.getPersonId().equals(keyPerson2.getPersonId())
                && StringUtils.isNotBlank(keyPerson1.getTbnName()) && keyPerson1.getTbnName().equals(keyPerson2.getTbnName())
                && keyPerson1.isTbn() && keyPerson1.isTbn() == keyPerson2.isTbn();
        final boolean rolodexIdMatches = keyPerson1.getRolodexId() != null && keyPerson1.getRolodexId().equals(keyPerson2.getRolodexId());
        return personIdMatches || rolodexIdMatches || tbnIdMatches;
    }

    private boolean isTbn(BudgetPersonContract person) {
        return StringUtils.isNotBlank(person.getTbnId());
    }


    private BigDecimal getPersonEffortMonths(BigDecimal effortMonths,
                                             BigDecimal numberOfMonths,
                                             BudgetPersonnelDetailsContract personDetails) {
        effortMonths = effortMonths.add(personDetails.getPercentEffort().bigDecimalValue()
                .multiply(numberOfMonths).multiply(POINT_ZERO_ONE));
        return effortMonths;
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }

    public S2SCommonBudgetService getS2SCommonBudgetService() {
        return s2SCommonBudgetService;
    }

    public void setS2SCommonBudgetService(S2SCommonBudgetService s2SCommonBudgetService) {
        this.s2SCommonBudgetService = s2SCommonBudgetService;
    }

    public OrganizationRepositoryService getOrganizationRepositoryService() {
        return organizationRepositoryService;
    }

    public void setOrganizationRepositoryService(OrganizationRepositoryService organizationRepositoryService) {
        this.organizationRepositoryService = organizationRepositoryService;
    }

    public ModularBudgetService getModularBudgetService() {
        return modularBudgetService;
    }

    public void setModularBudgetService(ModularBudgetService modularBudgetService) {
        this.modularBudgetService = modularBudgetService;
    }

    public BudgetCategoryMapService getBudgetCategoryMapService() {
        return budgetCategoryMapService;
    }

    public void setBudgetCategoryMapService(BudgetCategoryMapService budgetCategoryMapService) {
        this.budgetCategoryMapService = budgetCategoryMapService;
    }

    public BudgetPersonSalaryService getBudgetPersonSalaryService() {
        return budgetPersonSalaryService;
    }

    public void setBudgetPersonSalaryService(BudgetPersonSalaryService budgetPersonSalaryService) {
        this.budgetPersonSalaryService = budgetPersonSalaryService;
    }

    public SponsorHierarchyService getSponsorHierarchyService() {
        return sponsorHierarchyService;
    }

    public void setSponsorHierarchyService(SponsorHierarchyService sponsorHierarchyService) {
        this.sponsorHierarchyService = sponsorHierarchyService;
    }

    public KcPersonRepositoryService getKcPersonRepositoryService() {
        return kcPersonRepositoryService;
    }

    public void setKcPersonRepositoryService(KcPersonRepositoryService kcPersonRepositoryService) {
        this.kcPersonRepositoryService = kcPersonRepositoryService;
    }

    public TbnPersonService getTbnPersonService() {
        return tbnPersonService;
    }

    public void setTbnPersonService(TbnPersonService tbnPersonService) {
        this.tbnPersonService = tbnPersonService;
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public S2SProposalPersonService getS2SProposalPersonService() {
        return s2SProposalPersonService;
    }

    public void setS2SProposalPersonService(S2SProposalPersonService s2SProposalPersonService) {
        this.s2SProposalPersonService = s2SProposalPersonService;
    }

    public S2SBudgetCategoryMapService getS2SBudgetCategoryMapService() {
        return s2SBudgetCategoryMapService;
    }

    public void setS2SBudgetCategoryMapService(S2SBudgetCategoryMapService s2SBudgetCategoryMapService) {
        this.s2SBudgetCategoryMapService = s2SBudgetCategoryMapService;
    }
}
