/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrSF42412V12.*;
import gov.grants.apply.forms.rrSF42412V12.ApplicationTypeCodeDataType.Enum;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412.*;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412.ApplicantInfo.ContactPersonInfo;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412.ApplicantType.SmallBusinessOrganizationType;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412.ApplicantType.SmallBusinessOrganizationType.IsSociallyEconomicallyDisadvantaged;
import gov.grants.apply.forms.rrSF42412V12.RRSF42412Document.RRSF42412.ApplicantType.SmallBusinessOrganizationType.IsWomenOwned;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.ApplicantTypeCodeDataType;
import gov.grants.apply.system.globalLibraryV20.OrganizationDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.unit.UnitContract;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.common.budget.api.income.BudgetProjectIncomeContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemCalculatedAmountContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.budget.modular.BudgetModularContract;
import org.kuali.coeus.propdev.api.budget.modular.BudgetModularIdcContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityContract;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.validate.S2SErrorHandlerService;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.List;
import java.util.Map;

@FormGenerator("RRSF424V1_2Generator")
public class RRSF424V1_2Generator extends RRSF424BaseGenerator<RRSF42412Document> {

    private static final String MODULAR_BUDGET_REQUIRED = "s2s.budget.modularbudget.required";

    @Value("http://apply.grants.gov/forms/RR_SF424_1_2-V1.2")
    private String namespace;

    @Value("RR_SF424_1_2-V1.2")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_SF424-V1.2.xsl")
    private List<Resource> stylesheets;

    @Value("120")
    private int sortIndex;

    @Autowired
    @Qualifier("s2SErrorHandlerService")
    protected S2SErrorHandlerService s2SErrorHandlerService;

    private DepartmentalPersonDto departmentalPerson;
    private List<? extends AnswerHeaderContract> answerHeaders;

	private RRSF42412Document getRRSF424() {
        answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber());
        RRSF42412Document rrSF424Document = RRSF42412Document.Factory
				.newInstance();
		RRSF42412 rrsf42412 = RRSF42412.Factory.newInstance();
		rrsf42412.setFormVersion(FormVersion.v1_2.getVersion());
		rrsf42412.setSubmittedDate(departmentalPerson.getSubmittedDate());
		if(getSubmissionTypeCode() != null){
			rrsf42412.setSubmissionTypeCode(SubmissionTypeDataType.Enum.forInt(Integer.parseInt(getSubmissionTypeCode())));
		}
		rrsf42412.setStateID(getProposalSiteState());
		rrsf42412.setApplicantInfo(getApplicationInfo());
		rrsf42412.setEmployerID(getEmployerId());
		rrsf42412.setApplicantType(getApplicantType());
		if(getAgencyRoutingNumber()!=null){
	        rrsf42412.setAgencyRoutingNumber(getAgencyRoutingNumber());
	    }
		rrsf42412.setApplicationType(getApplicationType());
		rrsf42412.setApplicantID(pdDoc.getDevelopmentProposal()
				.getProposalNumber());
		rrsf42412.setFederalAgencyName(getFederalAgencyName());
		rrsf42412.setProjectTitle(getProjectTitle());
		rrsf42412.setProposedProjectPeriod(getProjectPeriod());
		rrsf42412.setCongressionalDistrict(getCongDistrict());
		final String cfdaNumber = getFirstCfdaNumber();
		if (cfdaNumber != null) {
			rrsf42412.setCFDANumber(cfdaNumber);
		}
		rrsf42412.setActivityTitle(getActivityTitle());
		setFederalId(rrsf42412);
		rrsf42412.setPDPIContactInfo(getPDPI());
		rrsf42412.setEstimatedProjectFunding(getProjectFunding());
		rrsf42412.setTrustAgree(YesNoDataType.Y_YES);// Value is hardcoded
		rrsf42412.setStateReview(getStateReview());
		rrsf42412.setAORInfo(getAORInfoType());
		rrsf42412.setAORSignature(getAORSignature());
		rrsf42412.setAORSignedDate(departmentalPerson.getSubmittedDate());
		setPreApplicationAttachment(rrsf42412);
		setSFLLLAttachment(rrsf42412);
		rrSF424Document.setRRSF42412(rrsf42412);
		return rrSF424Document;
	}

	private EstimatedProjectFunding getProjectFunding() {
		EstimatedProjectFunding funding = EstimatedProjectFunding.Factory
				.newInstance();
		funding.setTotalEstimatedAmount(BigDecimal.ZERO);
		funding.setTotalNonfedrequested(BigDecimal.ZERO);
		funding.setTotalfedNonfedrequested(BigDecimal.ZERO);
		funding.setEstimatedProgramIncome(BigDecimal.ZERO);
		boolean hasBudgetLineItem = false;
        ProposalDevelopmentBudgetExtContract budget = getS2SCommonBudgetService().getBudget(pdDoc.getDevelopmentProposal());

		if (budget != null) {

            ScaleTwoDecimal totalCost = ScaleTwoDecimal.ZERO;
            if (budget.getModularBudgetFlag()) {
				List<? extends BudgetModularContract> allModularAmounts = modularBudgetService.getModularBudgetAmounts(budget);

                ScaleTwoDecimal fundsRequested = ScaleTwoDecimal.ZERO;
                ScaleTwoDecimal totalDirectCost = ScaleTwoDecimal.ZERO;

				// get modular budget amounts instead of budget detail amounts
				for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
					BudgetModularContract modularAmounts = modularBudgetService.getModularBudgetForPeriod(allModularAmounts, budgetPeriod);
	                if (modularAmounts == null) {
	                    getAuditErrors().add(s2SErrorHandlerService.getError(MODULAR_BUDGET_REQUIRED, getFormName()));
	                    break;
	                } else {
					totalDirectCost = totalDirectCost.add(modularAmounts.getTotalDirectCost());
					for (BudgetModularIdcContract budgetModularIdc : modularAmounts.getBudgetModularIdcs()) {
						fundsRequested = fundsRequested.add(budgetModularIdc.getFundsRequested());
					}
				}
				}
				totalCost = totalCost.add(totalDirectCost);
				totalCost = totalCost.add(fundsRequested);
			} else {
            	 totalCost=budget.getTotalCost();
            }
            ScaleTwoDecimal fedNonFedCost = totalCost;
            ScaleTwoDecimal costSharingAmount = ScaleTwoDecimal.ZERO;

            for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    hasBudgetLineItem = true;
                    if (budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag()) {
                        costSharingAmount =  costSharingAmount.add(lineItem.getCostSharingAmount());
                        List<? extends BudgetLineItemCalculatedAmountContract> calculatedAmounts = lineItem.getBudgetLineItemCalculatedAmounts();
                        for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : calculatedAmounts) {
                             costSharingAmount =  costSharingAmount.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }

                    }
                }
            }
            if(!hasBudgetLineItem && budget.getSubmitCostSharingFlag()){
                costSharingAmount = budget.getCostSharingAmount();
            }
            fedNonFedCost = fedNonFedCost.add(costSharingAmount);
            funding = EstimatedProjectFunding.Factory.newInstance();
            funding.setTotalEstimatedAmount(totalCost
                    .bigDecimalValue());
            funding.setTotalNonfedrequested(costSharingAmount.bigDecimalValue());
            funding.setTotalfedNonfedrequested(fedNonFedCost.bigDecimalValue());
            funding.setEstimatedProgramIncome(getTotalProjectIncome(budget));
		}
		return funding;
	}

	private BigDecimal getTotalProjectIncome(BudgetContract budget) {
		BigDecimal totalProjectIncome = BigDecimal.ZERO;
		for (BudgetProjectIncomeContract budgetProjectIncome : budget
				.getBudgetProjectIncomes()) {
			totalProjectIncome = totalProjectIncome.add(budgetProjectIncome
					.getProjectIncome().bigDecimalValue());
		}
		return totalProjectIncome;
	}

	private ApplicantInfo getApplicationInfo() {
		ApplicantInfo appInfo = ApplicantInfo.Factory.newInstance();
		appInfo.setContactPersonInfo(getContactPersonInfo());
		appInfo.setOrganizationInfo(getOrganizationDataType());
		return appInfo;
	}

	private ContactPersonInfo getContactPersonInfo() {
		String contactType = getContactType();
		ContactPersonInfo contactInfo = ContactPersonInfo.Factory.newInstance();
		if (contactType.equals(CONTACT_TYPE_I)) {
			// use organization rolodex contact
			if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
				contactInfo = getContactInfo(pdDoc.getDevelopmentProposal()
						.getApplicantOrganization().getRolodex());
			}
		} else {
			// contact will come from unit or unit_administrators
			DepartmentalPersonDto depPerson = getContactPerson(pdDoc);
			if (depPerson != null) {
				contactInfo.setName(globLibV20Generator
						.getHumanNameDataType(depPerson));
				contactInfo.setPhone(depPerson.getOfficePhone());
				if (StringUtils.isNotEmpty(depPerson.getFaxNumber())) {
					contactInfo.setFax(depPerson.getFaxNumber());
				}
				if (depPerson.getEmailAddress() != null) {
					contactInfo.setEmail(depPerson.getEmailAddress());
				}
			}
		}
		return contactInfo;
	}

	private OrganizationDataType getOrganizationDataType() {
		OrganizationDataType orgType = OrganizationDataType.Factory
				.newInstance();
		ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
		orgType.setAddress(globLibV20Generator.getAddressDataType(proposalSite));

		OrganizationContract organization = proposalSite.getOrganization();
		if (organization != null) {
			orgType.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
			orgType.setDUNSID(organization.getDunsNumber());
		}
		final UnitContract leadUnit = pdDoc.getDevelopmentProposal().getOwnedByUnit();
		if (leadUnit != null) {
			orgType.setDepartmentName(getDepartmentName(pdDoc.getDevelopmentProposal().getOwnedByUnit()));

			final String divisionName = s2sDivisionService.getDivisionFromUnit(leadUnit.getUnitNumber());
			if (divisionName != null) {
				orgType.setDivisionName(divisionName);
			}
		}
		return orgType;
	}

	private ContactPersonInfo getContactInfo(RolodexContract rolodex) {
		ContactPersonInfo contactInfo = ContactPersonInfo.Factory.newInstance();
		contactInfo.setName(globLibV20Generator.getHumanNameDataType(rolodex));
		contactInfo.setPhone("");
		if (rolodex != null) {
			contactInfo.setPhone(rolodex.getPhoneNumber());
			if (StringUtils.isNotEmpty(rolodex.getFaxNumber())) {
				contactInfo.setFax(rolodex.getFaxNumber());
			}
			if (rolodex.getEmailAddress() != null) {
				contactInfo.setEmail(rolodex.getEmailAddress());
			}
		}
		return contactInfo;
	}

	private StateReview getStateReview() {
	    StateReview stateReview = StateReview.Factory.newInstance();
        Map<String, String> eoStateReview = getEOStateReview(pdDoc);
        StateReviewCodeTypeDataType.Enum stateReviewCodeType = null;
        String strReview = eoStateReview.get(YNQ_ANSWER);
        String stateReviewData;
        String stateReviewDate;
        Calendar reviewDate;
        
        if (STATE_REVIEW_YES.equals(strReview)) {
            stateReviewCodeType = StateReviewCodeTypeDataType.Y_YES;
            stateReviewDate = eoStateReview.get(YNQ_REVIEW_DATE);
            reviewDate = s2SDateTimeService.convertDateStringToCalendar(stateReviewDate);
            stateReview.setStateReviewDate(reviewDate);
        } else if (STATE_REVIEW_NO.equals(strReview)) {
            stateReviewData = eoStateReview.get(YNQ_STATE_REVIEW_DATA);
            if (YNQ_STATE_NOT_COVERED.equals(stateReviewData)) {
                stateReviewCodeType = StateReviewCodeTypeDataType.PROGRAM_IS_NOT_COVERED_BY_E_O_12372;
            } else if (YNQ_STATE_NOT_SELECTED.equals(stateReviewData)) {
                stateReviewCodeType = StateReviewCodeTypeDataType.PROGRAM_HAS_NOT_BEEN_SELECTED_BY_STATE_FOR_REVIEW;
            }
        }
        stateReview.setStateReviewCodeType(stateReviewCodeType);
        return stateReview;
    }

	private ApplicationType getApplicationType() {
		ApplicationType applicationType = ApplicationType.Factory.newInstance();
		Map<String, String> submissionInfo = getSubmissionType(pdDoc);
		String proposalTypeCode=pdDoc.getDevelopmentProposal().getProposalType().getCode();
		if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_REVISION).contains(proposalTypeCode)) {
			applicationType.setApplicationTypeCode(getApplicationTypeCodeDataType());
			String revisionCode;
			if (submissionInfo.get(KEY_REVISION_CODE) != null) {
				revisionCode = submissionInfo.get(KEY_REVISION_CODE);
				RevisionTypeCodeDataType.Enum revisionCodeApplication = RevisionTypeCodeDataType.Enum.forString(revisionCode);
				applicationType.setRevisionCode(revisionCodeApplication);
			}
			String revisionCodeOtherDesc;
			if (submissionInfo.get(KEY_REVISION_OTHER_DESCRIPTION) != null) {
				revisionCodeOtherDesc = submissionInfo.get(KEY_REVISION_OTHER_DESCRIPTION);
				applicationType.setRevisionCodeOtherExplanation(revisionCodeOtherDesc);
			}
		}
		if (pdDoc.getDevelopmentProposal().getProposalType() != null) {
			setProposalApplicationType(proposalTypeCode,applicationType);
		}
		setOtherAgencySubmissionDetails(applicationType);
		return applicationType;
	}
	
	private void setProposalApplicationType(String proposalTypeCode,ApplicationType applicationType) {
		if(s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_NEW).contains(proposalTypeCode)) {
			applicationType.setApplicationTypeCode(ApplicationTypeCodeDataType.Enum.forInt(ApplicationTypeCodeDataType.INT_NEW));
		}else if(s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_REVISION).contains(proposalTypeCode)) {
			applicationType.setApplicationTypeCode(ApplicationTypeCodeDataType.Enum.forInt(ApplicationTypeCodeDataType.INT_REVISION));
		}else if(s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_RENEWAL).contains(proposalTypeCode))  {
			applicationType.setApplicationTypeCode(ApplicationTypeCodeDataType.Enum.forInt(ApplicationTypeCodeDataType.INT_RENEWAL));
		}else if(s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_RESUBMISSION).contains(proposalTypeCode)) {
			applicationType.setApplicationTypeCode(ApplicationTypeCodeDataType.Enum.forInt(ApplicationTypeCodeDataType.INT_RESUBMISSION));
		}else if(s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_CONTINUATION).contains(proposalTypeCode)) {
			applicationType.setApplicationTypeCode(ApplicationTypeCodeDataType.Enum.forInt(ApplicationTypeCodeDataType.INT_CONTINUATION));
		}
	}

	private void setOtherAgencySubmissionDetails(ApplicationType applicationType) {
	    YesNoDataType.Enum answer = null;
	    String answerdetails = getAnswer(ANSWER_128, answerHeaders);
	    if(answerdetails != null && !answerdetails.equals(NOT_ANSWERED)){
	        answer =  answerdetails.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO;
	        applicationType.setIsOtherAgencySubmission(answer);
	    } else {
	        applicationType.setIsOtherAgencySubmission(null);
	    }


        if (answer != null && answer.equals(YesNoDataType.Y_YES)) {
            applicationType.setOtherAgencySubmissionExplanation(getOtherAgencySubmissionExplanation());
        }
	}

	private Enum getApplicationTypeCodeDataType() {
		return ApplicationTypeCodeDataType.Enum.forInt(Integer.parseInt(pdDoc
				.getDevelopmentProposal().getProposalType().getCode()));
	}

	private RRSF42412.ProposedProjectPeriod getProjectPeriod() {
		RRSF42412.ProposedProjectPeriod proposedProjectPeriod = RRSF42412.ProposedProjectPeriod.Factory
				.newInstance();
		proposedProjectPeriod.setProposedStartDate(s2SDateTimeService
				.convertDateToCalendar(pdDoc.getDevelopmentProposal()
						.getRequestedStartDateInitial()));
		proposedProjectPeriod.setProposedEndDate(s2SDateTimeService
				.convertDateToCalendar(pdDoc.getDevelopmentProposal()
						.getRequestedEndDateInitial()));
		return proposedProjectPeriod;
	}

	private RRSF42412.CongressionalDistrict getCongDistrict() {
		ProposalSiteContract applicantOrganization = pdDoc.getDevelopmentProposal()
				.getApplicantOrganization();
		RRSF42412.CongressionalDistrict congressionalDistrict = RRSF42412.CongressionalDistrict.Factory
				.newInstance();
		if (applicantOrganization != null) {
			congressionalDistrict
					.setApplicantCongressionalDistrict(applicantOrganization
							.getFirstCongressionalDistrictName());
		} else {
			congressionalDistrict.setApplicantCongressionalDistrict("");
		}
		return congressionalDistrict;
	}

	private OrganizationContactPersonDataType getPDPI() {
		OrganizationContactPersonDataType PDPI = OrganizationContactPersonDataType.Factory
				.newInstance();
		ProposalPersonContract PI;
		for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal()
				.getProposalPersons()) {
			if (PRINCIPAL_INVESTIGATOR.equals(proposalPerson
					.getProposalPersonRoleId())) {
				PI = proposalPerson;
				ProposalSiteContract applicantOrganization = pdDoc
						.getDevelopmentProposal().getApplicantOrganization();
				PDPI.setName(globLibV20Generator.getHumanNameDataType(PI));
				PDPI.setPhone(PI.getOfficePhone());
				PDPI.setEmail(PI.getEmailAddress());
				if (StringUtils.isNotEmpty(PI.getFaxNumber())) {
					PDPI.setFax(PI.getFaxNumber());
				}
				PDPI.setAddress(globLibV20Generator.getAddressDataType(PI));
				setDirectoryTitle(PDPI, PI);
				setDepartmentName(PDPI,PI);
				setDivisionName(PDPI,PI);
				if (applicantOrganization != null) {
					PDPI.setOrganizationName(StringUtils.substring(applicantOrganization.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
				}
			}
		}
		return PDPI;
	}

	private void setDivisionName(OrganizationContactPersonDataType PDPI,ProposalPersonContract PI) {
		final String divisionName = s2sDivisionService.getDivision(PI);
		if (divisionName != null) {
			PDPI.setDivisionName(divisionName);
		}
	}

	private void setDepartmentName(OrganizationContactPersonDataType PDPI,ProposalPersonContract PI) {
	    if(PI.getHomeUnit() != null) {
			PDPI.setDepartmentName(getDepartmentName(PI.getPerson()));
	    }
	    else
	    {
	        DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
	        PDPI.setDepartmentName(getDepartmentName(developmentProposal.getOwnedByUnit()));
	    }
	}

	private void setDirectoryTitle(OrganizationContactPersonDataType PDPI,
			ProposalPersonContract PI) {
		if (PI.getDirectoryTitle() != null) {
			if (PI.getDirectoryTitle().length() > DIRECTORY_TITLE_MAX_LENGTH) {
				PDPI.setTitle(PI.getDirectoryTitle().substring(0,
						DIRECTORY_TITLE_MAX_LENGTH));
			} else {
				PDPI.setTitle(PI.getDirectoryTitle());
			}
		}
	}

	private AORInfoType getAORInfoType() {
		ProposalSiteContract applicantOrganization = pdDoc.getDevelopmentProposal()
				.getApplicantOrganization();
		AORInfoType aorInfoType = AORInfoType.Factory.newInstance();
		if (departmentalPerson != null) {
			aorInfoType.setName(globLibV20Generator
					.getHumanNameDataType(departmentalPerson));

			setTitle(aorInfoType);
			setAddress(aorInfoType);
			setDivisionName(aorInfoType, departmentalPerson);
		}
		if (applicantOrganization != null) {
			aorInfoType.setOrganizationName(StringUtils.substring(applicantOrganization.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
		}

		return aorInfoType;
	}

	private void setTitle(AORInfoType aorInfoType) {
		if (departmentalPerson.getPrimaryTitle() != null) {
			aorInfoType.setTitle(StringUtils.substring(departmentalPerson.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
		} else {
			aorInfoType.setTitle("");
		}
	}

	private void setAddress(AORInfoType aorInfoType) {
		aorInfoType.setAddress(globLibV20Generator.getAddressDataType(departmentalPerson));
		aorInfoType.setPhone(departmentalPerson.getOfficePhone());
        if (StringUtils.isNotEmpty(departmentalPerson.getFaxNumber())) {
		    aorInfoType.setFax(departmentalPerson.getFaxNumber());
        }
		aorInfoType.setDepartmentName(getDepartmentName(departmentalPerson));
		aorInfoType.setEmail(departmentalPerson.getEmailAddress());
	}

	private void setDivisionName(AORInfoType aorInfoType, DepartmentalPersonDto departmentalPerson) {
		final String divisionName = getS2sDivisionService().getDivision(departmentalPerson);
		if (divisionName != null) {
			aorInfoType.setDivisionName(divisionName);
		}
	}

	private ApplicantType getApplicantType() {
		ApplicantType applicantType = ApplicantType.Factory.newInstance();
		SmallBusinessOrganizationType smallOrganizationType = SmallBusinessOrganizationType.Factory
				.newInstance();
		IsSociallyEconomicallyDisadvantaged isSociallyEconomicallyDisadvantaged = IsSociallyEconomicallyDisadvantaged.Factory
				.newInstance();
		IsWomenOwned isWomenOwned = IsWomenOwned.Factory.newInstance();
		boolean smallBusflag = false;
		int orgTypeCode = 0;
		if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null
				&& pdDoc.getDevelopmentProposal().getApplicantOrganization()
						.getOrganization().getOrganizationTypes() != null
				&& pdDoc.getDevelopmentProposal().getApplicantOrganization()
						.getOrganization().getOrganizationTypes().size() > 0) {
			orgTypeCode = pdDoc.getDevelopmentProposal()
					.getApplicantOrganization().getOrganization()
					.getOrganizationTypes().get(0).getOrganizationTypeList().getCode();
		}
		ApplicantTypeCodeDataType.Enum applicantTypeCode;

		switch (orgTypeCode) {
		case 1:
			applicantTypeCode = ApplicantTypeCodeDataType.C_CITY_OR_TOWNSHIP_GOVERNMENT;
			break;
		case 2:
			applicantTypeCode = ApplicantTypeCodeDataType.A_STATE_GOVERNMENT;
			break;
		case 3:
			applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
			break;
		case 4:
			applicantTypeCode = ApplicantTypeCodeDataType.M_NONPROFIT_WITH_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION;
			break;
		case 5:
			applicantTypeCode = ApplicantTypeCodeDataType.N_NONPROFIT_WITHOUT_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION;
			break;
		case 6:
			applicantTypeCode = ApplicantTypeCodeDataType.Q_FOR_PROFIT_ORGANIZATION_OTHER_THAN_SMALL_BUSINESS;
			break;
		case 7:
			applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
			break;
		case 8:
			applicantTypeCode = ApplicantTypeCodeDataType.I_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_FEDERALLY_RECOGNIZED;
			break;
		case 9:
			applicantTypeCode = ApplicantTypeCodeDataType.P_INDIVIDUAL;
			break;
		case 10:
			applicantTypeCode = ApplicantTypeCodeDataType.O_PRIVATE_INSTITUTION_OF_HIGHER_EDUCATION;
			break;
		case 11:
			applicantTypeCode = ApplicantTypeCodeDataType.R_SMALL_BUSINESS;
			break;
		case 14:
			applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
			isSociallyEconomicallyDisadvantaged.setStringValue(VALUE_YES);
			smallOrganizationType
					.setIsSociallyEconomicallyDisadvantaged(isSociallyEconomicallyDisadvantaged);
			smallBusflag = true;
			break;
		case 15:
			applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
			isWomenOwned.setStringValue(VALUE_YES);
			smallOrganizationType.setIsWomenOwned(isWomenOwned);
			smallBusflag = true;
			break;
		case 21:
			applicantTypeCode = ApplicantTypeCodeDataType.H_PUBLIC_STATE_CONTROLLED_INSTITUTION_OF_HIGHER_EDUCATION;
			break;
		case 22:
			applicantTypeCode = ApplicantTypeCodeDataType.B_COUNTY_GOVERNMENT;
			break;
		case 23:
			applicantTypeCode = ApplicantTypeCodeDataType.D_SPECIAL_DISTRICT_GOVERNMENT;
			break;
		case 24:
			applicantTypeCode = ApplicantTypeCodeDataType.G_INDEPENDENT_SCHOOL_DISTRICT;
			break;
		case 25:
			applicantTypeCode = ApplicantTypeCodeDataType.L_PUBLIC_INDIAN_HOUSING_AUTHORITY;
			break;
		case 26:
			applicantTypeCode = ApplicantTypeCodeDataType.J_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_OTHER_THAN_FEDERALLY_RECOGNIZED;
			break;
		case 100:
			applicantTypeCode = ApplicantTypeCodeDataType.E_REGIONAL_ORGANIZATION;
			break;
		case 101:
			applicantTypeCode = ApplicantTypeCodeDataType.F_U_S_TERRITORY_OR_POSSESSION;
			break;
		case 102:
			applicantTypeCode = ApplicantTypeCodeDataType.K_INDIAN_NATIVE_AMERICAN_TRIBALLY_DESIGNATED_ORGANIZATION;
			break;
		case 103:
			applicantTypeCode = ApplicantTypeCodeDataType.S_HISPANIC_SERVING_INSTITUTION;
			break;
		case 104:
			applicantTypeCode = ApplicantTypeCodeDataType.T_HISTORICALLY_BLACK_COLLEGES_AND_UNIVERSITIES_HBC_US;
			break;
		case 105:
			applicantTypeCode = ApplicantTypeCodeDataType.U_TRIBALLY_CONTROLLED_COLLEGES_AND_UNIVERSITIES_TCC_US;
			break;
		case 106:
			applicantTypeCode = ApplicantTypeCodeDataType.V_ALASKA_NATIVE_AND_NATIVE_HAWAIIAN_SERVING_INSTITUTIONS;
			break;
		case 107:
			applicantTypeCode = ApplicantTypeCodeDataType.W_NON_DOMESTIC_NON_US_ENTITY;
			break;
		default:
			applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
			break;
		}
		if (smallBusflag) {
			applicantType
					.setSmallBusinessOrganizationType(smallOrganizationType);
		}

		if (orgTypeCode == 3) {
			applicantType
					.setApplicantTypeCodeOtherExplanation("Federal Government");
		}
		applicantType.setApplicantTypeCode(applicantTypeCode);
		return applicantType;
	}

	private String getSubmissionTypeCode() {
		String submissionTypeCode = null;
		S2sOpportunityContract s2sOpportunity = pdDoc.getDevelopmentProposal()
				.getS2sOpportunity();
		if (s2sOpportunity != null
				&& s2sOpportunity.getS2sSubmissionType() != null) {
			submissionTypeCode = s2sOpportunity.getS2sSubmissionType().getCode();
		}
		return submissionTypeCode;
	}

	private String getProposalSiteState() {
		ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
		return (proposalSite != null && proposalSite.getState() != null) ? proposalSite.getState() : "";
	}

	private void setPreApplicationAttachment(RRSF42412 rrsf42412) {
		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
				.getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null
					&& Integer.parseInt(narrative.getNarrativeType().getCode()) == PRE_APPLICATION) {
				AttachedFileDataType preAttachment = addAttachedFileType(narrative);
				if(preAttachment != null){
					rrsf42412.setPreApplicationAttachment(preAttachment);
					break;
				}
			}
		}
	}
	private void setSFLLLAttachment(RRSF42412 rrsf42412) {
		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
				.getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null
					&& Integer.parseInt(narrative.getNarrativeType().getCode()) == SFLLL_OTHEREXPLANATORY) {
				AttachedFileDataType preAttachment = addAttachedFileType(narrative);
				if(preAttachment != null){
					rrsf42412.setSFLLLAttachment(preAttachment);
					break;
				}
			}
		}
	}

	private String getAORSignature() {
		String AORSignature = "";
		if (departmentalPerson != null) {
			AORSignature = departmentalPerson.getFullName();
		}
		return AORSignature;
	}

	private void setFederalId(RRSF42412 rrsf42412) {
		final String federalId = getFederalId();
		if (StringUtils.isNotBlank(federalId)) {
			rrsf42412.setFederalID(federalId);
		}
	}

	private String getActivityTitle() {
		String announcementTitle = "";
		if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle() != null) {
			if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle()
					.length() > 120) {
				announcementTitle = pdDoc.getDevelopmentProposal()
						.getProgramAnnouncementTitle().substring(0, 120);
			} else {
				announcementTitle = pdDoc.getDevelopmentProposal()
						.getProgramAnnouncementTitle();
			}
		}
		return announcementTitle;
	}

	private String getProjectTitle() {
		String title = pdDoc.getDevelopmentProposal().getTitle();
		if (title != null && title.length() > 200) {
			title = title.substring(0, 200);
		}
		return title;
	}
	private String getAgencyRoutingNumber(){
		return pdDoc.getDevelopmentProposal().getAgencyProgramCode();
	    }

	@Override
	public RRSF42412Document getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		departmentalPerson = departmentalPersonService
				.getDepartmentalPerson(proposalDevelopmentDocument);
		return getRRSF424();
	}


    @Override
    protected List<? extends AnswerHeaderContract> getAnswerHeaders() {
        return answerHeaders;
    }

    public S2SErrorHandlerService getS2SErrorHandlerService() {
        return s2SErrorHandlerService;
    }

    public void setS2SErrorHandlerService(S2SErrorHandlerService s2SErrorHandlerService) {
        this.s2SErrorHandlerService = s2SErrorHandlerService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

	@Override
	public DocumentFactory<RRSF42412Document> factory() {
		return RRSF42412Document.Factory;
	}
}
