/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrBudgetV11.*;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.*;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.Equipment.AdditionalEquipmentsAttachment;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.Equipment.EquipmentList;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.Equipment.TotalFundForAttachedEquipment;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.KeyPersons.AttachedKeyPersons;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.KeyPersons.TotalFundForAttachedKeyPersons;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.OtherDirectCosts.Others;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.OtherPersonnel.GraduateStudents;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.OtherPersonnel.PostDocAssociates;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.OtherPersonnel.SecretarialClerical;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.OtherPersonnel.UndergraduateStudents;
import gov.grants.apply.forms.rrBudgetV11.BudgetYearDataType.ParticipantTraineeSupportCosts.Other;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget.BudgetSummary;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget.BudgetSummary.CumulativeEquipments;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget.BudgetSummary.CumulativeOtherDirect;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget.BudgetSummary.CumulativeTrainee;
import gov.grants.apply.forms.rrBudgetV11.RRBudgetDocument.RRBudget.BudgetSummary.CumulativeTravels;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType.FileLocation;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import gov.grants.apply.system.globalV10.HashValueDocument;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.budget.*;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.budget.BudgetPeriodNum;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.util.UnsupportedOtherCostsErrorBuilder;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 
 * This class is used to generate XML Document object for grants.gov
 * RRBudgetV1.1. This form is generated using XMLBean API's generated by
 * compiling RRBudgetV1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("RRBudgetV1_1Generator")
public class RRBudgetV1_1Generator extends RRBudgetBaseGenerator<RRBudgetDocument> {

	private static final Logger LOG = LogManager.getLogger(RRBudgetV1_1Generator.class);
	private static final int SUPPORTED_OTHER_COSTS = 3;

	@Value("http://apply.grants.gov/forms/RR_Budget-V1.1")
    private String namespace;

    @Value("RR_Budget-V1.1")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_Budget-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("165")
    private int sortIndex;

	/**
	 * This method returns RRBudgetDocument object based on proposal development
	 * document which contains the informations such as
	 * DUNSID,OrganizationName,BudgetType,BudgetYear and BudgetSummary.
	 * 
	 * @return rrBudgetDocument {@link XmlObject} of type RRBudgetDocument.
	 */
	private RRBudgetDocument getRRBudget() {
		
		deleteAutoGenNarratives();
		RRBudgetDocument rrBudgetDocument = RRBudgetDocument.Factory
				.newInstance();
		RRBudget rrBudget = RRBudget.Factory.newInstance();
		rrBudget.setFormVersion(FormVersion.v1_1.getVersion());
		if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
			rrBudget.setDUNSID(pdDoc.getDevelopmentProposal()
					.getApplicantOrganization().getOrganization()
					.getDunsNumber());
			rrBudget.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));

		}
		rrBudget.setBudgetType(BudgetTypeDataType.PROJECT);
		// Set default values for mandatory fields
		rrBudget.setBudgetYear1(BudgetYear1DataType.Factory.newInstance());

		validateBudgetForForm(pdDoc);
		final S2SBudgetDto budgetSummary = s2sBudgetInfoService.getBudgetInfo(pdDoc);
		final List<S2SBudgetPeriodDto> budgetPeriodList = budgetSummary.getBudgetPeriods();

		for (S2SBudgetPeriodDto budgetPeriodData : budgetPeriodList) {
			if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
				rrBudget
						.setBudgetYear1(getBudgetYear1DataType(budgetPeriodData));
			} else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P2.getNum()) {
				rrBudget
						.setBudgetYear2(getBudgetYearDataType(budgetPeriodData));
			} else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P3.getNum()) {
				rrBudget
						.setBudgetYear3(getBudgetYearDataType(budgetPeriodData));
			} else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P4.getNum()) {
				rrBudget
						.setBudgetYear4(getBudgetYearDataType(budgetPeriodData));
			} else if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P5.getNum()) {
				rrBudget
						.setBudgetYear5(getBudgetYearDataType(budgetPeriodData));
			}
		}
		for (S2SBudgetPeriodDto budgetPeriodData : budgetPeriodList) {
		    if (budgetPeriodData.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
                rrBudget
                        .setBudgetYear1(getBudgetJustificationAttachment(rrBudget.getBudgetYear1()));
            }
		}
		rrBudget.setBudgetSummary(getBudgetSummary(budgetSummary));
		rrBudgetDocument.setRRBudget(rrBudget);
		return rrBudgetDocument;
	}

	/**
	 * This method gets BudgetYear1DataType details like
	 * BudgetPeriodStartDate,BudgetPeriodEndDate,BudgetPeriod
	 * KeyPersons,OtherPersonnel,TotalCompensation,Equipment,ParticipantTraineeSupportCosts,Travel,OtherDirectCosts
	 * DirectCosts,IndirectCosts,CognizantFederalAgency,TotalCosts 
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget summary entry.
	 * @return BudgetYear1DataType corresponding to the BudgetSummaryInfo
	 *         object.
	 */
	private BudgetYear1DataType getBudgetYear1DataType(
			S2SBudgetPeriodDto periodInfo) {

		BudgetYear1DataType budgetYear = BudgetYear1DataType.Factory
				.newInstance();
		if (periodInfo != null) {
			budgetYear.setBudgetPeriodStartDate(s2SDateTimeService
					.convertDateToCalendar(periodInfo.getStartDate()));
			budgetYear.setBudgetPeriodEndDate(s2SDateTimeService
					.convertDateToCalendar(periodInfo.getEndDate()));
			BudgetPeriod.Enum budgetPeriod = BudgetPeriod.Enum
					.forInt(periodInfo.getBudgetPeriod());
			budgetYear.setBudgetPeriod(budgetPeriod);
			budgetYear.setKeyPersons(getKeyPersons(periodInfo));
			budgetYear.setOtherPersonnel(getOtherPersonnel(periodInfo));
			if (periodInfo.getTotalCompensation() != null) {
				budgetYear.setTotalCompensation(periodInfo
						.getTotalCompensation().bigDecimalValue());
			}
			budgetYear.setEquipment(getEquipment(periodInfo));
			budgetYear.setTravel(getTravel(periodInfo));
			budgetYear
					.setParticipantTraineeSupportCosts(getParticipantTraineeSupportCosts(periodInfo));
			budgetYear.setOtherDirectCosts(getOtherDirectCosts(periodInfo));
			budgetYear.setDirectCosts(periodInfo.getDirectCostsTotal()
					.bigDecimalValue());
			IndirectCosts indirectCosts = getIndirectCosts(periodInfo);
			if (indirectCosts != null) {
				budgetYear.setIndirectCosts(indirectCosts);
				budgetYear.setTotalCosts(periodInfo.getDirectCostsTotal().bigDecimalValue().add(indirectCosts.getTotalIndirectCosts()));
			}else{
				budgetYear.setTotalCosts(periodInfo.getDirectCostsTotal().bigDecimalValue());
			}
			budgetYear.setCognizantFederalAgency(periodInfo
					.getCognizantFedAgency());
		}
		return budgetYear;
	}
	/*
	 * This method gets BudgetJustificationAttachment from ProposalDevelopmentDocumentContract for the RRBudget.
	 */
	private BudgetYear1DataType getBudgetJustificationAttachment(BudgetYear1DataType budgetYear) {

		AttachedFileDataType attachedFileDataType = AttachedFileDataType.Factory.newInstance();
		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null
					&& Integer.parseInt(narrative.getNarrativeType().getCode()) == BUDGET_JUSTIFICATION_ATTACHMENT) {
				attachedFileDataType = addAttachedFileType(narrative);
				if (attachedFileDataType != null) {
					break;
				}
			}
		}
        budgetYear.setBudgetJustificationAttachment(attachedFileDataType);
		return budgetYear;
	}

	/**
	 * This method gets BudgetYearDataType details like
	 * BudgetPeriodStartDate,BudgetPeriodEndDate,BudgetPeriod
	 * KeyPersons,OtherPersonnel,TotalCompensation,Equipment,ParticipantTraineeSupportCosts,Travel,OtherDirectCosts
	 * DirectCosts,IndirectCosts,CognizantFederalAgency,TotalCosts based on
	 * BudgetPeriodInfo for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return BudgetYear1DataType corresponding to the BudgetSummaryInfo
	 *         object.
	 */
	private BudgetYearDataType getBudgetYearDataType(S2SBudgetPeriodDto periodInfo) {

		BudgetYearDataType budgetYear = BudgetYearDataType.Factory
				.newInstance();
		if (periodInfo != null) {
			budgetYear.setBudgetPeriodStartDate(s2SDateTimeService
					.convertDateToCalendar(periodInfo.getStartDate()));
			budgetYear.setBudgetPeriodEndDate(s2SDateTimeService
					.convertDateToCalendar(periodInfo.getEndDate()));
			BudgetPeriod.Enum budgetPeriod = BudgetPeriod.Enum
					.forInt(periodInfo.getBudgetPeriod());
			budgetYear.setBudgetPeriod(budgetPeriod);
			budgetYear.setKeyPersons(getKeyPersons(periodInfo));
			budgetYear.setOtherPersonnel(getOtherPersonnel(periodInfo));
			if (periodInfo.getTotalCompensation() != null) {
				budgetYear.setTotalCompensation(periodInfo
						.getTotalCompensation().bigDecimalValue());
			}
			budgetYear.setEquipment(getEquipment(periodInfo));
			budgetYear.setTravel(getTravel(periodInfo));
			budgetYear
					.setParticipantTraineeSupportCosts(getParticipantTraineeSupportCosts(periodInfo));
			budgetYear.setOtherDirectCosts(getOtherDirectCosts(periodInfo));
			BigDecimal directCosts = periodInfo.getDirectCostsTotal()
					.bigDecimalValue();
			budgetYear.setDirectCosts(directCosts);
			IndirectCosts indirectCosts = getIndirectCosts(periodInfo);
			budgetYear.setIndirectCosts(indirectCosts);
			budgetYear.setCognizantFederalAgency(periodInfo
					.getCognizantFedAgency());
			if(indirectCosts != null){
				budgetYear.setTotalCosts(directCosts.add(indirectCosts.getTotalIndirectCosts()));
			}else{
				budgetYear.setTotalCosts(periodInfo.getTotalCosts()
						.bigDecimalValue());
			}
		}
		return budgetYear;
	}

	/**
	 * This method gets BudgetSummary details such as
	 * CumulativeTotalFundsRequestedSeniorKeyPerson,CumulativeTotalFundsRequestedOtherPersonnel
	 * CumulativeTotalNoOtherPersonnel,CumulativeTotalFundsRequestedPersonnel,CumulativeEquipments,CumulativeTravels
	 * CumulativeTrainee,CumulativeOtherDirect,CumulativeTotalFundsRequestedDirectCosts,CumulativeTotalFundsRequestedIndirectCost
	 * CumulativeTotalFundsRequestedDirectIndirectCosts and CumulativeFee based
	 * on BudgetSummaryInfo for the RRBudget.
	 * 
	 * @param budgetSummaryData
	 *            (BudgetSummaryInfo) budget summary entry.
	 * @return BudgetSummary details corresponding to the BudgetSummaryInfo
	 *         object.
	 */
	private BudgetSummary getBudgetSummary(S2SBudgetDto budgetSummaryData) {

		BudgetSummary budgetSummary = BudgetSummary.Factory.newInstance();
		// Set default values for mandatory fields
		budgetSummary
				.setCumulativeTotalFundsRequestedSeniorKeyPerson(BigDecimal.ZERO);
		budgetSummary
				.setCumulativeTotalFundsRequestedPersonnel(BigDecimal.ZERO);
		budgetSummary
				.setCumulativeTotalFundsRequestedDirectCosts(BigDecimal.ZERO);

		if (budgetSummaryData.getCumTotalFundsForSrPersonnel() != null) {
			budgetSummary
					.setCumulativeTotalFundsRequestedSeniorKeyPerson(budgetSummaryData
							.getCumTotalFundsForSrPersonnel().bigDecimalValue());
		}
		if (budgetSummaryData.getCumTotalFundsForOtherPersonnel() != null) {
			budgetSummary
					.setCumulativeTotalFundsRequestedOtherPersonnel(budgetSummaryData
							.getCumTotalFundsForOtherPersonnel()
							.bigDecimalValue());
		}
		if (budgetSummaryData.getCumNumOtherPersonnel() != null) {
			budgetSummary.setCumulativeTotalNoOtherPersonnel(budgetSummaryData
					.getCumNumOtherPersonnel().intValue());
		}
		if (budgetSummaryData.getCumTotalFundsForPersonnel() != null) {
			budgetSummary
					.setCumulativeTotalFundsRequestedPersonnel(budgetSummaryData
							.getCumTotalFundsForPersonnel().bigDecimalValue());
		}
		budgetSummary
				.setCumulativeEquipments(getCumulativeEquipments(budgetSummaryData));
		budgetSummary
				.setCumulativeTravels(getCumulativeTravels(budgetSummaryData));
		budgetSummary
				.setCumulativeTrainee(getCumulativeTrainee(budgetSummaryData));
		budgetSummary
				.setCumulativeOtherDirect(getCumulativeOtherDirect(budgetSummaryData));
		budgetSummary
				.setCumulativeTotalFundsRequestedDirectCosts(budgetSummaryData
						.getCumTotalDirectCosts().bigDecimalValue());
		budgetSummary
				.setCumulativeTotalFundsRequestedIndirectCost(budgetSummaryData
						.getCumTotalIndirectCosts().bigDecimalValue());
		budgetSummary
				.setCumulativeTotalFundsRequestedDirectIndirectCosts(budgetSummaryData
						.getCumTotalCosts().bigDecimalValue());
		if (budgetSummaryData.getCumFee() != null) {
			budgetSummary.setCumulativeFee(budgetSummaryData.getCumFee()
					.bigDecimalValue());
		}
		return budgetSummary;
	}

	/**
	 * This method gets CumulativeEquipments information
	 * CumulativeTotalFundsRequestedEquipment based on BudgetSummaryInfo for the
	 * form RRBudget.
	 * 
	 * @param budgetSummaryData
	 *            (BudgetSummaryInfo) budget summary entry.
	 * @return CumulativeEquipments details corresponding to the
	 *         BudgetSummaryInfo object.
	 */
	private CumulativeEquipments getCumulativeEquipments(
			S2SBudgetDto budgetSummaryData) {

		CumulativeEquipments cumulativeEquipments = CumulativeEquipments.Factory
				.newInstance();
		if (budgetSummaryData != null
				&& budgetSummaryData.getCumEquipmentFunds() != null) {
			cumulativeEquipments
					.setCumulativeTotalFundsRequestedEquipment(budgetSummaryData
							.getCumEquipmentFunds().bigDecimalValue());
		}
		return cumulativeEquipments;
	}

	/**
	 * This method gets CumulativeTravels
	 * details,CumulativeTotalFundsRequestedTravel,CumulativeDomesticTravelCosts
	 * and CumulativeForeignTravelCosts based on BudgetSummaryInfo for the
	 * RRBudget.
	 * 
	 * @param budgetSummaryData
	 *            (BudgetSummaryInfo) budget summary entry.
	 * @return CumulativeTravels details corresponding to the BudgetSummaryInfo
	 *         object.
	 */
	private CumulativeTravels getCumulativeTravels(
			S2SBudgetDto budgetSummaryData) {
		CumulativeTravels cumulativeTravels = CumulativeTravels.Factory
				.newInstance();
		if (budgetSummaryData.getCumDomesticTravel() != null) {
			cumulativeTravels
					.setCumulativeDomesticTravelCosts(budgetSummaryData
							.getCumDomesticTravel().bigDecimalValue());
		}
		if (budgetSummaryData.getCumForeignTravel() != null) {
			cumulativeTravels.setCumulativeForeignTravelCosts(budgetSummaryData
					.getCumForeignTravel().bigDecimalValue());
		}
		if (budgetSummaryData.getCumTravel() != null) {
			cumulativeTravels
					.setCumulativeTotalFundsRequestedTravel(budgetSummaryData
							.getCumTravel().bigDecimalValue());
		}
		return cumulativeTravels;
	}

	/**
	 * This method gets CumulativeTrainee details,
	 * CumulativeTotalFundsRequestedTraineeCosts,CumulativeTraineeTuitionFeesHealthInsurance
	 * CumulativeTraineeStipends,CumulativeTraineeTravel,CumulativeTraineeSubsistence,CumulativeOtherTraineeCost
	 * and CumulativeNoofTrainees based on BudgetSummaryInfo for the RRBudget.
	 * 
	 * @param budgetSummaryData
	 *            (BudgetSummaryInfo) budget summary entry.
	 * @return CumulativeTrainee details corresponding to the BudgetSummaryInfo
	 *         object.
	 */
	private CumulativeTrainee getCumulativeTrainee(
			S2SBudgetDto budgetSummaryData) {
		CumulativeTrainee cumulativeTrainee = CumulativeTrainee.Factory
				.newInstance();
		cumulativeTrainee
				.setCumulativeTotalFundsRequestedTraineeCosts(BigDecimal.ZERO);
		if (budgetSummaryData != null) {
			cumulativeTrainee
					.setCumulativeTotalFundsRequestedTraineeCosts(budgetSummaryData
							.getPartOtherCost()
							.add(
									budgetSummaryData
											.getPartStipendCost()
											.add(
													budgetSummaryData
															.getPartTravelCost()
															.add(
																	budgetSummaryData
																			.getPartTuition()
																			.add(
																					budgetSummaryData
																							.getPartSubsistence()))))
							.bigDecimalValue());
			cumulativeTrainee
					.setCumulativeTraineeTuitionFeesHealthInsurance(budgetSummaryData
							.getPartTuition().bigDecimalValue());
			cumulativeTrainee.setCumulativeTraineeStipends(budgetSummaryData
					.getPartStipendCost().bigDecimalValue());
			cumulativeTrainee.setCumulativeTraineeTravel(budgetSummaryData
					.getPartTravelCost().bigDecimalValue());
			cumulativeTrainee.setCumulativeTraineeSubsistence(budgetSummaryData
					.getPartSubsistence().bigDecimalValue());
			cumulativeTrainee.setCumulativeOtherTraineeCost(budgetSummaryData
					.getPartOtherCost().bigDecimalValue());
			cumulativeTrainee.setCumulativeNoofTrainees(budgetSummaryData
					.getParticipantCount());
		}
		return cumulativeTrainee;
	}

	/**
	 * This method gets CumulativeOtherDirectCost
	 * details,CumulativeMaterialAndSupplies,CumulativePublicationCosts,
	 * CumulativeConsultantServices,CumulativeADPComputerServices,CumulativeSubawardConsortiumContractualCosts
	 * CumulativeEquipmentFacilityRentalFees,CumulativeAlterationsAndRenovations
	 * and CumulativeOther1DirectCost based on BudgetSummaryInfo for the
	 * RRBudget.
	 * 
	 * @param budgetSummaryData
	 *            (BudgetSummaryInfo) budget summary entry.
	 * @return CumulativeOtherDirect details corresponding to the
	 *         BudgetSummaryInfo object.
	 */
	private CumulativeOtherDirect getCumulativeOtherDirect(
			S2SBudgetDto budgetSummaryData) {
		CumulativeOtherDirect cumulativeOtherDirect = CumulativeOtherDirect.Factory
				.newInstance();
		cumulativeOtherDirect
				.setCumulativeTotalFundsRequestedOtherDirectCosts(BigDecimal.ZERO);
		if (budgetSummaryData != null
				&& budgetSummaryData.getOtherDirectCosts() != null) {
			for (S2SOtherDirectCostInfoDto cumOtherDirect : budgetSummaryData
					.getOtherDirectCosts()) {
				cumulativeOtherDirect
						.setCumulativeTotalFundsRequestedOtherDirectCosts(cumOtherDirect
								.getTotalOtherDirect().bigDecimalValue());
				if (cumOtherDirect.getMaterials() != null) {
					cumulativeOtherDirect
							.setCumulativeMaterialAndSupplies(cumOtherDirect
									.getMaterials().bigDecimalValue());
				}
				if (cumOtherDirect.getPublications() != null) {
					cumulativeOtherDirect
							.setCumulativePublicationCosts(cumOtherDirect
									.getPublications().bigDecimalValue());
				}
				if (cumOtherDirect.getConsultants() != null) {
					cumulativeOtherDirect
							.setCumulativeConsultantServices(cumOtherDirect
									.getConsultants().bigDecimalValue());
				}
				if (cumOtherDirect.getComputer() != null) {
					cumulativeOtherDirect
							.setCumulativeADPComputerServices(cumOtherDirect
									.getComputer().bigDecimalValue());
				}
				if (cumOtherDirect.getSubAwards() != null) {
					cumulativeOtherDirect
							.setCumulativeSubawardConsortiumContractualCosts(cumOtherDirect
									.getSubAwards().bigDecimalValue());
				}
				if (cumOtherDirect.getEquipRental() != null) {
					cumulativeOtherDirect
							.setCumulativeEquipmentFacilityRentalFees(cumOtherDirect
									.getEquipRental().bigDecimalValue());
				}
				if (cumOtherDirect.getAlterations() != null) {
					cumulativeOtherDirect
							.setCumulativeAlterationsAndRenovations(cumOtherDirect
									.getAlterations().bigDecimalValue());
				}

				List<S2SOtherCostDto> cvOthers = cumOtherDirect.getOtherCosts();
				if (cvOthers.size() > SUPPORTED_OTHER_COSTS) {
					getAuditErrors().add(UnsupportedOtherCostsErrorBuilder.build(SUPPORTED_OTHER_COSTS));
				} else {
					for (int j = 0; j < cvOthers.size(); j++) {
						S2SOtherCostDto hmCosts = cvOthers.get(j);
						if (j == 0) {
							cumulativeOtherDirect.setCumulativeOther1DirectCost(hmCosts.getCost().bigDecimalValue());
						} else if (j == 1) {
							cumulativeOtherDirect.setCumulativeOther2DirectCost(hmCosts.getCost().bigDecimalValue());
						} else {
							cumulativeOtherDirect.setCumulativeOther3DirectCost(hmCosts.getCost().bigDecimalValue());
						}
					}
				}
			}
		}
		return cumulativeOtherDirect;
	}

	/**
	 * This method gets ParticipantTraineeSupportCosts details in
	 * BudgetYearDataType such as TuitionFeeHealthInsurance
	 * Stipends,Subsistence,Travel,Other,ParticipantTraineeNumber and TotalCost
	 * based on the BudgetPeriodInfo for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return ParticipantTraineeSupportCosts corresponding to the
	 *         BudgetPeriodInfo object.
	 */
	private ParticipantTraineeSupportCosts getParticipantTraineeSupportCosts(
			S2SBudgetPeriodDto periodInfo) {

		ParticipantTraineeSupportCosts traineeSupportCosts = ParticipantTraineeSupportCosts.Factory
				.newInstance();
		if (periodInfo != null) {
			traineeSupportCosts.setTuitionFeeHealthInsurance(periodInfo
					.getPartTuition().bigDecimalValue());
			traineeSupportCosts.setStipends(periodInfo.getPartStipendCost()
					.bigDecimalValue());
			traineeSupportCosts.setTravel(periodInfo.getPartTravelCost()
					.bigDecimalValue());
			traineeSupportCosts.setSubsistence(periodInfo.getPartSubsistence()
					.bigDecimalValue());
			traineeSupportCosts.setOther(getOtherPTSupportCosts(periodInfo));
			traineeSupportCosts.setParticipantTraineeNumber(periodInfo
					.getParticipantCount());
			traineeSupportCosts
					.setTotalCost(traineeSupportCosts
							.getTuitionFeeHealthInsurance()
							.add(
									traineeSupportCosts
											.getStipends()
											.add(
													traineeSupportCosts
															.getTravel()
															.add(
																	traineeSupportCosts
																			.getSubsistence()
																			.add(
																					traineeSupportCosts
																							.getOther()
																							.getCost())))));
		}
		return traineeSupportCosts;
	}

	/**
	 * This method gets Other type description and total cost for
	 * ParticipantTraineeSupportCosts based on BudgetPeriodInfo.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return Other other participant trainee support costs corresponding to
	 *         the BudgetPeriodInfo object.
	 */
	private Other getOtherPTSupportCosts(S2SBudgetPeriodDto periodInfo) {
		Other other = Other.Factory.newInstance();
		other.setDescription(OTHERCOST_DESCRIPTION);
		ScaleTwoDecimal otherCost = ScaleTwoDecimal.ZERO;
		if (periodInfo != null && periodInfo.getPartOtherCost() != null) {
			otherCost = periodInfo.getPartOtherCost();
		}
		other.setCost(otherCost.bigDecimalValue());
		return other;
	}

	/**
	 * This method gets OtherDirectCosts details such as
	 * PublicationCosts,MaterialsSupplies,ConsultantServices,
	 * ADPComputerServices,SubawardConsortiumContractualCosts,EquipmentRentalFee,AlterationsRenovations
	 * and TotalOtherDirectCost in BudgetYearDataType based on BudgetPeriodInfo
	 * for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return OtherDirectCosts corresponding to the BudgetPeriodInfo object.
	 */
	private OtherDirectCosts getOtherDirectCosts(S2SBudgetPeriodDto periodInfo) {

		OtherDirectCosts otherDirectCosts = OtherDirectCosts.Factory
				.newInstance();
		if (periodInfo != null && periodInfo.getOtherDirectCosts().size() > 0) {
			if (periodInfo.getOtherDirectCosts().get(0).getPublications() != null) {
				otherDirectCosts.setPublicationCosts(periodInfo
						.getOtherDirectCosts().get(0).getPublications()
						.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getMaterials() != null) {
				otherDirectCosts.setMaterialsSupplies(periodInfo
						.getOtherDirectCosts().get(0).getMaterials()
						.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getConsultants() != null) {
				otherDirectCosts.setConsultantServices(periodInfo
						.getOtherDirectCosts().get(0).getConsultants()
						.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getComputer() != null) {
				otherDirectCosts.setADPComputerServices(periodInfo
						.getOtherDirectCosts().get(0).getComputer()
						.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getSubAwards() != null) {
				otherDirectCosts
						.setSubawardConsortiumContractualCosts(periodInfo
								.getOtherDirectCosts().get(0).getSubAwards()
								.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getAlterations() != null) {
				otherDirectCosts.setAlterationsRenovations(periodInfo
						.getOtherDirectCosts().get(0).getAlterations()
						.bigDecimalValue());
			}
			if (periodInfo.getOtherDirectCosts().get(0).getEquipRental() != null) {
				otherDirectCosts.setEquipmentRentalFee(periodInfo
						.getOtherDirectCosts().get(0).getEquipRental()
						.bigDecimalValue());
			}
			otherDirectCosts
					.setOthers(getOthersForOtherDirectCosts(periodInfo));
			if (periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect() != null) {
				otherDirectCosts.setTotalOtherDirectCost(periodInfo
						.getOtherDirectCosts().get(0).getTotalOtherDirect()
						.bigDecimalValue());
			}
		}
		return otherDirectCosts;
	}

	/**
	 * This method returns IndirectCosts details such as
	 * Base,CostType,FundRequested,Rate and TotalIndirectCosts in
	 * BudgetYearDataType based on BudgetPeriodInfo for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return IndirectCosts corresponding to the BudgetPeriodInfo object.
	 */
	private IndirectCosts getIndirectCosts(S2SBudgetPeriodDto periodInfo) {

		IndirectCosts indirectCosts = null;

		if (periodInfo != null
				&& periodInfo.getIndirectCosts() != null
				&& periodInfo.getIndirectCosts().getIndirectCostDetails() != null) {

			List<IndirectCosts.IndirectCost> indirectCostList = new ArrayList<>();
			int IndirectCostCount = 0;
			for (S2SIndirectCostDetailsDto indirectCostDetails : periodInfo
					.getIndirectCosts().getIndirectCostDetails()) {
				IndirectCosts.IndirectCost indirectCost = IndirectCosts.IndirectCost.Factory
						.newInstance();
				if (indirectCostDetails.getBase() != null) {
					indirectCost.setBase(indirectCostDetails.getBase()
							.bigDecimalValue());
				}
				indirectCost.setCostType(indirectCostDetails.getCostType());
				if (indirectCostDetails.getFunds() != null) {
					indirectCost.setFundRequested(indirectCostDetails
							.getFunds().bigDecimalValue());
				}
				if (indirectCostDetails.getRate() != null) {
					indirectCost.setRate(indirectCostDetails.getRate()
							.bigDecimalValue());
				}
				indirectCostList.add(indirectCost);
				IndirectCostCount++;
				if (IndirectCostCount == ARRAY_LIMIT_IN_SCHEMA) {
					LOG
							.warn("Stopping iteration over indirect cost details because array limit in schema is only 4");
					break;
				}
			}
			if (IndirectCostCount > 0) {
				indirectCosts = IndirectCosts.Factory.newInstance();

				indirectCosts.setIndirectCostArray(indirectCostList
						.toArray(new IndirectCosts.IndirectCost[0]));
				if (periodInfo.getIndirectCosts().getTotalIndirectCosts() != null) {
					indirectCosts.setTotalIndirectCosts(periodInfo
							.getIndirectCosts().getTotalIndirectCosts()
							.bigDecimalValue());
				}
			}
		}
		return indirectCosts;
	}

	private Others getOthersForOtherDirectCosts(S2SBudgetPeriodDto periodInfo) {
		Others others = Others.Factory.newInstance();

		if (periodInfo != null && periodInfo.getOtherDirectCosts() != null) {
			for (S2SOtherDirectCostInfoDto otherDirectCostInfo : periodInfo.getOtherDirectCosts()) {
				if (CollectionUtils.isNotEmpty(otherDirectCostInfo.getOtherCosts())) {
					for (S2SOtherCostDto oc : otherDirectCostInfo.getOtherCosts()) {
						final Others.Other other = others.addNewOther();
						other.setCost(oc.getCost().bigDecimalValue());
						other.setDescription(oc.getDescription());
					}
				}
			}
		}
		return others;
	}

	/**
	 * This method gets Travel cost information including
	 * DomesticTravelCost,ForeignTravelCost and TotalTravelCost in the
	 * BudgetYearDataType based on BudgetPeriodInfo for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return Travel costs corresponding to the BudgetPeriodInfo object.
	 */
	private Travel getTravel(S2SBudgetPeriodDto periodInfo) {

		Travel travel = Travel.Factory.newInstance();
		if (periodInfo != null) {
			travel.setDomesticTravelCost(periodInfo.getDomesticTravelCost()
					.bigDecimalValue());
			travel.setForeignTravelCost(periodInfo.getForeignTravelCost()
					.bigDecimalValue());
			travel.setTotalTravelCost(periodInfo.getTotalTravelCost()
					.bigDecimalValue());
		}
		return travel;
	}

	/**
	 * This method gets Equipment details such as
	 * EquipmentItem,FundsRequested,TotalFundForAttachedEquipment, TotalFund and
	 * AdditionalEquipmentsAttachment based on BudgetPeriodInfo for the
	 * RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return Equipment costs corresponding to the BudgetPeriodInfo object.
	 * 
	 */
	private Equipment getEquipment(S2SBudgetPeriodDto periodInfo) {
		Equipment equipment = Equipment.Factory.newInstance();
		NarrativeContract extraEquipmentNarr = null;
		if (periodInfo != null && periodInfo.getEquipment() != null
				&& periodInfo.getEquipment().size() > 0) {
			// Evaluating Equipments.
			List<EquipmentList> equipmentArrayList = new ArrayList<>();
			ScaleTwoDecimal totalFund = ScaleTwoDecimal.ZERO;
			for (S2SCostDto costInfo : periodInfo.getEquipment().get(0)
					.getEquipmentList()) {
				EquipmentList equipmentList = EquipmentList.Factory.newInstance();
				equipmentList.setEquipmentItem(costInfo.getDescription());
				if (costInfo.getCost() != null) {
					equipmentList.setFundsRequested(costInfo.getCost().bigDecimalValue());
				}
				totalFund = totalFund.add(costInfo.getCost());
				equipmentArrayList.add(equipmentList);
			}

			// Evaluating Extra Equipments.
			List<S2SCostDto> extraEquipmentArrayList = new ArrayList<>();
			ScaleTwoDecimal totalExtraEquipFund = ScaleTwoDecimal.ZERO;
			for(S2SCostDto costInfo:periodInfo.getEquipment().get(0).getExtraEquipmentList()){
				extraEquipmentArrayList.add(costInfo);
				totalExtraEquipFund = totalExtraEquipFund.add(costInfo.getCost());
			}

			EquipmentList[] equipmentArray = equipmentArrayList.toArray(new EquipmentList[0]);
			equipment.setEquipmentListArray(equipmentArray);
			TotalFundForAttachedEquipment totalFundForAttachedEquipment = TotalFundForAttachedEquipment.Factory
					.newInstance();
			totalFundForAttachedEquipment
					.setTotalFundForAttachedEquipmentExist(YesNoDataType.Y_YES);
			totalFundForAttachedEquipment.setBigDecimalValue(totalExtraEquipFund.bigDecimalValue());
			equipment.setTotalFundForAttachedEquipment(totalFundForAttachedEquipment);
			totalFund = totalFund.add(totalExtraEquipFund);
			equipment.setTotalFund(totalFund.bigDecimalValue());
			extraEquipmentNarr = saveAdditionalEquipments(periodInfo,extraEquipmentArrayList);
		}

		if(extraEquipmentNarr!=null){
			AdditionalEquipmentsAttachment equipmentAttachment = AdditionalEquipmentsAttachment.Factory
					.newInstance();
			FileLocation fileLocation = FileLocation.Factory.newInstance();
			equipmentAttachment.setFileLocation(fileLocation);
			String contentId = createContentId(extraEquipmentNarr);
			fileLocation.setHref(contentId);
			equipmentAttachment.setFileLocation(fileLocation);
			equipmentAttachment.setFileName(extraEquipmentNarr.getNarrativeAttachment().getName());
			equipmentAttachment.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);
			if (extraEquipmentNarr.getNarrativeAttachment() != null) {
				final HashValueDocument.HashValue hashValue = getHashValue(extraEquipmentNarr.getNarrativeAttachment().getData());
				equipmentAttachment.setHashValue(hashValue);

				addAttachment(new AttachmentData(extraEquipmentNarr.getNarrativeAttachment().getFileDataId(), extraEquipmentNarr.getNarrativeAttachment().getName(), contentId, extraEquipmentNarr
						.getNarrativeAttachment().getData(), InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), extraEquipmentNarr.getNarrativeAttachment().getUploadUser(), extraEquipmentNarr.getNarrativeAttachment().getUploadTimestamp()));
				equipmentAttachment.setTotalFundForAttachedEquipmentExist(YesNoDataType.Y_YES);
				equipment.setAdditionalEquipmentsAttachment(equipmentAttachment);
			}
		}
		return equipment;
	}

	/**
	 * This method gets OtherPersonnel informations like
	 * PostDocAssociates,GraduateStudents,UndergraduateStudents
	 * SecretarialClerical based on PersonnelType and also gets
	 * NumberOfPersonnel, ProjectRole,Compensation OtherPersonnelTotalNumber and
	 * TotalOtherPersonnelFund based on BudgetPeriodInfo for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return OtherPersonnel details corresponding to the BudgetPeriodInfo
	 *         object.
	 */
	private OtherPersonnel getOtherPersonnel(S2SBudgetPeriodDto periodInfo) {
		OtherPersonnel otherPersonnel = OtherPersonnel.Factory.newInstance();
		int OtherpersonalCount = 0;
		List<OtherPersonnelDataType> otherPersonnelList = new ArrayList<>();

		if (periodInfo != null) {
			for (S2SOtherPersonnelDto otherPersonnelInfo : periodInfo
					.getOtherPersonnel()) {
				if (OTHERPERSONNEL_POSTDOC.equals(otherPersonnelInfo
						.getPersonnelType())) {
					otherPersonnel
							.setPostDocAssociates(getPostDocAssociates(otherPersonnelInfo));
				} else if (OTHERPERSONNEL_GRADUATE.equals(otherPersonnelInfo
						.getPersonnelType())) {
					otherPersonnel
							.setGraduateStudents(getGraduateStudents(otherPersonnelInfo));
				} else if (OTHERPERSONNEL_UNDERGRADUATE
						.equals(otherPersonnelInfo.getPersonnelType())) {
					otherPersonnel
							.setUndergraduateStudents(getUndergraduateStudents(otherPersonnelInfo));
				} else if (OTHERPERSONNEL_SECRETARIAL.equals(otherPersonnelInfo
						.getPersonnelType())) {
					otherPersonnel
							.setSecretarialClerical(getSecretarialClerical(otherPersonnelInfo));
				} else if (OtherpersonalCount < OTHERPERSONNEL_MAX_ALLOWED) {// Max
					// allowed
					// is 6
					OtherPersonnelDataType otherPersonnelDataType = OtherPersonnelDataType.Factory
							.newInstance();
					otherPersonnelDataType
							.setNumberOfPersonnel(otherPersonnelInfo
									.getNumberPersonnel());
					otherPersonnelDataType.setProjectRole(otherPersonnelInfo
							.getRole());
					otherPersonnelDataType
							.setCompensation(getSectBCompensationDataType(otherPersonnelInfo
									.getCompensation()));
					otherPersonnelList.add(otherPersonnelDataType);
					OtherpersonalCount++;
				}
			}

			otherPersonnel.setOtherArray(otherPersonnelList
					.toArray(new OtherPersonnelDataType[0]));

			if (periodInfo.getOtherPersonnelTotalNumber() != null) {
				otherPersonnel.setOtherPersonnelTotalNumber(periodInfo
						.getOtherPersonnelTotalNumber().intValue());
			}
			if (periodInfo.getTotalOtherPersonnelFunds() != null) {
				otherPersonnel.setTotalOtherPersonnelFund(periodInfo
						.getTotalOtherPersonnelFunds().bigDecimalValue());
			}
		}
		return otherPersonnel;
	}

	/**
	 * This method gets the PostDocAssociates details,ProjectRole,
	 * NumberOfPersonnel,Compensation based on OtherPersonnelInfo for the
	 * RRBudget,if it is a PostDocAssociates type.
	 * 
	 * @param otherPersonnel
	 *            (OtherPersonnelInfo)other personnel info entry.
	 * @return PostDocAssociates details corresponding to the OtherPersonnelInfo
	 *         object.
	 */
	private PostDocAssociates getPostDocAssociates(
			S2SOtherPersonnelDto otherPersonnel) {

		PostDocAssociates postDocAssociates = PostDocAssociates.Factory
				.newInstance();
		if (otherPersonnel != null) {
			postDocAssociates.setNumberOfPersonnel(otherPersonnel
					.getNumberPersonnel());
			postDocAssociates.setProjectRole(otherPersonnel.getRole());
			postDocAssociates
					.setCompensation(getSectBCompensationDataType(otherPersonnel
							.getCompensation()));
		}
		return postDocAssociates;
	}

	/**
	 * This method gets the GraduateStudents details,ProjectRole,
	 * NumberOfPersonnel,Compensation based on OtherPersonnelInfo for the
	 * RRBudget, if it is a GraduateStudents type.
	 * 
	 * @param otherPersonnel
	 *            (OtherPersonnelInfo) other personnel info entry.
	 * @return GraduateStudents details corresponding to the OtherPersonnelInfo
	 *         object.
	 */
	private GraduateStudents getGraduateStudents(
			S2SOtherPersonnelDto otherPersonnel) {

		GraduateStudents graduate = GraduateStudents.Factory.newInstance();
		if (otherPersonnel != null) {
			graduate.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
			graduate.setProjectRole(otherPersonnel.getRole());
			graduate
					.setCompensation(getSectBCompensationDataType(otherPersonnel
							.getCompensation()));
		}
		return graduate;
	}

	/**
	 * This method is to get the UndergraduateStudents details,ProjectRole,
	 * NumberOfPersonnel,Compensation based on OtherPersonnelInfo for the
	 * RRBudget,if it is a UndergraduateStudents type.
	 * 
	 * @param otherPersonnel
	 *            (OtherPersonnelInfo) other personnel info entry.
	 * @return UndergraduateStudents details corresponding to the
	 *         OtherPersonnelInfo object.
	 */
	private UndergraduateStudents getUndergraduateStudents(
			S2SOtherPersonnelDto otherPersonnel) {

		UndergraduateStudents undergraduate = UndergraduateStudents.Factory
				.newInstance();
		if (otherPersonnel != null) {
			undergraduate.setNumberOfPersonnel(otherPersonnel
					.getNumberPersonnel());
			undergraduate.setProjectRole(otherPersonnel.getRole());
			undergraduate
					.setCompensation(getSectBCompensationDataType(otherPersonnel
							.getCompensation()));
		}
		return undergraduate;
	}

	/**
	 * This method is to get the SecretarialClerical details,ProjectRole,
	 * NumberOfPersonnel,Compensation based on OtherPersonnelInfo for the
	 * RRBudget,if it is a SecretarialClerical type.
	 * 
	 * @param otherPersonnel
	 *            (OtherPersonnelInfo) other personnel info entry.
	 * @return SecretarialClerical corresponding to the OtherPersonnelInfo
	 *         object.
	 */
	private SecretarialClerical getSecretarialClerical(
			S2SOtherPersonnelDto otherPersonnel) {

		SecretarialClerical secretarialClerical = SecretarialClerical.Factory
				.newInstance();
		if (otherPersonnel != null) {
			secretarialClerical.setNumberOfPersonnel(otherPersonnel
					.getNumberPersonnel());
			secretarialClerical.setProjectRole(otherPersonnel.getRole());
			secretarialClerical
					.setCompensation(getSectBCompensationDataType(otherPersonnel
							.getCompensation()));
		}
		return secretarialClerical;
	}

	/**
	 * This method gets SectBCompensationDataType details
	 * AcademicMonths,CalendarMonths,FringeBenefits
	 * FundsRequested,SummerMonths,and RequestedSalary based on KeyPersonInfo
	 * for the OtherPersonnel.
	 * 
	 * @param compensation
	 *            (CompensationInfo) compensation info entry.
	 * @return SectBCompensationDataType details corresponding to the
	 *         CompensationInfo object.
	 */
	private SectBCompensationDataType getSectBCompensationDataType(
			S2SCompensationDto compensation) {

		SectBCompensationDataType sectBCompensation = SectBCompensationDataType.Factory
				.newInstance();
		if (compensation != null) {
			sectBCompensation.setAcademicMonths(compensation
					.getAcademicMonths().bigDecimalValue());
			sectBCompensation.setCalendarMonths(compensation
					.getCalendarMonths().bigDecimalValue());
			sectBCompensation.setSummerMonths(compensation.getSummerMonths()
					.bigDecimalValue());
			sectBCompensation.setFringeBenefits(compensation.getFringe()
					.bigDecimalValue());
			sectBCompensation.setFundsRequested(compensation
					.getFundsRequested().bigDecimalValue());
			sectBCompensation.setRequestedSalary(compensation
					.getRequestedSalary().bigDecimalValue());
		}
		return sectBCompensation;
	}

	/**
	 * This method gets KeyPersons details such as
	 * Name,ProjectRole,Compensation,TotalFundForAttachedKeyPersons
	 * TotalFundForKeyPersons and AttachedKeyPersons based on BudgetPeriodInfo
	 * for the RRBudget.
	 * 
	 * @param periodInfo
	 *            (BudgetPeriodInfo) budget period entry.
	 * @return KeyPersons details corresponding to the BudgetPeriodInfo object.
	 */
	private KeyPersons getKeyPersons(S2SBudgetPeriodDto periodInfo) {

		KeyPersons keyPersons = KeyPersons.Factory.newInstance();
		ScaleTwoDecimal extraFunds = ScaleTwoDecimal.ZERO;
		if (periodInfo != null) {
			if (periodInfo.getKeyPersons() != null) {
			    List<KeyPersonDataType> keyPersonList = new ArrayList<>();
				int keyPersonCount = 0;
				for (S2SKeyPersonDto keyPerson : periodInfo.getKeyPersons()) {
				  if(keyPerson.getRole().equals(NID_PD_PI) || hasPersonnelBudget(keyPerson,periodInfo.getBudgetPeriod())){
					KeyPersonDataType keyPersonDataType = KeyPersonDataType.Factory
							.newInstance();
					keyPersonDataType.setName(globLibV20Generator
							.getHumanNameDataType(keyPerson));
					if (isSponsorNIH(pdDoc)
							&& KEYPERSON_CO_PD_PI.equals(keyPerson.getRole())) {
					    DevelopmentProposalContract developmentProposal=pdDoc.getDevelopmentProposal();
					    
					    for (ProposalPersonContract proposalPerson : developmentProposal.getInvestigators()) {
					        if(isProposalPersonEqualsKeyPerson(proposalPerson,keyPerson)){	
					            if(proposalPerson.isMultiplePi())
					                keyPersonDataType.setProjectRole(NID_PD_PI); 					           
					            else 
					                keyPersonDataType.setProjectRole(NID_CO_PD_PI);                                 				            
					        }
					    }					
					} 					
					else if(keyPerson.getKeyPersonRole()!=null){
					    keyPersonDataType.setProjectRole(keyPerson.getKeyPersonRole());
                    }
					else {
						keyPersonDataType.setProjectRole(keyPerson.getRole());						
					}
					keyPersonDataType
							.setCompensation(getCompensation(keyPerson, periodInfo.getBudgetPeriod()));
					keyPersonList.add(keyPersonDataType);
					keyPersonCount++;
					LOG.info("keyPersonCount:" + keyPersonCount);
				}
			}
				keyPersons.setKeyPersonArray(keyPersonList.toArray(new KeyPersonDataType[0]));
			}
			if (periodInfo.getTotalFundsKeyPersons() != null) {
				keyPersons.setTotalFundForKeyPersons(periodInfo
						.getTotalFundsKeyPersons().bigDecimalValue());
			}
			for (S2SKeyPersonDto keyPerson : periodInfo.getExtraKeyPersons()) {
				extraFunds = extraFunds.add(keyPerson.getCompensation().getFundsRequested());
			}
		}
		TotalFundForAttachedKeyPersons totalFundForAttachedKeyPersons = TotalFundForAttachedKeyPersons.Factory
				.newInstance();
		totalFundForAttachedKeyPersons.setTotalFundForAttachedKeyPersonsExist(YesNoDataType.Y_YES);
		totalFundForAttachedKeyPersons.setBigDecimalValue(extraFunds.bigDecimalValue());
		keyPersons.setTotalFundForAttachedKeyPersons(totalFundForAttachedKeyPersons);
		NarrativeContract extraKeyPersonNarr = saveExtraKeyPersons(periodInfo);
		if(extraKeyPersonNarr!=null){
			AttachedKeyPersons attachedKeyPersons = AttachedKeyPersons.Factory
					.newInstance();
			FileLocation fileLocation = FileLocation.Factory.newInstance();
			attachedKeyPersons.setFileLocation(fileLocation);
			String contentId = createContentId(extraKeyPersonNarr);
			fileLocation.setHref(contentId);
			attachedKeyPersons.setFileLocation(fileLocation);
			attachedKeyPersons.setFileName(extraKeyPersonNarr.getNarrativeAttachment().getName());
			attachedKeyPersons.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);

			byte[] narrativeContent = null;
			if (extraKeyPersonNarr.getNarrativeAttachment() != null) {
				narrativeContent = extraKeyPersonNarr
				.getNarrativeAttachment().getData();
				
			}
			if(narrativeContent != null && narrativeContent.length > 0){
				final HashValueDocument.HashValue hashValue = getHashValue(narrativeContent);
				attachedKeyPersons.setHashValue(hashValue);
				addAttachment(new AttachmentData(extraKeyPersonNarr.getNarrativeAttachment().getFileDataId(), extraKeyPersonNarr.getNarrativeAttachment().getName(), contentId, narrativeContent, InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), extraKeyPersonNarr.getNarrativeAttachment().getUploadUser(), extraKeyPersonNarr.getNarrativeAttachment().getUploadTimestamp()));
				attachedKeyPersons.setTotalFundForAttachedKeyPersonsExist(YesNoDataType.Y_YES);
				keyPersons.setAttachedKeyPersons(attachedKeyPersons);
			}
		}
		return keyPersons;
	}



    /**
	 * This method gets KeyPersonCompensationDataType informations such as
	 * AcademicMonths,CalendarMonths,FringeBenefits
	 * SummerMonths,RequestedSalary,FundsRequested and BaseSalary based on
	 * KeyPersonInfo for the keyPerson.
	 * 
	 * @param keyPerson
	 *            (KeyPersonInfo) key person entry.
	 * @return KeyPersonCompensationDataType details corresponding to the
	 *         KeyPersonInfo object.
	 */
	private KeyPersonCompensationDataType getCompensation(
			S2SKeyPersonDto keyPerson, int budgetPeriod) {

		KeyPersonCompensationDataType compensation = KeyPersonCompensationDataType.Factory
				.newInstance();
        ScaleTwoDecimal baseSalaryByPeriod;
		if (keyPerson != null) {
			compensation.setAcademicMonths(keyPerson.getCompensation().getAcademicMonths()
					.bigDecimalValue());
			compensation.setCalendarMonths(keyPerson.getCompensation().getCalendarMonths()
					.bigDecimalValue());
			compensation.setSummerMonths(keyPerson.getCompensation().getSummerMonths()
					.bigDecimalValue());
			compensation.setFringeBenefits(keyPerson.getCompensation().getFringe()
					.bigDecimalValue());
			compensation.setRequestedSalary(keyPerson.getCompensation().getRequestedSalary()
					.bigDecimalValue());
			compensation.setFundsRequested(keyPerson.getCompensation().getFundsRequested()
					.bigDecimalValue());
			if (pdDoc.getDevelopmentProposal().getBudgets() != null) {
                baseSalaryByPeriod = keyPerson.getCompensation().getBaseSalary();
                if (baseSalaryByPeriod != null && baseSalaryByPeriod.isGreaterThan(ScaleTwoDecimal.ZERO)) {
                    compensation.setBaseSalary(baseSalaryByPeriod.bigDecimalValue());
                }
                else {
                    if (keyPerson.getCompensation().getBaseSalary() != null) {
                        compensation.setBaseSalary(keyPerson.getCompensation().getBaseSalary().bigDecimalValue());
                    }
                }
            }
            else {
                compensation.setBaseSalary(keyPerson.getCompensation().getBaseSalary().bigDecimalValue());
            }			
		}
		return compensation;
	}



	/**
	 * This method creates {@link XmlObject} of type {@link RRBudgetDocument} by
	 * populating data from the given {@link ProposalDevelopmentDocumentContract}
	 * 
	 * @param proposalDevelopmentDocument
	 *            for which the {@link XmlObject} needs to be created
	 * @return {@link XmlObject} which is generated using the given
	 *         {@link ProposalDevelopmentDocumentContract}
	 */
	@Override
	public RRBudgetDocument getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		return getRRBudget();
	}

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

	@Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

	@Override
	public DocumentFactory<RRBudgetDocument> factory() {
		return RRBudgetDocument.Factory;
	}
}
