/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.translog;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.lucene.util.Counter;
import org.elasticsearch.Assertions;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.index.translog.TranslogReader;
import org.elasticsearch.index.translog.TranslogWriter;

public class TranslogDeletionPolicy {
    private final Map<Object, RuntimeException> openTranslogRef;
    private final Map<Long, Counter> translogRefCounts = new HashMap<Long, Counter>();
    private long minTranslogGenerationForRecovery = 1L;
    private long translogGenerationOfLastCommit = 1L;
    private long retentionSizeInBytes;
    private long retentionAgeInMillis;

    public void assertNoOpenTranslogRefs() {
        if (!this.openTranslogRef.isEmpty()) {
            AssertionError e = new AssertionError((Object)"not all translog generations have been released");
            this.openTranslogRef.values().forEach(arg_0 -> e.addSuppressed(arg_0));
            throw e;
        }
    }

    public TranslogDeletionPolicy(long retentionSizeInBytes, long retentionAgeInMillis) {
        this.retentionSizeInBytes = retentionSizeInBytes;
        this.retentionAgeInMillis = retentionAgeInMillis;
        this.openTranslogRef = Assertions.ENABLED ? new ConcurrentHashMap<Object, RuntimeException>() : null;
    }

    public synchronized void setMinTranslogGenerationForRecovery(long newGen) {
        if (newGen < this.minTranslogGenerationForRecovery || newGen > this.translogGenerationOfLastCommit) {
            throw new IllegalArgumentException("Invalid minTranslogGenerationForRecovery can't go backwards; new [" + newGen + "],current [" + this.minTranslogGenerationForRecovery + "], lastGen [" + this.translogGenerationOfLastCommit + "]");
        }
        this.minTranslogGenerationForRecovery = newGen;
    }

    public synchronized void setTranslogGenerationOfLastCommit(long lastGen) {
        if (lastGen < this.translogGenerationOfLastCommit || lastGen < this.minTranslogGenerationForRecovery) {
            throw new IllegalArgumentException("Invalid translogGenerationOfLastCommit; new [" + lastGen + "],current [" + this.translogGenerationOfLastCommit + "], minRequiredGen [" + this.minTranslogGenerationForRecovery + "]");
        }
        this.translogGenerationOfLastCommit = lastGen;
    }

    public synchronized void setRetentionSizeInBytes(long bytes) {
        this.retentionSizeInBytes = bytes;
    }

    public synchronized void setRetentionAgeInMillis(long ageInMillis) {
        this.retentionAgeInMillis = ageInMillis;
    }

    synchronized Releasable acquireTranslogGen(long translogGen) {
        this.translogRefCounts.computeIfAbsent(translogGen, l -> Counter.newCounter((boolean)false)).addAndGet(1L);
        AtomicBoolean closed = new AtomicBoolean();
        assert (this.assertAddTranslogRef(closed));
        return () -> {
            if (closed.compareAndSet(false, true)) {
                this.releaseTranslogGen(translogGen);
                assert (this.assertRemoveTranslogRef(closed));
            }
        };
    }

    private boolean assertAddTranslogRef(Object reference) {
        RuntimeException existing = this.openTranslogRef.put(reference, new RuntimeException());
        if (existing != null) {
            throw new AssertionError("double adding of closing reference", existing);
        }
        return true;
    }

    private boolean assertRemoveTranslogRef(Object reference) {
        return this.openTranslogRef.remove(reference) != null;
    }

    synchronized int pendingTranslogRefCount() {
        return this.translogRefCounts.size();
    }

    private synchronized void releaseTranslogGen(long translogGen) {
        Counter current = this.translogRefCounts.get(translogGen);
        if (current == null || current.get() <= 0L) {
            throw new IllegalArgumentException("translog gen [" + translogGen + "] wasn't acquired");
        }
        if (current.addAndGet(-1L) == 0L) {
            this.translogRefCounts.remove(translogGen);
        }
    }

    synchronized long minTranslogGenRequired(List<TranslogReader> readers, TranslogWriter writer) throws IOException {
        long minByLocks = this.getMinTranslogGenRequiredByLocks();
        long minByAge = TranslogDeletionPolicy.getMinTranslogGenByAge(readers, writer, this.retentionAgeInMillis, this.currentTime());
        long minBySize = TranslogDeletionPolicy.getMinTranslogGenBySize(readers, writer, this.retentionSizeInBytes);
        long minByAgeAndSize = minBySize == Long.MIN_VALUE && minByAge == Long.MIN_VALUE ? Long.MAX_VALUE : Math.max(minByAge, minBySize);
        return Math.min(minByAgeAndSize, Math.min(minByLocks, this.minTranslogGenerationForRecovery));
    }

    static long getMinTranslogGenBySize(List<TranslogReader> readers, TranslogWriter writer, long retentionSizeInBytes) {
        if (retentionSizeInBytes >= 0L) {
            TranslogReader reader;
            long totalSize = writer.sizeInBytes();
            long minGen = writer.getGeneration();
            for (int i = readers.size() - 1; i >= 0 && totalSize < retentionSizeInBytes; totalSize += reader.sizeInBytes(), --i) {
                reader = readers.get(i);
                minGen = reader.getGeneration();
            }
            return minGen;
        }
        return Long.MIN_VALUE;
    }

    static long getMinTranslogGenByAge(List<TranslogReader> readers, TranslogWriter writer, long maxRetentionAgeInMillis, long now) throws IOException {
        if (maxRetentionAgeInMillis >= 0L) {
            for (TranslogReader reader : readers) {
                if (now - reader.getLastModifiedTime() > maxRetentionAgeInMillis) continue;
                return reader.getGeneration();
            }
            return writer.getGeneration();
        }
        return Long.MIN_VALUE;
    }

    protected long currentTime() {
        return System.currentTimeMillis();
    }

    private long getMinTranslogGenRequiredByLocks() {
        return this.translogRefCounts.keySet().stream().reduce(Math::min).orElse(Long.MAX_VALUE);
    }

    public synchronized long getMinTranslogGenerationForRecovery() {
        return this.minTranslogGenerationForRecovery;
    }

    public synchronized long getTranslogGenerationOfLastCommit() {
        return this.translogGenerationOfLastCommit;
    }

    synchronized long getTranslogRefCount(long gen) {
        Counter counter = this.translogRefCounts.get(gen);
        return counter == null ? 0L : counter.get();
    }
}

