/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.document.validation.impl.MaintenancePreRulesBase;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleBillingService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cg.CGConstants;
import org.kuali.kfs.module.cg.CGKeyConstants;
import org.kuali.kfs.module.cg.businessobject.Award;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.sql.Date;
import java.text.MessageFormat;

/**
 * PreRules checks for the Account that needs to occur while still in the Struts processing. This includes defaults,
 * confirmations, etc.
 */
public class AwardPreRules extends MaintenancePreRulesBase {

    protected AccountsReceivableModuleBillingService accountsReceivableModuleBillingService;
    protected DataDictionaryService dataDictionaryService;

    protected Award newAward;

    public AwardPreRules() {
        super();
        accountsReceivableModuleBillingService = SpringContext.getBean(AccountsReceivableModuleBillingService.class);
        dataDictionaryService = SpringContext.getBean(DataDictionaryService.class);
    }

    @Override
    protected boolean doCustomPreRules(MaintenanceDocument document) {
        setupConvenienceObjects(document);

        boolean proceed = continueIfEntryDateBeforeBeginDate();

        if (proceed) {
            proceed = continueIfSubcontractorTotalGreaterThanAwardTotal();
        }

        if (proceed && accountsReceivableModuleBillingService.isContractsGrantsBillingEnhancementActive()) {
            proceed = continueIfAwardTotalAmountLessThanScheduledAmount();
        }

        if (!proceed) {
            abortRulesCheck();
        }

        return true;
    }

    /**
     * Checks if the entry date is before the begin date. if so asks the user if they want to continue validation.
     * If no is selected further validation is aborted and the user is returned to the award document.
     *
     * @return true if the user selects yes, false otherwise
     */
    protected boolean continueIfEntryDateBeforeBeginDate() {
        boolean proceed = true;
        Date entryDate = newAward.getAwardEntryDate();
        Date beginDate = newAward.getAwardBeginningDate();

        if (ObjectUtils.isNotNull(entryDate) && ObjectUtils.isNotNull(beginDate) && entryDate.before(beginDate)) {
            String entryDateLabel = dataDictionaryService.getAttributeErrorLabel(Award.class,
                    KFSPropertyConstants.AWARD_ENTRY_DATE);
            String beginDateLabel = dataDictionaryService.getAttributeErrorLabel(Award.class,
                    KFSPropertyConstants.AWARD_BEGINNING_DATE);
            proceed = askOrAnalyzeYesNoQuestion(CGConstants.ENTRY_DATE_BEFORE_START_DATE_QUESTION,
                    MessageFormat.format(configService
                                    .getPropertyValueAsString(KFSKeyConstants.WARNING_AWARD_ENTRY_BEFORE_START_DATE),
                            entryDateLabel, beginDateLabel));
        }
        return proceed;
    }

    /**
     * Checks if the Subcontractor total amount is greater than the award total. If so asks the user if they want to
     * continue validation. if no is selected further validation is aborted and the user is returned to the award
     * document.
     *
     * @return true if the user selects yes, false otherwise
     */
    protected boolean continueIfSubcontractorTotalGreaterThanAwardTotal() {
        boolean proceed = true;

        KualiDecimal awardTotal = newAward.getAwardTotalAmount();
        KualiDecimal subcontractorTotal = newAward.getAwardSubcontractorsTotalAmount();
        if ((ObjectUtils.isNotNull(awardTotal) && subcontractorTotal.isGreaterThan(awardTotal))
                || (ObjectUtils.isNull(awardTotal) && subcontractorTotal.isPositive())) {

            String subcontractorLabel = dataDictionaryService.getCollectionLabel(Award.class,
                    KFSPropertyConstants.AWARD_SUBCONTRACTORS);
            String awardLabel = dataDictionaryService.getAttributeErrorLabel(Award.class,
                    KFSPropertyConstants.AWARD_TOTAL_AMOUNT);

            proceed = askOrAnalyzeYesNoQuestion(CGConstants.SUBCONTRACTOR_TOTAL_GREATER_THAN_AWARD_TOTAL_QUESTION,
                    MessageFormat.format(configService.getPropertyValueAsString(
                            KFSKeyConstants.WARNING_AWARD_SUBCONTRACTOR_TOTAL_GREATER_THAN_AWARD_TOTAL),
                            subcontractorLabel, awardLabel));
        }

        return proceed;
    }

    protected boolean continueIfAwardTotalAmountLessThanScheduledAmount() {
        KualiDecimal newAwardTotalAmount = newAward.getAwardTotalAmount();

        if (StringUtils.equals(newAward.getBillingFrequencyCode(), CGConstants.MILESTONE_BILLING_SCHEDULE_CODE)) {
            KualiDecimal milestoneTotal = SpringContext.getBean(AccountsReceivableModuleBillingService.class)
                    .getMilestonesTotalAmount(newAward.getProposalNumber());
            if (newAwardTotalAmount.isLessThan(milestoneTotal)) {
                return askOrAnalyzeYesNoQuestion(CGConstants.AWARD_AMOUNT_LESS_THAN_SCHEDULED_AMOUNT_QUESTION,
                        MessageFormat.format(configService.getPropertyValueAsString(CGKeyConstants.AwardConstants
                                .WARNING_CG_AWARD_TOTAL_AWARD_AMOUNT_LESS_THAN_SCHEDULED_AMOUNT),
                                CGKeyConstants.AwardConstants.MILESTONE_SCHEDULE_STRING));
            }
        }
        if (StringUtils.equals(newAward.getBillingFrequencyCode(), CGConstants.PREDETERMINED_BILLING_SCHEDULE_CODE)) {
            KualiDecimal billTotal = SpringContext.getBean(AccountsReceivableModuleBillingService.class)
                    .getBillsTotalAmount(newAward.getProposalNumber());
            if (newAwardTotalAmount.isLessThan(billTotal)) {
                return askOrAnalyzeYesNoQuestion(CGConstants.AWARD_AMOUNT_LESS_THAN_SCHEDULED_AMOUNT_QUESTION,
                        MessageFormat.format(configService.getPropertyValueAsString(CGKeyConstants.AwardConstants
                                .WARNING_CG_AWARD_TOTAL_AWARD_AMOUNT_LESS_THAN_SCHEDULED_AMOUNT),
                                CGKeyConstants.AwardConstants.PREDETERMINED_BILLING_SCHEDULE_STRING));
            }
        }

        return true;
    }

    protected void setupConvenienceObjects(MaintenanceDocument document) {
        // setup newAccount convenience objects, make sure all possible sub-objects are populated
        newAward = (Award) document.getNewMaintainableObject().getBusinessObject();
    }
}
