/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.maintenance;

import org.kuali.kfs.krad.bo.DocumentHeader;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.uif.service.ViewHelperService;
import org.kuali.rice.krad.bo.BusinessObject;

import java.util.List;
import java.util.Map;

/**
 * Provides contract for implementing a maintenance object within the maintenance framework.
 * <p>
 * <p> Currently the {@link Maintainable} serves many purposes. First since all maintenance documents share the
 * same document class {@link MaintenanceDocumentBase} certain document callbacks such as workflow post processing
 * are invoked on the maintainable. Second the maintainable provides a hook for custom actions on the maintenance view.
 * Finally since the maintainable extends {@link ViewHelperService} it is used to customize {@link View}
 * configuration for {@link MaintenanceView} instances.
 */
public interface Maintainable extends ViewHelperService, java.io.Serializable {

    /**
     * @param documentNumber document number for for referencing back to the containing {@code MaintenanceDocument}
     */
    void setDocumentNumber(String documentNumber);

    /**
     * @param document maintenance document instance to build title for
     * @return the title for the document instance in workflow (doc search results)
     */
    String getDocumentTitle(MaintenanceDocument document);

    /**
     * @return Object containing data object instance being maintained
     */
    Object getDataObject();

    /**
     * @param object data object instance that should be maintained
     */
    void setDataObject(Object object);

    /**
     * @return Class data object class being maintained
     */
    Class getDataObjectClass();

    /**
     * @param dataObjectClass class for maintenance data object
     */
    void setDataObjectClass(Class dataObjectClass);

    /**
     * If this method is overridden, most likely  getPersistableBusinessObject() should be overridden as well.
     *
     * @return true if maintenance is lockable, false otherwise
     */
    boolean isLockable();

    /**
     * @return the persistable business object or null if none exists.
     */
    PersistableBusinessObject getPersistableBusinessObject();

    /**
     * @return String containing the type of maintenance action this maintainable has been configured with
     */
    String getMaintenanceAction();

    /**
     * @param maintenanceAction string identifying the action type to be performed (new, edit, or copy)
     */
    void setMaintenanceAction(String maintenanceAction);

    /**
     * Invoked to generating the list of maintenance locks used to block other edits of the same data object record
     *
     * @return the locking representation(s) of this document, which are reproducible given the same keys and the same
     *         maintainable object
     */
    List<MaintenanceLock> generateMaintenanceLocks();

    /**
     * Invoked to persist changes to the data object being maintained
     * <p>
     * <p>
     * Called after the maintenance document has become final indicating the changes should be applied
     */
    void saveDataObject();

    /**
     * Invokes to delete the data object being maintained
     * <p>
     * <p>
     * Called after the maintenance document has become final indicating the changes should be applied
     */
    void deleteDataObject();

    /**
     * Invoked do perform custom processing when the route status for the containing maintenance document changes
     * <p>
     * <p>
     * Usually used for determining when the document has become final so further actions can take place in addition to
     * the usual persistence of the object changes
     *
     * @param documentHeader document header instance for containing maintenance document which can be used to check the
     *                       new status
     */
    void doRouteStatusChange(DocumentHeader documentHeader);

    /**
     * @return String locking document id for the maintainable which is used to create the maintenance lock string
     */
    String getLockingDocumentId();

    /**
     * @return List<String> of document ids to lock prior to processing this document  in the workflow engine
     */
    List<String> getWorkflowEngineDocumentIdsToLock();

    /**
     * Note this is only applicable if the data object is an instance of {@link BusinessObject}
     *
     * @return boolean true if notes are supported, false if they are not supported
     */
    boolean isNotesEnabled();

    /**
     * For the case when we want to maintain a business object that doesn't necessarily map to a single table in the
     * database or may doesn't map to a database at all
     *
     * @return boolean true if the data object is an instance of {@link ExternalizableBusinessObject}, false if not
     */
    boolean isExternalBusinessObject();

    /**
     * Invoked to prepare a new {@link BusinessObject} instance that is external
     *
     * @param businessObject new business object instance to prepare
     */
    void prepareExternalBusinessObject(BusinessObject businessObject);

    /**
     * @return boolean true if old data object exists, false if not
     */
    boolean isOldDataObjectInDocument();

    /**
     * Hook for performing any custom processing before the maintenance object is saved
     */
    void prepareForSave();

    /**
     * Hook for performing any custom processing after the maintenance object is retrieved from persistence storage
     */
    void processAfterRetrieve();

    /**
     * Called during setupMaintenanceObject to retrieve the original dataObject that is being edited or copied.
     * Override this method for non BusinessObject external persistence, Maintainable objects that extend BO should
     * override isExternalBusinessObject and prepareBusinessObject instead.
     * <p>
     * Do not override this method and isExternalBusinessObject.
     *
     * @param document       document instance for the maintenance object
     * @param dataObjectKeys Map of keys for the requested object
     * @return the object identified by the dataObjectKeys
     */
    Object retrieveObjectForEditOrCopy(MaintenanceDocument document, Map<String, String> dataObjectKeys);

    /**
     * Performs the setting of some attributes that might be necessary if we're creating a new business object using on
     * an existing business object. For example, create a division Vendor based on an existing parent Vendor.
     * (Please see VendorMaintainableImpl.java)
     *
     * @param document   maintenance document instance this maintainable belong to
     * @param parameters map of request parameters sent for the request
     */
    void setupNewFromExisting(MaintenanceDocument document, Map<String, String[]> parameters);

    /**
     * Hook for performing any custom processing after the maintenance object has been setup for a copy action
     *
     * @param document          maintenance document instance this maintainable belong to
     * @param requestParameters map of request parameters sent for the copy request
     */
    void processAfterCopy(MaintenanceDocument document, Map<String, String[]> requestParameters);

    /**
     * Hook for performing any custom processing after the maintenance object has been setup for a edit action
     *
     * @param document          maintenance document instance this maintainable belong to
     * @param requestParameters map of request parameters sent for the copy request
     */
    void processAfterEdit(MaintenanceDocument document, Map<String, String[]> requestParameters);

    /**
     * Hook for performing any custom processing after the maintenance object has been setup for a new action
     *
     * @param document          maintenance document instance this maintainable belong to
     * @param requestParameters map of request parameters sent for the copy request
     */
    void processAfterNew(MaintenanceDocument document, Map<String, String[]> requestParameters);

    /**
     * Hook for performing any custom processing after each posting of the maintenance document (for various actions
     * like add line, refresh)
     *
     * @param document          maintenance document instance this maintainable belong to
     * @param requestParameters map of request parameters from the post
     */
    void processAfterPost(MaintenanceDocument document, Map<String, String[]> requestParameters);
}
