/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.modifier;

import org.kuali.kfs.krad.uif.component.ConfigurableBase;
import org.kuali.kfs.krad.uif.component.Ordered;

import java.io.Serializable;

/**
 * Provides configuration for comparing an object with another object.
 * <p>
 * <p>
 * Used with a comparison view (such as in maintenance documents edit mode) where two objects with the same properties
 * are compared. This class configures the object paths for the objects that will be compared, and has additional
 * configuration for the generated comparison group.
 * <p>
 * <p>
 * All comparison objects must have the same fields and collection rows.
 *
 * @see CompareFieldCreateModifier
 */
public class ComparableInfo extends ConfigurableBase implements Serializable, Ordered {

    private static final long serialVersionUID = -5926058412202550266L;

    private String bindingObjectPath;
    private String headerText;
    private boolean readOnly;

    private int order;
    private String idSuffix;

    private boolean compareToForValueChange;
    private boolean highlightValueChange;

    public ComparableInfo() {
        super();

        readOnly = false;
        compareToForValueChange = false;
        highlightValueChange = true;
    }

    /**
     * When a comparison view is rendered, a group will be rendered for each comparison object using the fields defined
     * on the view. This gives the path to one of the comparison objects.
     * <p>
     * <p>
     * e.g. For maintenance documents the compare object paths would be document.newMaintainableObject.businessObject
     * and document.oldMaintainableObject.businessObject.
     *
     * @return the path (from the form) for the object to compare to.
     */
    public String getBindingObjectPath() {
        return this.bindingObjectPath;
    }

    /**
     * @param bindingObjectPath the path to the compare object to set.
     */
    public void setBindingObjectPath(String bindingObjectPath) {
        this.bindingObjectPath = bindingObjectPath;
    }

    /**
     * In the comparison view each compare group can be labeled, this gives the text that should be used for that label.
     * For example in the maintenance view the compare record is labeled 'Old' to indicate it is the old version of the
     * record.
     *
     * @return header text that should display on the header for the compare group.
     */
    public String getHeaderText() {
        return this.headerText;
    }

    /**
     * @param headerText the compare group header text to set.
     */
    public void setHeaderText(String headerText) {
        this.headerText = headerText;
    }

    /**
     * @return boolean true if the group should be read-only, false if edits are allowed.
     */
    public boolean isReadOnly() {
        return this.readOnly;
    }

    /**
     * @param readOnly the read-only indicator to set.
     */
    public void setReadOnly(boolean readOnly) {
        this.readOnly = readOnly;
    }

    /**
     * Sets the order value that will be used to determine where the compare group should be placed in relation to the
     * other compare groups.
     * <p>
     * <p>
     * For example if the compare groups are being rendered from left to right in columns, a lower order value would be
     * placed to the left of a compare group with a higher order value.
     *
     * @see org.springframework.core.Ordered#getOrder()
     */
    public int getOrder() {
        return this.order;
    }

    /**
     * @param order the compare object order to set.
     */
    public void setOrder(int order) {
        this.order = order;
    }

    /**
     * For the given string, all components created for the comparison group will contain the string on their id. This
     * can be helpful for scripting. If not given, the items will receive a default id suffix.
     *
     * @return id suffix to use for the generated comparison fields.
     */
    public String getIdSuffix() {
        return this.idSuffix;
    }

    /**
     * @param idSuffix the id prefix to set for use with the generated comparison components.
     */
    public void setIdSuffix(String idSuffix) {
        this.idSuffix = idSuffix;
    }

    /**
     * Indicates whether this comparable group's field values should be compared to when highlighting changes of values
     * between comparables (versions).
     *
     * @return boolean true if this comparable group should be used for comparison, false if not.
     * @see #isHighlightValueChange
     */
    public boolean isCompareToForValueChange() {
        return this.compareToForValueChange;
    }

    /**
     * @param compareToForValueChange the use comparable group values for comparison indicator to set.
     */
    public void setCompareToForValueChange(boolean compareToForValueChange) {
        this.compareToForValueChange = compareToForValueChange;
    }

    /**
     * Indicates whether the fields in this comparable group should be highlighted if their values defer from the
     * comparable group marked for comparison.
     *
     * @return boolean true if the comparable fields should be highlighted, false if they should not be highlighted (no
     *         comparison will be performed).
     * @see #isCompareToForValueChange
     */
    public boolean isHighlightValueChange() {
        return this.highlightValueChange;
    }

    /**
     * @param highlightValueChange the highlight comparable field value changed indicator to set.
     */
    public void setHighlightValueChange(boolean highlightValueChange) {
        this.highlightValueChange = highlightValueChange;
    }

}
