/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.view;

import org.kuali.kfs.krad.service.DataObjectMetaDataService;
import org.kuali.kfs.krad.uif.UifConstants.ViewType;
import org.kuali.kfs.krad.web.form.UifFormBase;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Interface that must be implemented for clases the provide the backing data (model) for a {@link View}
 * <p>
 * <p>
 * Since the View relies on helper properties from the model it is necessary the backing object implement this
 * interface. Note model objects can extend {@link UifFormBase} which implements this interface
 */
public interface ViewModel extends Serializable {

    /**
     * @return Unique Id for the <code>View</code> instance. This is specified for a view in its definition by setting
     *         the 'id' property.
     */
    String getViewId();

    /**
     * @param viewId the unique view id to set.
     */
    void setViewId(String viewId);

    /**
     * @return Name for the <code>View</code> instance. This is specified for a view in its definition by setting the
     *         'id' property. The name is not necessary unique and cannot be used by itself to retrieve a view.
     *         Typically it is used with other parameters to identify a view with a certain type (view type).
     */
    String getViewName();

    /**
     * @param viewName the view name to set.
     */
    void setViewName(String viewName);

    /**
     * @return Name for the type of view being requested. This can be used to find <code>View</code> instances by request
     *         parameters (not necessary the unique id)
     */
    ViewType getViewTypeName();

    /**
     * @param viewTypeName the view type name to set.
     */
    void setViewTypeName(ViewType viewTypeName);

    /**
     * @return View instance associated with the model. Used to render the user interface
     */
    View getView();

    /**
     * @param view the view instance to set.
     */
    void setView(View view);

    /**
     * @return View instance for the page that made a request. Since a new view instance= gets initialized for each
     * request before the controller logic is invoked, any state about the previous view is lost. This could be needed
     * to read metadata from the view for such things as collection processing. When this is necessary the previous view
     * instance can be retrieved
     */
    View getPostedView();

    /**
     * @param previousView the previous view instance to set.
     */
    void setPostedView(View previousView);

    /**
     * @return String page id for the current page being displayed within the view.
     */
    String getPageId();

    /**
     * @param pageId the current page id to set.
     */
    void setPageId(String pageId);

    /**
     * @return String form post URL the form generated for the view should post to
     */
    String getFormPostUrl();

    /**
     * @param formPostUrl the form post URL to set.
     */
    void setFormPostUrl(String formPostUrl);

    /**
     * @return Map<String, String> of view parameters that was used to configured the <code>View</code>. Maintained on
     *         the form to rebuild the view on posts and session timeout
     * @see View#getViewRequestParameters()
     */
    Map<String, String> getViewRequestParameters();

    /**
     * @param viewRequestParameters the view's request parameter map to set.
     */
    void setViewRequestParameters(Map<String, String> viewRequestParameters);

    /**
     * If the view being rendered supports request setting of read-only fields, the readOnlyFields request parameter
     * can be sent to mark fields as read only that might not have been otherwise
     * <p>
     * <p>
     * Note the paths specified should be the simple property names (not the full binding path). Therefore if the
     * property name appears multiple times in the view, all instances will be set as read only
     *
     * @return List<String> of fields that should be read only on the view
     * @see View#isSupportsReadOnlyFieldsOverride()
     */
    List<String> getReadOnlyFieldsList();

    /**
     * @param readOnlyFieldsList the list of read only fields to set.
     */
    void setReadOnlyFieldsList(List<String> readOnlyFieldsList);

    /**
     * @return Map<String, Object> of new collection lines. The key of the Map gives the collection name the line
     *         instance applies to, the Map value is an instance of the collection object class that holds the new line
     *         data.
     */
    Map<String, Object> getNewCollectionLines();

    /**
     * @param newCollectionLines the new collection lines Map to set.
     */
    void setNewCollectionLines(Map<String, Object> newCollectionLines);

    /**
     * Many times besides just setting the method to call actions need to send additional parameters. For instance the
     * method being called might do a redirect, in which case the action needs to send parameters for the redirect URL.
     * An example of this is redirecting to a <code>Lookup</code> view. In some cases the parameters that need to be
     * sent conflict with properties already on the form, and putting all the action parameters as form properties would
     * grow massive (in addition to adds an additional step from the XML config). So this general map solves those issues.
     *
     * @return Map<String, String> action parameters for the invoked action.
     */
    Map<String, String> getActionParameters();

    /**
     * @param actionParameters the action parameters map to set.
     */
    void setActionParameters(Map<String, String> actionParameters);

    /**
     * Used when a request is made that refreshes part of the view. The current state for components (which have state
     * that can be changed on the client), is populated into this map which is then used by the
     * {@link org.kuali.kfs.krad.uif.service.ViewHelperService} to update the components so that the state is maintained
     * when they render.
     *
     * @return Map<String, Object> map populated from the component state maintained on the client. The key is name of
     *         property or component id, and value is the property value or another map of component key/value pairs.
     */
    Map<String, Object> getClientStateForSyncing();

    /**
     * When the select field is enabled for a <code>CollectionGroup</code>, the framework will be default bind the
     * selected identifier strings to this property. The key of the map uniquely identifies the collection by the full
     * binding path to the collection, and the value is a set of Strings for the checked lines.
     *
     * @return Map<String, Set<String>> map of collections for lines that were selected.
     * @see DataObjectMetaDataService#getDataObjectIdentifierString(java.lang.Object)
     */
    Map<String, Set<String>> getSelectedCollectionLines();

    /**
     * @param selectedCollectionLines the map that holds selected collection lines to set.
     */
    void setSelectedCollectionLines(Map<String, Set<String>> selectedCollectionLines);

    /**
     * @return boolean true if default values from the configured <code>View</code> applied have been applied (This
     *         happens only once for each form instance), false if not.
     */
    boolean isDefaultsApplied();

    /**
     * @param defaultsApplied the defaults applied indicator to set.
     */
    void setDefaultsApplied(boolean defaultsApplied);

}
