/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.web.form;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kuali.kfs.krad.service.KRADServiceLocatorWeb;
import org.kuali.kfs.krad.uif.UifConstants;
import org.kuali.kfs.krad.uif.UifConstants.ViewType;
import org.kuali.kfs.krad.uif.UifParameters;
import org.kuali.kfs.krad.uif.service.ViewService;
import org.kuali.kfs.krad.uif.view.History;
import org.kuali.kfs.krad.uif.view.HistoryEntry;
import org.kuali.kfs.krad.uif.view.View;
import org.kuali.kfs.krad.uif.view.ViewModel;
import org.kuali.kfs.krad.util.KRADUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;

/**
 * Base form class for views within the KRAD User Interface Framework.
 * <p>
 * <p>
 * Holds properties necessary to determine the {@link View} instance that will be used to render the UI.
 */
public class UifFormBase implements ViewModel {
    private static final long serialVersionUID = 8432543267099454434L;

    private static final Log LOG = LogFactory.getLog(UifFormBase.class);

    // current view
    protected String viewId;
    protected String viewName;
    protected ViewType viewTypeName;
    protected String pageId;
    protected String methodToCall;
    protected String formKey;
    protected String jumpToId;
    protected String jumpToName;
    protected String focusId;
    protected String formPostUrl;

    protected boolean defaultsApplied;
    protected boolean skipViewInit;

    protected View view;
    protected View postedView;

    protected Map<String, String> viewRequestParameters;
    protected List<String> readOnlyFieldsList;

    protected Map<String, Object> newCollectionLines;
    protected Map<String, String> actionParameters;
    protected Map<String, Object> clientStateForSyncing;
    protected Map<String, Set<String>> selectedCollectionLines;

    protected MultipartFile attachmentFile;

    // navigation
    protected String returnLocation;
    protected String returnFormKey;

    protected History formHistory;

    protected boolean renderFullView;
    protected boolean validateDirty;

    protected String csrfToken;

    public UifFormBase() {
        formKey = generateFormKey();
        renderFullView = true;
        defaultsApplied = false;
        skipViewInit = false;

        formHistory = new History();

        readOnlyFieldsList = new ArrayList<>();
        viewRequestParameters = new HashMap<>();
        newCollectionLines = new HashMap<>();
        actionParameters = new HashMap<>();
        clientStateForSyncing = new HashMap<>();
        selectedCollectionLines = new HashMap<>();
    }

    /**
     * @return the unique id used to store this "conversation" in the session. The default method generates a java UUID.
     */
    protected String generateFormKey() {
        return UUID.randomUUID().toString();
    }


    /**
     * Called after Spring binds the request to the form and before the controller method is invoked.
     *
     * @param request request object containing the query parameters
     */
    public void postBind(HttpServletRequest request) {
        // default form post URL to request URL
        formPostUrl = request.getRequestURL().toString();

        // get any sent client view state and parse into map
        if (request.getParameterMap().containsKey(UifParameters.CLIENT_VIEW_STATE)) {
            String clientStateJSON = request.getParameter(UifParameters.CLIENT_VIEW_STATE);
            if (StringUtils.isNotBlank(clientStateJSON)) {
                // change single quotes to double quotes (necessary because the reverse was done for sending)
                clientStateJSON = StringUtils.replace(clientStateJSON, "'", "\"");

                ObjectMapper mapper = new ObjectMapper();
                try {
                    clientStateForSyncing = mapper.readValue(clientStateJSON, Map.class);
                } catch (IOException e) {
                    throw new RuntimeException("Unable to decode client side state JSON", e);
                }
            }
        }

        // populate read only fields list
        if (request.getParameter(UifParameters.READ_ONLY_FIELDS) != null) {
            String readOnlyFields = request.getParameter(UifParameters.READ_ONLY_FIELDS);
            setReadOnlyFieldsList(KRADUtils.convertStringParameterToList(readOnlyFields));
        }

        // reset skip view init parameter if not passed
        if (!request.getParameterMap().containsKey(UifParameters.SKIP_VIEW_INIT)) {
            skipViewInit = false;
        }
    }

    public String getViewId() {
        return this.viewId;
    }

    public void setViewId(String viewId) {
        this.viewId = viewId;
    }

    public String getViewName() {
        return this.viewName;
    }

    public void setViewName(String viewName) {
        this.viewName = viewName;
    }

    public ViewType getViewTypeName() {
        return this.viewTypeName;
    }

    public void setViewTypeName(ViewType viewTypeName) {
        this.viewTypeName = viewTypeName;
    }

    public String getPageId() {
        return this.pageId;
    }

    public void setPageId(String pageId) {
        this.pageId = pageId;
    }

    public String getFormPostUrl() {
        return this.formPostUrl;
    }

    public void setFormPostUrl(String formPostUrl) {
        this.formPostUrl = formPostUrl;
    }

    public String getReturnLocation() {
        return this.returnLocation;
    }

    public void setReturnLocation(String returnLocation) {
        this.returnLocation = returnLocation;
    }

    public String getReturnFormKey() {
        return this.returnFormKey;
    }

    public void setReturnFormKey(String returnFormKey) {
        this.returnFormKey = returnFormKey;
    }

    /**
     * @return String the controller method that should be invoked to fulfill a request. The value will be matched up
     *         against the 'params' setting on the {@link org.springframework.web.bind.annotation.RequestMapping}
     *         annotation for the controller method.
     */
    public String getMethodToCall() {
        return this.methodToCall;
    }

    /**
     * @param methodToCall the method to call value to set.
     */
    public void setMethodToCall(String methodToCall) {
        this.methodToCall = methodToCall;
    }

    public Map<String, String> getViewRequestParameters() {
        return this.viewRequestParameters;
    }

    public void setViewRequestParameters(Map<String, String> viewRequestParameters) {
        this.viewRequestParameters = viewRequestParameters;
    }

    public List<String> getReadOnlyFieldsList() {
        return readOnlyFieldsList;
    }

    public void setReadOnlyFieldsList(List<String> readOnlyFieldsList) {
        this.readOnlyFieldsList = readOnlyFieldsList;
    }

    public Map<String, Object> getNewCollectionLines() {
        return this.newCollectionLines;
    }

    public void setNewCollectionLines(Map<String, Object> newCollectionLines) {
        this.newCollectionLines = newCollectionLines;
    }

    public Map<String, String> getActionParameters() {
        return this.actionParameters;
    }

    /**
     * @return the action parameters map as a {@link Properties} instance.
     */
    public Properties getActionParametersAsProperties() {
        return KRADUtils.convertMapToProperties(actionParameters);
    }

    public void setActionParameters(Map<String, String> actionParameters) {
        this.actionParameters = actionParameters;
    }

    /**
     * @param actionParameterName name of the action parameter to retrieve value for
     * @return String the value for the given action parameter, or empty string if not found.
     */
    public String getActionParamaterValue(String actionParameterName) {
        if ((actionParameters != null) && actionParameters.containsKey(actionParameterName)) {
            return actionParameters.get(actionParameterName);
        }

        return "";
    }

    /**
     * The action event is a special action parameter that can be sent to indicate a type of action being taken. This
     * can be looked at by the view or components to render differently.
     * <p>
     * TODO: make sure action parameters are getting reinitialized on each request
     *
     * @return String action event name that was sent in the action parameters or blank if action event was not sent.
     */
    public String getActionEvent() {
        if ((actionParameters != null) && actionParameters.containsKey(UifConstants.UrlParams.ACTION_EVENT)) {
            return actionParameters.get(UifConstants.UrlParams.ACTION_EVENT);
        }

        return "";
    }

    public Map<String, Object> getClientStateForSyncing() {
        return clientStateForSyncing;
    }

    public Map<String, Set<String>> getSelectedCollectionLines() {
        return selectedCollectionLines;
    }

    public void setSelectedCollectionLines(Map<String, Set<String>> selectedCollectionLines) {
        this.selectedCollectionLines = selectedCollectionLines;
    }

    /**
     * When the view is posted, the previous form instance is retrieved and then populated from the request parameters.
     * This key string is retrieve the session form from the session service.
     *
     * @return String string that identifies the form instance in session storage.
     */
    public String getFormKey() {
        return this.formKey;
    }

    /**
     * @param formKey the form's session key value to set.
     */
    public void setFormKey(String formKey) {
        this.formKey = formKey;
    }

    public boolean isDefaultsApplied() {
        return this.defaultsApplied;
    }

    public void setDefaultsApplied(boolean defaultsApplied) {
        this.defaultsApplied = defaultsApplied;
    }

    /**
     * Indicates whether a new view is being initialized or the call is refresh (or query) call.
     *
     * @return boolean true if view initialization was skipped, false if new view is being created.
     */
    public boolean isSkipViewInit() {
        return skipViewInit;
    }

    /**
     * @param skipViewInit the skip view initialization flag value to set.
     */
    public void setSkipViewInit(boolean skipViewInit) {
        this.skipViewInit = skipViewInit;
    }

    /**
     * @return MultipartFile representing files that are attached through the view.
     */
    public MultipartFile getAttachmentFile() {
        return this.attachmentFile;
    }

    /**
     * @param attachmentFile the form's attachment file value to set.
     */
    public void setAttachmentFile(MultipartFile attachmentFile) {
        this.attachmentFile = attachmentFile;
    }

    /**
     * @return the renderFullView
     */
    public boolean isRenderFullView() {
        return this.renderFullView;
    }

    /**
     * @param renderFullView
     */
    public void setRenderFullView(boolean renderFullView) {
        this.renderFullView = renderFullView;
    }

    public View getView() {
        return this.view;
    }

    public void setView(View view) {
        this.view = view;
        initHomewardPathList();
    }

    /**
     * Set the "Home" url of the homewardPathList (ie. breadcrumbs history).
     */
    private void initHomewardPathList() {
        if (getReturnLocation() == null) {
            LOG.warn("Could not init homewardPathList.  returnLocation is null.");
            return;
        }

        List<HistoryEntry> homewardPathList = new ArrayList<HistoryEntry>();
        if ((view != null) && (view.getBreadcrumbs() != null) && (view.getBreadcrumbs().getHomewardPathList() != null)) {
            homewardPathList = view.getBreadcrumbs().getHomewardPathList();
        }

        HistoryEntry historyEntry = new HistoryEntry("", "", "Home", getReturnLocation(), "");
        if (homewardPathList.isEmpty()) {
            homewardPathList.add(historyEntry);
        } else if (StringUtils.equals(homewardPathList.get(0).getTitle(), "Home")) {
            homewardPathList.set(0, historyEntry);
        } else {
            homewardPathList.add(0, historyEntry);
        }
    }

    public View getPostedView() {
        return this.postedView;
    }

    public void setPostedView(View postedView) {
        this.postedView = postedView;
    }

    /**
     * @return ViewService implementation that can be used to retrieve {@link View} instances.
     */
    protected ViewService getViewService() {
        return KRADServiceLocatorWeb.getViewService();
    }

    /**
     * Using "TOP" or "BOTTOM" will jump to the top or the bottom of the resulting page.
     * jumpToId always takes precedence over jumpToName, if set.
     *
     * @return the jumpToId for this form, the element with this id will be jumped to automatically when the form is
     *         loaded in the view.
     */
    public String getJumpToId() {
        return this.jumpToId;
    }

    /**
     * @param jumpToId the jumpToId value to set
     */
    public void setJumpToId(String jumpToId) {
        this.jumpToId = jumpToId;
    }

    /**
     * WARNING: jumpToId always takes precedence over jumpToName, if set.
     *
     * @return the jumpToName for this form, the element with this name will be jumped to automatically when the form is
     *         loaded in the view.
     */
    public String getJumpToName() {
        return this.jumpToName;
    }

    /**
     * @param jumpToName the jumpToName value to set
     */
    public void setJumpToName(String jumpToName) {
        this.jumpToName = jumpToName;
    }

    /**
     * @return the field to place focus on when the page loads. An empty focusId will result in focusing on the first
     *         visible input element by default.
     */
    public String getFocusId() {
        return this.focusId;
    }

    /**
     * @param focusId the focusId value to set.
     */
    public void setFocusId(String focusId) {
        this.focusId = focusId;
    }

    /**
     * Used for breadcrumb widget generation on the view and also for navigating back to previous or hub locations.
     *
     * @return History instance representing the History of views that have come before the viewing of the current view.
     */
    public History getFormHistory() {
        return formHistory;
    }

    /**
     * @param history the current History object to set.
     */
    public void setFormHistory(History history) {
        this.formHistory = history;
    }

    /**
     * For FormView, it's necessary to validate when the user tries to navigate out of the form. If set, all the
     * InputFields will be validated on refresh, navigate, cancel or close Action or on form unload and if dirty,
     * displays a message and user can decide whether to continue with the action or stay on the form.
     *
     * @return boolean true if dirty validation should be enabled; false otherwise.
     */
    public boolean isValidateDirty() {
        return this.validateDirty;
    }

    /**
     * @param validateDirty dirty validation indicator value to set.
     */
    public void setValidateDirty(boolean validateDirty) {
        this.validateDirty = validateDirty;
    }

    public String getCsrfToken() {
        return csrfToken;
    }

    public void setCsrfToken(String csrfToken) {
        this.csrfToken = csrfToken;
    }
}
